#!/usr/bin/perl

use my_inc "..";

=head1 METADATA

<crontab>
    time: */59 * * * *
    sharded: 1
    package: scripts-switchman
    <switchman>
        group: scripts-other
        <leases>
            mem: 50
        </leases>
    </switchman>
</crontab>
<juggler>
    host:   checks_auto.direct.yandex.ru
    sharded: 1
    ttl:     2h13m20s
    tag: direct_group_internal_systems
</juggler>

=cut

=head1 NAME

    ppcForceCurrencyConvertNotify.pl

=head1 DESCRIPTION

    Отправляет клиентам уведомление о дате принудительной конвертации
    и проставляет эту дату в таблицу force_currency_convert.

=head1 RUNNING

    Можно запускать по конкретным клиентам:
        --login — логин клиента или одного из его представителей
        --ignore-day - ставить клиентов конвертироваться, даже если их конвертация приходится на выходные и праздники

    ./protected/ppcForceCurrencyConvertNotify.pl --shard-id 1
    ./protected/ppcForceCurrencyConvertNotify.pl --shard-id=2 --login holodilnikru

=cut

use Direct::Modern;

use Settings;
use ScriptHelper sharded => 1, 'Yandex::Log' => 'messages';

use Client;
use Primitives;
use PrimitivesIds;
use RBAC2::Extended;
use RBACElementary;
use Notification;
use Client::ConvertToRealMoney ();
use Holidays qw/is_holiday/;
use geo_regions;

use Yandex::TimeCommon;
use Yandex::DBTools;

=head1 SUBROUTINES/METHODS/VARIABLES

=cut

=head2 $FORCE_CONVERT_AFTER_DAYS

    Через сколько дней конвертировать клиента (с момента как он стал подходить под все условия)

=cut

my $FORCE_CONVERT_AFTER_DAYS = 14;

my $OPERATOR_UID = 1;
my $IGNORE_DAY = 0;

my $logins = [];
extract_script_params(
    'login=s@' => $logins,
    'ignore-day' => \$IGNORE_DAY,
);

my $LASTRUN_PROPERTY_NAME = "ppcForceCurrencyConvertNotify_last_run_date_shard$SHARD";

$log->out('START');

my (@additional_conds, $prop);
my $today = today();
if ($logins && @$logins) {
    $log->out('Working only on logins:', $logins);
    my @only_client_ids = map { get_clientid(uid => get_uid_by_login2($_)) } @$logins;
    @additional_conds = ('AND', {'fcc.ClientID' => \@only_client_ids});
} else {
    $prop = Property->new($LASTRUN_PROPERTY_NAME);
    my $last_date = $prop->get() // '';
    if ($last_date eq $today) {
        $log->out("Last run date according to $LASTRUN_PROPERTY_NAME property is $last_date, exitting");
        juggler_ok(description => 'Script has already done its work today');
        exit 0;
    }
}

$log->out('Connection to RBAC');
my $rbac = eval { RBAC2::Extended->get_singleton($OPERATOR_UID) } or $log->die("Can't connect to RBAC: $@");

$log->out("Fetching clients to probably force currency convert");
my $all_clients = get_all_sql(PPC(shard => $SHARD), ["
        SELECT fcc.ClientID
             , DATE(NOW() + INTERVAL $FORCE_CONVERT_AFTER_DAYS DAY) AS convert_date
             , MIN(cfcc.country_region_id) AS country_region_id
             , MIN(cfcc.currency) AS currency
        FROM force_currency_convert fcc
        INNER JOIN users u ON u.ClientID = fcc.ClientID
        LEFT JOIN clients cl ON fcc.ClientID = cl.ClientID
        INNER JOIN clients_to_force_multicurrency_teaser t ON fcc.ClientID = t.ClientID
        INNER JOIN client_teaser_data_lastupdate ctdl ON fcc.ClientID = ctdl.ClientID
        INNER JOIN client_firm_country_currency cfcc ON cfcc.ClientID = fcc.ClientID
        LEFT JOIN currency_convert_queue q ON fcc.ClientID = q.ClientID
        INNER JOIN campaigns c ON u.uid = c.uid
        WHERE
            q.ClientID IS NULL
            AND fcc.convert_date IS NULL
            AND IFNULL(cl.work_currency, 'YND_FIXED') = 'YND_FIXED'
            AND ctdl.last_update > NOW() - INTERVAL $Client::ConvertToRealMoney::COUNTRY_CURRENCY_LAST_UPDATE_INTERVAL_DAYS DAY
    ", @additional_conds, "
        GROUP BY fcc.ClientID
        HAVING
                COUNT(DISTINCT cfcc.country_region_id) = 1
            AND COUNT(DISTINCT cfcc.currency) = 1
        ORDER BY fcc.accepted_at
    "]) || [];

my $clients;
if (@$all_clients) {
    my $clients_cnt = scalar @$all_clients;
    my $daily_client_limit = Client::ConvertToRealMoney::get_force_convert_daily_client_limit();
    $log->out("Got $clients_cnt ClientID's to probably currency convert with MODIFY clients limit of $daily_client_limit");

    my $modify_clients_cnt = 0;
    for my $client (@$all_clients) {
        if ($client->{country_region_id} == $geo_regions::RUS) {
            if ($modify_clients_cnt < $daily_client_limit) {
                $client->{convert_type} = 'MODIFY';
                push @$clients, $client;
                $modify_clients_cnt++;
            }
        # белорусов конвертируем только в белорусские рубли
        } elsif ($client->{country_region_id} == $geo_regions::BY && $client->{currency} ne 'BYN') {
            next;
        } else {
            $client->{convert_type} = 'COPY';
            push @$clients, $client;
        }
    }
}

if ($clients && @$clients) {
    my $clients_cnt = scalar @$clients;
    $log->out("Got $clients_cnt ClientID's to probably currency convert after filtering");

    my $client_ids = [ map { $_->{ClientID} } @$clients ];

    $log->out('Fetching NDS, currencies and chief reps data for ClientIDs:', $client_ids);
    my $clients_nds = mass_get_client_NDS($client_ids, fetch_for_ynd_fixed_too => 1);
    my $clients_currencies = mass_get_client_currencies($client_ids);
    my $clients_chief_rep = rbac_get_chief_reps_of_clients($client_ids);
    my $chief_rep2login = get_uid2login(uid => [values %$clients_chief_rep]);
    my $clid2agid = Primitives::mass_get_client_first_agency($client_ids);

    for my $client (@$clients) {
        my $client_id = $client->{ClientID};
        my $country = $client->{country_region_id};
        my $currency = $client->{currency};

        my $msg_prefix_guard = $log->msg_prefix_guard("[ClientID $client_id]");

        my $start_convert_date = tomorrow($client->{convert_date});
        # Если скрипт запущен не вручную и завтра выходной/праздник в стране клиента, пропускаем его.
        if (!$IGNORE_DAY && is_holiday($start_convert_date, $country)) {
            $log->out("Converting is to be on $start_convert_date, but it's holiday. Skipping.");
            next;
        }

        my $chief_uid = $clients_chief_rep->{$client_id};

        $log->out('Checking whether client can be converted');
        my $error = Client::can_convert_to_real_money(
            ClientID => $client_id,
            NDS => $clients_nds->{$client_id},
            client_currencies => $clients_currencies->{$client_id},
            client_chief_uid => $chief_uid,
            agency_id => $clid2agid->{$client_id},
        );
        if ($error) {
            $log->out('Got error from can_convert_to_real_money', $error);
            next;
        }

        my $mailvars = {
            convert_date => $start_convert_date,
            currency => $currency,
            chief_login => $chief_rep2login->{$chief_uid},
            chief_uid => $chief_uid,
            ClientID => $client_id,
            convert_type => $client->{convert_type},
        };
        $log->out('Sending notification with mailvars:', $mailvars);
        my $success = eval { add_notification($rbac, 'force_currency_convert', $mailvars); return 1; };
        if (!$success || $@) {
            $log->out("Error sending notification, skipping:", $@);
            next;
        }

        $log->out("Adding force convert for client $client_id at $start_convert_date into $currency with country $country");
        do_update_table(PPC(ClientID => $client_id), 'force_currency_convert', {
            convert_date => $start_convert_date,
            currency => $currency,
            country_region_id => $country,
        }, where => {ClientID => $client_id});
    }
} else {
        $log->out('No clients found to probably force currency convert');
}

if ($prop) {
    $log->out("Writing date $today to property");
    $prop->set($today);
}

$log->out('Sending OK to juggler');
juggler_ok();

$log->out('FINISH');
