#!/usr/bin/perl

=head1 DESCRIPTION

Скрипт для периодической очистки устаревших файлов в МДС

Если у файла есть специальное имя (в mds_custom_names), то при удалении
файла мы не удаляем запись из mds_metadata, а только из mds_custom_names.


Опционально можно указать:
--login $login -- только для пользователя $login
--type $type -- только для типа $type. Обязательно и имеет смысл только если указан --login
--once -- сделать одну итерацию и выйти

=head1 METADATA

<crontab>
    time: */2 * * * *
    sharded: 1
    <switchman>
        group: scripts-other
        <leases>
            mem: 100
        </leases>
    </switchman>
    package: scripts-switchman
</crontab>

=cut

use Direct::Modern;
use Time::HiRes qw//;

use my_inc '..';
use Settings;


use ScriptHelper sharded => 1;

use Direct::Storage;
use Direct::Storage::Types;
use PrimitivesIds;

my ($ONCE, $LOGIN, $TYPE);
extract_script_params(
    "once" => \$ONCE,
    "login=s" => \$LOGIN,
    "type=s" => \$TYPE,
);

my $MAX_ITERATIONS = 10_000;
my $ITERATION_TIMEOUT = 60;

$log->out('START');

my $storage = Direct::Storage->new();

if ($LOGIN) {
    $log->die("--type required") unless $TYPE;
    my $client_id = get_clientid(login => $LOGIN);
    unless ($client_id) {
        $log->die("can not find ClientID for '$LOGIN'");
    }
    $log->out("working only for user $LOGIN ($client_id) and type = $TYPE");
    my $removed = $storage->remove_dangling_files(PPC(ClientID => $client_id), $TYPE, ClientID => $client_id, limit => 10_000_000);
    $log->out(sprintf "removed %d files:", scalar @$removed);
    $log->out($_) for @$removed;
    $log->out("exit");
    exit;
}

for my $iter (1 .. $MAX_ITERATIONS) {
    $log->out("begin iteration");
    my $t0 = Time::HiRes::time();
    for my $type (sort keys %Direct::Storage::Types::MDS_FILE_TYPES) {
        next unless Direct::Storage::Types::mds_check_type_trait($type, 'custom_name');
        my $db = PPC(shard => $SHARD);
        if (Direct::Storage::Types::mds_check_type_trait($type, 'empty_client_id')) {
            # при обработке второго шарда заодно обрабатываем PPCDICT
            if ($SHARD == 2) {
                $db = PPCDICT;
            } else {
                next;
            }
        }
        
        $log->out("removing for type $type");
        my $removed = $storage->remove_dangling_files($db, $type);
        $log->out(sprintf "removed %d files:", scalar @$removed);
        $log->out($_) for @$removed;
    }

    if ($ONCE) {
        $log->out("exit after first iteration");
        last;
    }

    my $iter_time = Time::HiRes::time() - $t0;

    if ($iter_time < $ITERATION_TIMEOUT) {
        $log->out(sprintf("sleep for %.2f seconds", $ITERATION_TIMEOUT - $iter_time));
        sleep $ITERATION_TIMEOUT - $iter_time;
    }
    $log->out("end iteration");
    restart_tracing('worker');
}

$log->out('FINISH');
