#!/usr/bin/perl

=head1 NAME

ppcMoneyOutReminder - уведомления про закончившиеся деньги на общем счёте (через 3 и 7 дней после факта).

=head1 SYNOPSIS

    ./ppcMoneyOutReminder.pl --shard-id XXXX [--ignore-progress --clientid 123]

=head1 METADATA

<crontab>
    time: */59 1-23 * * *
    sharded: 1
    <switchman>
        group: scripts-other
        <leases>
            mem: 250
        </leases>
    </switchman>
    package: scripts-switchman
</crontab>

<juggler>
    host:   checks_auto.direct.yandex.ru
    sharded:        1
    ttl:            3h
    tag: direct_group_internal_systems
</juggler>

=head1 DESCRIPTION

Скрипт ежедневный, но запускаем раз в час, на случай если упадём. Факт того, что действительно отработали за
нужный нам день храним в Property и до какой позиции. Первый час пропускаем на всякий случай, чтобы дать
ppcSendOrderWarnings.pl гарантировано выполнить обработку за предыдущие сутки.

Выбираем все события money_out_wallet и money_out_wallet_with_ao за 7 предыдущих дней и группируем их по кампаниям и типам событий.
Получается, что к каждой кампании будет привязан список вида [X, Y, ...], который значит, что нужные нам события происходили X, Y
дней назад и т.п.

Если минимальное число из этого списка соответсвует периоду, в который мы должны выполнять нотификацию, и на
общем счёте нет денег - то посылаем уведомление.

Фильтрация по минимальному числу из списка обеспечивает то, что клиент не будет получать уведомления очень
часто. Например, если предположить что для кампании результатом было [2,3], то это значит что обычное
уведомление было послано в т.ч. и 2 дня назад, и напомнинать ещё рано.

Для тестирования можно использовать флаг --ignore-progress, в этом случае скрипт можно запускать более чем один
раз в день, и во время каждого запуска будут рассылаться уведомления.

Для работы с конкретным клиентом нужно использовать флаг --clientid, в этом случае скрипт будет отправлять письма 
только для указанного клиента

=cut

use my_inc "..";
use Direct::Modern;

use Date::Calc;
use List::Util qw/max min/;

use Settings;
use Currencies;
use ScriptHelper sharded => 1, 'Yandex::Log' => 'messages';
use Yandex::DBTools;
use WalletUtils;
use Campaign;
use EventLog;
use User;
use Client;
use Notification;
use RBAC2::Extended;
use Property;
use Yandex::ProcInfo;

# В конце выполнения скрипта проверяем, что не вылезли за указанное количество памяти (в мегабайтах). Таким
# образом мы проверяем во время каждого запуска, что мы не вылезаем за указанное выше в метаданных значение.
my $SWITCHMAN_FQDN_MEM = 250;

# Снова напоминать об исчерпании средств через указанное количество дней. Списком, т.к. напоминаний может быть
# несколько.
my %REMIND_ON_DAY = map { $_ => 1 } qw/3 7/;

my $ignore_progress;
my @CLIENTIDS;
extract_script_params(
    'ignore-progress' => \$ignore_progress,
    'clientid=i@' => \@CLIENTIDS,

);

#не запускаем скрипт, если включена проперти money_out_reminder_job_on
my $is_script_off = Property->new("money_out_reminder_job_on");
if ($is_script_off->get(60)) {
    $log->out('script is not active, finishing the execution');
    exit;
}

# Сохраняем прогресс обработки заданий в виде строки "вчерашняя дата,cid". Где cid - это номер кампании, до
# которой мы успели разослать уведомления в предыдущие запуски текущего дня, или 'done' - когда все уведомления
# на указанный день уже разосланы.
my $progress = Property->new("ppcMoneyOutReminder_progress_$SHARD");

$log->out('start');

my $from_timestamp = sprintf '%04d-%02d-%02d',  Date::Calc::Add_Delta_Days(Date::Calc::Today(), -(max keys %REMIND_ON_DAY));
my $to_date = sprintf '%04d-%02d-%02d',  Date::Calc::Add_Delta_Days(Date::Calc::Today(), -1);
my $to_timestamp = "$to_date 23:59:59";

my $current_progress = !$ignore_progress ? ($progress->get // "$to_date,0") : "$to_date,0";
my ($progress_date, $progress_cid) = split /,/, $current_progress;

if ($progress_date eq $to_date && $progress_cid eq 'done') {
    my $msg = "Script has already done its work for period up to $to_date";
    $log->out($msg);
    juggler_ok(description => "Script has already done its work for period up to $to_date");
    exit 0;
}

my %additional_conds;
if (@CLIENTIDS) {
    $log->out('Working with ClientIDs from commandline: ' . join(', ', @CLIENTIDS));
    $additional_conds{ClientID} = \@CLIENTIDS;
}
my $wallets = get_all_sql(
    PPC(shard => $SHARD),
    ["select cid, type, group_concat(distinct to_days(now()) - to_days(eventtime) order by 1) as days from eventlog",
        where => {
            type => [
                $EventLog::EVENTS{money_out_wallet}{type},
                $EventLog::EVENTS{money_out_wallet_with_ao}{type}
            ],
            eventtime__between => [$from_timestamp, $to_timestamp],
            cid__gt => $progress_cid,
            %additional_conds,
        },
     "group by cid, type order by cid, type",
    ],
);

my @notifies;
for my $wallet_info (@$wallets) {
    my $wallet_cid = $wallet_info->{cid};
    my $stop_events_seen_days_ago = $wallet_info->{days};
    my $min_stop_day = min split /,/, $stop_events_seen_days_ago;
    next unless $REMIND_ON_DAY{$min_stop_day};

    my $camp = get_camp_info($wallet_cid);
    if (!$camp) {
        $log->out("Camp $wallet_cid is missing, skipping");
        next;
    }

    WalletUtils::calc_camp_uni_sums_with_wallet($camp);

    # Если ещё есть деньги на кошельке (с учётом возможно включённого автоовердрафта), пропускаем этот кошелёк
    if ($camp->{sums_uni}{sum} <= 0
        || $camp->{sums_uni}{total} + $camp->{sums_uni}->{auto_overdraft_addition} >= $Currencies::EPSILON) {
        next;
    }

    my $user = get_user_data($camp->{uid}, [qw/fio login phone email/]);
    my $client_email = $camp->{email} || $user->{email};

    next unless $client_email;

    # Отправляем нотификацию только если на кошельке нет автоовердрафта, либо если он есть, но порог отключения уже достигнут
    if ($wallet_info->{type} == $EventLog::EVENTS{money_out_wallet}{type} && $camp->{sums_uni}->{auto_overdraft_addition} == 0
        || $wallet_info->{type} == $EventLog::EVENTS{money_out_wallet_with_ao}{type}) {
        push @notifies, {
            uid           => $camp->{uid},
            client_uid    => $camp->{uid},
            cid           => $camp->{cid},
            campaign_id   => $camp->{cid},
            camp_name     => $camp->{name},
            campaign_name => $camp->{name},
            campaign_type => $camp->{type},
            client_id     => $camp->{ClientID},
            client_email  => $client_email,
            fio           => $user->{fio},
            client_login  => $user->{login},
            client_fio    => $user->{fio},
            client_phone  => $user->{phone},
            auto_overdraft_active => ($wallet_info->{type} == $EventLog::EVENTS{money_out_wallet_with_ao}{type}) ? 'true' : 'false',
        };
    }
}

my $rbac = RBAC2::Extended->get_singleton(1);
for my $vars (@notifies) {
    $log->out("Sending notification to $vars->{client_login} ($vars->{client_email}) regarding wallet $vars->{cid}, auto_overdraft: $vars->{auto_overdraft_active}");
    add_notification($rbac, 'active_orders_money_out_reminder', $vars, {mail_fio_tt_name => 'fio'});
    $progress->set("$to_date,$vars->{cid}");
}

$progress->set("$to_date,done");
my $max_proc_memory_mb = int(Yandex::ProcInfo::proc_memory() / (1024 * 1024));
if ($max_proc_memory_mb > $SWITCHMAN_FQDN_MEM) {
    juggler_warn(description => "Successful run, but process rss was $max_proc_memory_mb MB (limit is $SWITCHMAN_FQDN_MEM)");
} else {
    juggler_ok();
}

$log->out("Finish, used $max_proc_memory_mb MB of RAM (limit is $SWITCHMAN_FQDN_MEM)");
