#!/usr/bin/perl

use my_inc "..";



=head1 METADATA

<crontab>
    time: */59 */12 * * *
    <switchman>
        group: scripts-other
    </switchman>
    package: scripts-switchman
</crontab>
<juggler>
    host:   checks_auto.direct.yandex.ru
    tag: direct_gray
    ttl: 1d
</juggler>

=cut

=head1 NAME

    ppcMonitorAutoIncrement.pl
    Мониторинг значений авто-инкрементных полей в базах данных

=cut

use Direct::Modern;

use Yandex::Advmon;
use Yandex::DBTools;

use ScriptHelper 'Yandex::Log' => 'messages';

use Settings;
use ShardingTools qw(ppc_shards);

my @DB_INSTANCES = (
    { db => 'ppc', shards => [ppc_shards()] },
    'ppcdict',
    'monitor',
    'ppclog:heavy',
);

use constant {
    WARN_BORDER => 50,
    CRIT_BORDER => 80,
};

$log->out('START');

my %graphite_stat;
for my $instance (@DB_INSTANCES) {
    $instance = {db => $instance} unless ref($instance);
    for my $shard (@{$instance->{shards}||[undef]}) {
        my $db_name = defined $shard ? "$instance->{db}:$shard" : $instance->{db};
        my $db_name_dotted = $db_name;
        $db_name_dotted =~ s/:/./;

        my $msg_prefix_guard = $log->msg_prefix_guard("[$db_name]");

        $log->out('Fetching auto_increment data');
        my $column_stat = _get_auto_increment_stat($db_name);

        for my $column (@$column_stat) {
            my $proc_used = 100 * $column->{auto_increment} / $column->{maxval};

            $log->out({%$column, proc_used => $proc_used});

            $graphite_stat{ $db_name_dotted }->{ $column->{table_schema} }->{ $column->{table_name} }->{ $column->{column_name} } = {
                auto_increment => $column->{auto_increment},
                proc_used => $proc_used,
            };

            # NB: это только сырые события без мониторинга. нужна проверка, но для нее нужно явное указание имен всех событий.
            juggler_check(
                service => join('.', ('direct.auto_increment_used_percent', $db_name_dotted, $column->{table_schema}, $column->{table_name}, $column->{column_name})),
                description => 'Процент использованных автоинкрементных айдишников',
                value => $proc_used,
                warn => WARN_BORDER,
                crit => CRIT_BORDER,
            );
        }
    }
}

local $Yandex::Advmon::GRAPHITE_PREFIX = sub {[qw/direct_one_min db_configurations/, $Settings::CONFIGURATION]};
monitor_values({auto_increment => \%graphite_stat});

juggler_ok();

$log->out('FINISH');

sub _get_auto_increment_stat {
    my ($db) = @_;

    # копипаста серёжиного запроса из https://wiki.yandex-team.ru/users/ppalex/Notes/#proverkaispolzovanijaavtoinkrementnyxpolejj
    return get_all_sql($db, q!
        SELECT table_schema, table_name, column_name, auto_increment, column_type
             , if(column_type like '%unsigned%', 2, 1) * ((1 << case when column_type like '%bigint%' then 64 when column_type like '%tinyint%' then 8 when column_type like '%mediumint%' then 24 when column_type like '%int%' then 32 else 1 end - 1) - 1) maxval
        FROM INFORMATION_SCHEMA.TABLES
        JOIN INFORMATION_SCHEMA.COLUMNS USING (TABLE_SCHEMA, TABLE_NAME)
        WHERE AUTO_INCREMENT > 0
          AND extra = 'auto_increment'
          AND table_schema NOT IN ('mysql')
    !);
}
