#!/usr/bin/perl

=head1 METADATA

<crontab>
    time: * * * * *
    <switchman>
        group: scripts-other
        <leases>
            mem: 40
        </leases>
    </switchman>
    package: scripts-switchman
</crontab>
<juggler>
    host:   checks_auto.direct.yandex.ru
    tag:    direct_group_internal_systems
</juggler>

=cut

=head1 DESCRIPTION

    Быстрый мониторинг показателей, связанных с биллингом

    На графики смотрит биллинг при выкладках, - https://wiki.yandex-team.ru/Balance/Release/checklist/
        чеклист / "Ручные проверки" / "Проверить, что платежи нормально зачисляются в Директ"

=cut

use Direct::Modern;

use my_inc "..";

use Yandex::Advmon;
use Yandex::TimeCommon;
use Yandex::XsUtils qw/clh_quote/;

use Currency::Rate;
use Currencies;
use Campaign::Types;
use Monitor;
use Property;
use ScriptHelper 'Yandex::Log' => 'messages';
use Settings;
use Tools qw/get_clickhouse_handler/;

# период, по которому строятся графики платежей в advmon
my $ADVMON_TOTAL_SECONDS = 30 * 60;

# шаги сдвига скользящего окна
my @STEPS = (0..4);

# резрешение показателей в monitor_*
my $MONITOR_RESOLUTION = 60;

my @CAMP_TYPES = (qw/mcb geo/, @{ get_camp_kind_types('text_campaign_in_balance', 'cpm') });
my @CURRENCIES = get_currencies_list();


my $MONEY_COEF = 1_000_000;


my $CLH = get_clickhouse_handler('cloud');
$CLH->query_format('JSON');

$log->out("start advmon_data");
advmon_data();

$log->out("start monitor_data");
monitor_data();

juggler_ok();

$log->out("finish");

sub advmon_data {

    my $send_sub = sub {
        # mct = method-currency-type
        my ($period, $mct_data) = @_;

        my %advmon_data;

        for my $method (keys %$mct_data) {
            my $ct_data = $mct_data->{$method} //= {};

            for my $currency (@CURRENCIES) {
                my $t_data = $ct_data->{$currency} //= {};

                for my $type (@CAMP_TYPES) {
                    my $data = $t_data->{$type} //= {};
                    for my $measure (qw/sum_delta cnt/) {
                        # дополняем нулями
                        $data->{$measure} ||= 0;

                        # старые данные без мультивалютности и методов
                        $advmon_data{$type}{$measure} += $data->{$measure};
                    }

                    # добавляем данные для всех валют
                    my $chips = convert_currency($data->{sum_delta}, $currency, 'YND_FIXED', with_nds => 1);
                    $mct_data->{TOTAL_CHIPS_BY_TYPE}->{$type}->{sum_delta} += $chips;
                    $mct_data->{TOTAL_CHIPS_BY_TYPE}->{$type}->{cnt} += $data->{cnt};

                    $mct_data->{GRAND_TOTAL_CHIPS}->{sum_delta} += $chips;
                    $mct_data->{GRAND_TOTAL_CHIPS}->{cnt} += $data->{cnt};
                }
            }
        }

        for my $type (@CAMP_TYPES) {
            $advmon_data{$type}{sum_delta} /= $MONEY_COEF;
        }

        $log->out("advmon data for $period: ", \%advmon_data);
        monitor_values({balance_change => \%advmon_data}, time => mysql2unix($period));

        $log->out({period => $period, total_stat => $mct_data->{GRAND_TOTAL_CHIPS}});
        local $Yandex::Advmon::GRAPHITE_PREFIX = sub {[qw/direct_one_min db_configurations/, $Settings::CONFIGURATION]};
        monitor_values({flow => {balance_change => $mct_data}}, time => mysql2unix($period));
    };

    # каждую минуту пересчитываем данные за последние
    # полчаса на случай если доехали логи
    my $interval = clh_quote($ADVMON_TOTAL_SECONDS);
    my $steps = clh_quote(join ',', @STEPS);
    my $steps_count = clh_quote(scalar(@STEPS));

    my $query = <<__QUERY__;
SELECT
    method,
    currency,
    type,
    logtime_step + ($steps_count * 60) AS period,
    SUM(sum_delta) AS sum_delta,
    COUNT(*) AS cnt
FROM
(
    SELECT
        toStartOfMinute(log_time) AS logtime_start,
        toStartOfMinute(log_time - (60 * arrayJoin([$steps]))) AS logtime_step,
        method,
        type,
        sum_delta,
        currency
    FROM campaign_balance
    WHERE (log_date >= toDate(now() - $interval)) AND (log_time >= toStartOfMinute(now() - $interval))
)
GROUP BY
    logtime_step,
    method,
    currency,
    type
HAVING (logtime_step >= toStartOfMinute(now() - $interval)) AND (logtime_step <= toStartOfMinute(now() - $steps_count * 60))
ORDER BY period ASC
__QUERY__

    my $payments = $CLH->query($query)->json->{data};

    my $period = '';
    my %data = ();

    # цикл по минутам
    for my $row (@$payments) {

        # начался новый период
        if ($period ne $row->{period}) {

            # если предыдущий период был, то посылаем данные в мониторинг
            if ($period) {
                $send_sub->($period, \%data);
            }

            $period = $row->{period};
            %data = ();
        }

        $log->out("process time: $period");

        $data{ $row->{method} }->{ $row->{currency} }->{ $row->{type} } = {
            sum_delta => $row->{sum_delta},
            cnt       => $row->{cnt},
        };
    }

    $send_sub->($period, \%data) if @$payments;
}

sub monitor_data {
    my $prop = Property->new('MONITOR_BALANCE_LAST_START');

    my $send_sub = sub {
        my ($period, $data) = @_;

        for my $type (@CAMP_TYPES) {
            Monitor::save_target_value(
                "direct.balance_change.$type.sum_delta", 
                {
                    value => ($data->{$type}{sum_delta} || 0) / $MONEY_COEF,
                    desc => "Изменение баланса на кампаниях типа $type за $MONITOR_RESOLUTION секунд",
                }, 
                mysql2unix($period)
            );
            Monitor::save_target_value(
                "direct.balance_change.$type.cnt", 
                {
                    value => $data->{$type}{cnt} || 0,
                    desc => "Количество транзакций по кампаниям типа $type за $MONITOR_RESOLUTION секунд",
                }, 
                mysql2unix($period)
            );
        }
    };

    my $seconds = $prop->get() ? 5 * $MONITOR_RESOLUTION : 30 * $MONITOR_RESOLUTION;

    my $interval = clh_quote($seconds);

    my $query = <<__QUERY__;
SELECT
    toStartOfMinute(log_time) as period, type,
    sum(sum_delta) as sum_delta, count() as cnt
FROM
    campaign_balance
WHERE
        log_date = toDate(now() - $interval)
    AND log_time >= toStartOfMinute(now() - $interval)
GROUP BY period, type
ORDER BY period ASC
__QUERY__

    my $payments = $CLH->query($query)->json->{data};

    my $period = '';
    my %data = ();

    # цикл по минутам
    for my $row (@$payments) {
        # начался новый период
        if ($period ne $row->{period}) {

            # если предыдущий период был, то посылаем данные в мониторинг
            if ($period) {
                $send_sub->($period, \%data);
            }

            $period = $row->{period};
            %data = ();
        }

        $log->out("process time: $period");

        $data{ $row->{type} } = {
            sum_delta => $row->{sum_delta},
            cnt       => $row->{cnt},
        };
    }

    $send_sub->($period, \%data) if @$payments;

    $prop->set($period);
}
