#!/usr/bin/perl

use my_inc "..";


=head1 METADATA

<crontab>
    time: */5 * * * *
    <switchman>
        group: scripts-other
    </switchman>
    package: scripts-switchman
</crontab>
<crontab>
    time: */5 * * * *
    <switchman>
        group: scripts-test
    </switchman>
    package: conf-test-scripts
</crontab>

<juggler>
    host:   checks_auto.direct.yandex.ru
    ttl: 15m
    tag: direct_group_internal_systems
</juggler>
# мониторинг давности последней обработки результатов модерации
<juggler_check>
    host:   checks_auto.direct.yandex.ru
    name:           direct.process_moderate_result.last_time_ago
    raw_events:     direct.process_moderate_result.last_time_ago.production
    ttl:            15m
    tag: direct_group_internal_systems
</juggler_check>

=cut

=head1 NAME

ppcMonitorModerateResults.pl

=head1 DESCRIPTION

    Запись в графит количества принятых и обработанных результатов модерации.

    Измерение времени с последней успешной обработки результатов модерации и зажигание монитора если оно слишком высокое
    Время считается по последней записи статистики в Monitor.pm под названием moderate_results.objects.processed

=cut


use Direct::Modern;

use Yandex::Advmon;
use Yandex::HashUtils;

use Monitor;

use ScriptHelper 'Yandex::Log' => 'messages';
use Settings;

# пороговые значения кол-ва секунд с последней обработки результатов модерации, после которых зажигаются мониторинги.
use constant WARN_BORDER_SECONDS => 5 * 60;
use constant CRIT_BORDER_SECONDS => 20 * 60;

# интервал за который будем суммировать количество принятых и обработанных объектов
use constant STAT_INTERVAL => '5min';

my $target_names = [qw/
    moderate_results.objects.received
    moderate_results.objects.processed
    moderate_results.magic_queue_items.received
    moderate_results.magic_queue_items.processed
/];

while(1) {
    if (my $reason = smart_check_stop_file()) {
        $log->out("$reason! Exiting.");
        exit 0;
    } else {
        restart_tracing();
    }

    $log->out({message => 'START'});

    my $stats = Monitor::get_values_for_interval(targets => $target_names, interval => STAT_INTERVAL);

    # приписываем интервал к названиям статистик
    $stats = hash_kmap { $_.'_'.STAT_INTERVAL } $stats;

    my $last_processed_time = Monitor::get_last_target_time('moderate_results.objects.processed');
    my $last_processed_ago;
    if (defined $last_processed_time) {
        $last_processed_ago = time - $last_processed_time;
        $stats->{'moderate_results.objects.last_processed_ago'} = $last_processed_ago;
    }

    juggler_check(service => 'direct.process_moderate_result.last_time_ago',
        description => 'кол-во секунд с последней обработки результатов модерации',
        value => $last_processed_ago,
        warn => WARN_BORDER_SECONDS,
        crit => CRIT_BORDER_SECONDS);

    $log->out($stats);

    local $Yandex::Advmon::GRAPHITE_PREFIX = sub {[qw/direct_one_min db_configurations/, $Settings::CONFIGURATION, qw/flow/]};
    monitor_values($stats);

    juggler_ok();

    $log->out({message => 'FINISH'});

    sleep(120);
}
