#!/usr/bin/perl

use my_inc "..";



=head1 METADATA

<crontab>
    time: 12 0-7 * * *
    <switchman>
        group: scripts-other
        <leases>
            mem: 50
        </leases>
    </switchman>
    package: scripts-switchman
</crontab>
<juggler>
    host:   checks_auto.direct.yandex.ru
    tag: direct_gray
    <meta_url>
        title:  Документация
        url:    https://docs.yandex-team.ru/direct-dev/reference/perl-scripts/list/ppcMonitorTablesSize
    </meta_url>
    ttl: 3d8h
</juggler>

=cut

# $Id$

=head1 NAME

    ppcMonitorTablesSize.pl
    Мониторинг размеров таблиц в базах данных

=head1 DESCRIPTION

    запускается раз в сутки и записывает в monitor_* физические фарактеристики таблиц
    в базах данных

=cut

use warnings;
use strict;

use Yandex::Advmon;
use Yandex::DBTools;
use Yandex::TimeCommon;

use ScriptHelper 'Yandex::Log' => 'messages';

use Settings;
use Monitor;
use ShardingTools;
use Property;

use utf8;

# сколько раз спрашиваем table status - чем больше, тем точнее
my $STATUS_TRIES = 100;

my @DB_INSTANCES = (
    { db => 'ppc', shards => [ppc_shards()], },
    { db => PPCDICT, },
    { db => MONITOR, },
    );

$log->out("start, configuration: $Settings::CONFIGURATION");

# чтобы не запускаться несколько раз за один день
my $now = today();
my $laststart_prop = new Property('MONITOR_TABLES_SIZE_LASTSTART'); 
my $laststart = $laststart_prop->get() || '';
if ($laststart eq $now ) {
    $log->out("nothing to do for today...");
    juggler_ok(description => 'already worked today');
    exit;
}

# для таблиц с датами - выводим данные по вчерашней таблице
my $yesterday = yesterday();

my $monitor_values = {};
my $graphite = {};

for my $instance (@DB_INSTANCES) {
    for my $shard (@{$instance->{shards}||[undef]}) {
        my $db = defined $shard ? "$instance->{db}:$shard" : $instance->{db};
        # выбраем только известные нам базы, а то грантов не хватит
        my @databases = grep { /^(?:ppc|ppcdict|monitor)$/} @{get_one_column_sql($db, "show databases") || []};
        for my $database (sort @databases) {
            my @tables = sort @{ get_one_column_sql($db, "show tables from $database") || [] };
            my %stat;
            TABLE: for my $table (@tables) {
                $log->out("table $db.$database.$table");
                (my $table_nick = $table) =~ s/_$yesterday$/_YESTERDAY/;
                next if $table_nick =~ /_\d{8}$/;
                # пропускаем временные таблицы
                next if $table_nick =~ /^(?:suggest_phrases_tmp|suggest_phrases_links_tmp)$/;
                my $avg_status = {};
                # для увеличения точности - запрашиваем table status несколько раз
                for (1..$STATUS_TRIES) {
                    my $status = get_one_line_sql($db, "show table status from $database like ?", $table);
                    if (!keys %$status) {
                        $log->warn("no status for $db.$database.$table, skipping");
                        next TABLE;
                    }
                    for my $key (qw/Rows Data_length Index_length Data_free/) {
                        $avg_status->{$key} += ($status->{$key}||0) / $STATUS_TRIES;
                    }
                }
                my %stat = (
                    rows => [int($avg_status->{Rows}), "количество строк"],
                    data_length => [int($avg_status->{Data_length}), "объём данных"],
                    index_length => [int($avg_status->{Index_length}), "объём индекса"],
                    data_free => [int($avg_status->{Data_free}), "объём неиспользуемого места"],
                );
                my $table_desc = get_all_sql($db, "DESC $database.$table");
                if (my ($field) = grep {$_->{Extra} =~ /auto_increment/i} @$table_desc) {
                    my ($min, $max) = get_one_line_array_sql($db, "SELECT min($field->{Field}), max($field->{Field}) FROM $database.$table");
                    $stat{max_autoincr} = [$max || 0, "максимальное значение auto_increment"];
                    $stat{min_autoincr} = [$min || 0, "минимальное значение auto_increment"];
                }
                
                while(my ($name, $data) = each %stat) {
                    $monitor_values->{"db.$db.$database.$table_nick.$name"} = {
                        value => $data->[0],
                        desc => "Таблица $db.$database.$table_nick - $data->[1]",
                    };
                    $graphite->{"$database.$table_nick.$name".(defined $shard ? ".shard$shard" : "")} = $data->[0];
                }
            }
        }
    }
}
Monitor::save_values($monitor_values);

local $Yandex::Advmon::GRAPHITE_PREFIX = sub {[qw/direct_one_min db_configurations/, $Settings::CONFIGURATION]};
monitor_values({ 
    objects => {db => $graphite},
});

$laststart_prop->set($now);
juggler_ok();
$log->out("finish");
