#!/usr/bin/perl

=head1 NAME

=head1 DESCRIPTION

=head1 METADATA

<crontab>
    time: */10 * * * *
    <switchman>
        group: scripts-other
        <leases>
            mem: 45
        </leases>
    </switchman>
    package: scripts-switchman
</crontab>
<juggler>
    host:   checks_auto.direct.yandex.ru
    ttl: 35m
    tag: direct_group_sre
</juggler>

<juggler_check>
    host:   checks_auto.direct.yandex.ru
    name:           direct.sharding.schema_diffs
    raw_events:     direct.sharding.schema_diffs.production
    ttl:            1h
    tag: direct_group_sre
</juggler_check>

=cut

use Direct::Modern;

use Text::Diff;

use Yandex::DBTools;
use Yandex::ListUtils;

use my_inc "..";

use ScriptHelper;
use Settings;
use ShardingTools;

my $SKIP_RE = qr/^(warnings_.*)$/;

extract_script_params();


my $schema = get_shards_schema();
my @changes = find_schema_changes($schema);
show_changes(\@changes);

juggler_check(service => 'direct.sharding.schema_diffs',
              description => 'Количество отличий в схеме таблиц по шардам',
              value => scalar(@changes),
              crit => 1,
              );

juggler_ok();

sub get_shards_schema {
    my %schema;
    for my $shard (ppc_shards()) {
        for my $table (@{get_one_column_sql(PPC(shard=>$shard), "show tables") || []}) {
            next if $table =~ $SKIP_RE;
            my (undef, $sql) = get_one_line_array_sql(PPC(shard => $shard), "show create table ".sql_quote_identifier($table));
            $sql =~ s/\s+AUTO_INCREMENT=\d+//;
            $schema{$table}{$shard} = $sql;
        }
    }
    return \%schema;
}


sub find_schema_changes {
    my ($schema) = @_;

    my @shards = nsort ppc_shards();
    my $first_shard = $shards[0];
    my @rest_shards = @shards[1..$#shards];

    my @changes;
    for my $table (sort keys %$schema) {
        my $tinfo = $schema->{$table};
        for my $shard (@rest_shards) {
            my $add_err = sub {
                push @changes, {
                    code => $_[0],
                    msg => $_[1],
                    details => $_[2],
                    shard => $shard, 
                    table => $table,
                };
            };
            if (!exists $tinfo->{$first_shard} && exists $tinfo->{$shard}) {
                $add_err->(excess => "Table $table not exists in shard $first_shard, but exists in shard $shard",
                           $tinfo->{$shard});
            } elsif (exists $tinfo->{$first_shard} && !exists $tinfo->{$shard}) {
                $add_err->(missed => "Table $table not exists in shard $shard",
                           $tinfo->{$first_shard});
            } elsif (($tinfo->{$shard}||'') ne ($tinfo->{$first_shard}||'')) {
                $add_err->(differs => "Table $table differs in shard $first_shard and $shard",
                           diff(\$tinfo->{$first_shard}, \$tinfo->{$shard}));
            }
        }
    }
    return @changes;
}


sub show_changes {
    my ($changes) = @_;
    for my $c (@$changes) {
        print "ppc:$c->{shard}, $c->{table} - $c->{code} ($c->{msg})\n";
        print $c->{details}, "\n";
        print "\n";
    }
}
