#!/usr/bin/perl

=head1 METADATA

<crontab>
    params: --db=ppc
    sharded: 1
    time: 20 1 * * *
    <switchman>
        group: scripts-other
        <leases>
            mem: 100
        </leases>
    </switchman>
    package: scripts-switchman
</crontab>
<juggler>
    host:   checks_auto.direct.yandex.ru
    name:       scripts.ppcMysqlConsistencyMonitor.working.ppc
    raw_events: scripts.ppcMysqlConsistencyMonitor.working.ppc.shard_$shard
    sharded:    1
    ttl:        2d8h
    tag: direct_group_sre
</juggler>

#<crontab>
#    params: --db=ppcdict
#    time: 50 1 * * *
#    <switchman>
#        group: scripts-other
#        <leases>
#            mem: 100
#        </leases>
#    </switchman>
#    package: scripts-switchman
#</crontab>
<juggler>
    host:   checks_auto.direct.yandex.ru
    name:       scripts.ppcMysqlConsistencyMonitor.working.ppcdict
    raw_events: scripts.ppcMysqlConsistencyMonitor.working.ppcdict
    ttl:        2d8h
    tag: direct_group_sre
</juggler>

=cut

=head1 NAME

    ppcMysqlConsistencyMonitor.pl - проверка крнсистентности реплик.

=head1 DESCRIPTION

    Скрипт запускает mk-table-checksum на мастерах нескольких базах,
    указанных прямо в скрипте.
    В табличках checksum на репликах оказываются результаты проверки,
    следить за ними должны локальные мониторинги.

    Возможные опции:
    --db - выполнить проверку только для указанной базы
    --shard-id - выполнить проверку только для указанного шарда
    --debug - выводить информацию в STDOUT, анализировать только часть таблиц, работать быстрее и агрессивнее ;-)
    --help - справка

=cut

use Direct::Modern;

use Path::Tiny;
use POSIX qw/strftime/;
use List::Util qw/shuffle/;
use List::MoreUtils qw/uniq/;

use my_inc "..";

use Yandex::HashUtils;
use Yandex::Shell;

use ScriptHelper 'Yandex::Log' => 'messages', get_file_lock => undef;

use Settings;
use Yandex::DBTools;
use LockTools;
use ShardingTools;

my $DEBUG;
my ($DB, $ORIGINAL_DB);
my $SHARD;
my $ALL_ROWS;
extract_script_params(
    'db=s' => \$ORIGINAL_DB,
    'shard-id=i' => \$SHARD,
    'all-rows' => \$ALL_ROWS,
    'debug' => \$DEBUG,
);

$DB = $ORIGINAL_DB;

if (defined $SHARD && defined $DB) {
    $DB = "$DB:$SHARD";
}

$log->msg_prefix($DB);

# лочим файлик
get_file_lock(undef, "ppcMysqlConsistencyMonitor.".( ($DB//'all') =~ s/:/_/gr ).".lock");

# аргументы mk-table-checksum по-умолчанию, на них накладываются
# агрументы, специфичные для конкретной базы
my %DEFAULT_ARGS = (
    '--nocheck-replication-filters' => undef,
    '--throttle-method' => 'none',

    '--algorithm' => 'ACCUM',
    '--float-precision' => 2,

    '--chunk-size' => 50000, 
    '--chunk-size-limit' => 10000,
    '--sleep-coef' => 3,
    '--unchunkable-tables' => undef,
    '--set-vars' => 'tx_isolation="REPEATABLE-READ"',

    '--quiet' => undef,
    );

if ($DEBUG) {
    # для дебаговых быстрых запусков
    hash_merge \%DEFAULT_ARGS, {
        '--probability' => '1',
    };
    delete @DEFAULT_ARGS{qw/--quiet --sleep-coef/};
    $log->{tee} = 1;
}

# список проверяемых баз и их специфичные агументы
my $pt_sleep_plugin = pt_sleep_plugin(1);
my %CONFIG = (
    (map {(
        "ppc:$_" => [
            {
                # warnings_.* - пишутся в слейва
                '--ignore-tables-regex' => '^(bids|bids_arc|bids_href_params|banners|phrases|sitelinks_links|bs_auction_stat|bids_phraseid_associate|catalogia_banners_rubrics|client_firm_country_currency|eventlog|warnings_.*|checksum|checksum_result|bsClient.*)$',
                '--probability' => 100,
            },
            {
                # bids - тоже большая, чексуммим не всю таблицу, а только N процентов
                '--tables' => 'bids,bids_arc,bids_href_params,banners,phrases,sitelinks_links',
                '--probability' => 15,
            },
            {
                # эти тоже большие ж-(
                '--tables' => 'bs_auction_stat,bids_phraseid_associate,catalogia_banners_rubrics,client_firm_country_currency,eventlog',
                '--probability' => 7,
            },
        ],
    )} ppc_shards()),
    PPCDICT() => [
        {
            # shard_login - PRIMARY для разбиения на чанки не подходит, проверяем дальше отдельно
            # остальные таблицы - плохие индексы, таблицы не бъются по чанкам
            '--ignore-tables-regex' => '^(advq_cache|api_domain_stat|checksum|checksum_result|mirrors|shard_login)$',
            '--recursion-method' => 'none',
            '--chunk-size' => 10000,
            '--chunk-size-limit' => 2,
            '--plugin' => "$pt_sleep_plugin",
        },
        {
            '--tables' => 'shard_login',
            '--recursion-method' => 'none',
            '--chunk-index' => 'uid',
            '--chunk-size' => 10000,
            '--chunk-size-limit' => 2,
            '--plugin' => "$pt_sleep_plugin",
        },
    ],
);

my %CHECKSUM_PROGRAM_OVERRIDE = (
    PPCDICT() => 'pt-table-checksum',
);

my %ARGS_BLACKLIST = (
    PPCDICT() => [
        qw(
            --unchunkable-tables
            --throttle-method
            --sleep-coef
            --algorithm
            --probability
        ),
    ],
);

$log->out("start");
for my $dbname (grep { !defined $DB || /^\Q$DB\E(:|$)/ } sort keys %CONFIG) {
    $log->out("db $dbname");
    debug_log_db_params($dbname);
    $log->out('change transaction isolation level');
    do_sql($dbname, "SET SESSION TRANSACTION ISOLATION LEVEL REPEATABLE READ");
    debug_log_db_params($dbname);

    # добавляем данные о коннекте
    my $cfg = get_pxc_replica_config($dbname) // get_db_config($dbname);

    my %dsn_params = (
        h => $cfg->{host},
        P => $cfg->{port} || 3306,
        u => $cfg->{user},
        p => $cfg->{pass},
    );

    my $connect = join( ',', map { $_ . '=' . escape_dsn_param( $dsn_params{$_} ) } sort keys %dsn_params );

    # ищем, какие таблицы на репликах отличаются от мастера
    my $tables = get_one_column_sql($dbname, "SHOW TABLES") || [];
    my %changed_tables;
    my $master_db = get_db_config($dbname)->{db};
    my @replicas = grep {get_db_config($_)->{db} eq $master_db} uniq @{get_db_childs($dbname)};
  TABLE:
    for my $table (@$tables) {
        next if $table =~ /^bsClient.*/;
        my $master_ddl = get_ddl($dbname, $table);
        for my $repl (@replicas) {
            if (!is_table_exists($repl, $table) || $master_ddl ne get_ddl($repl, $table)) {
                $changed_tables{$table} = undef;
                next TABLE;
            }
        }
    }

    my $checksum_table = $cfg->{db}.".checksum";
    
    my $now = get_one_field_sql($dbname, "SELECT now()");

    # информация и имени базы
    die "Database for $dbname is not defined" if !$cfg->{db};

    my @system_errors;
    for my $config_args (ref $CONFIG{$dbname} eq 'ARRAY' ? @{$CONFIG{$dbname}} : ($CONFIG{$dbname})) {
        my $args = hash_merge {}, \%DEFAULT_ARGS, $config_args;
        
        if (%changed_tables) {
            push @system_errors, "Ignore changed tables on $dbname: ".join(', ', sort keys %changed_tables);
            my @ignore = keys %changed_tables;
            push @ignore, split ',', $args->{'--ignore-tables'} if $args->{'--ignore-tables'};
            $args->{'--ignore-tables'} = join ',', uniq @ignore;
        }
    
        $args->{'--databases'} ||= $cfg->{db};
        $args->{'--replicate'} ||= $checksum_table;

        my $program = $CHECKSUM_PROGRAM_OVERRIDE{$dbname} || 'mk-table-checksum';

        delete @$args{ @{ $ARGS_BLACKLIST{$dbname} } } if $ARGS_BLACKLIST{$dbname};
        delete $args->{'--probability'} if $ALL_ROWS;

        # выполняем запрос
        my @cmd = ($program, $connect, grep {defined} %$args);
        $log->out(\@cmd);
        eval {
            my $profile = Yandex::Trace::new_profile('ppcMysqlConsistencyMonitor:mk-table-checksum', tags => "dbname,$dbname");
            yash_system(@cmd);
        };
        if ($@) {
            my $error = $@;
            $log->warn("System error: $error");
            push @system_errors, $error;
        }
    }

    debug_log_db_params($dbname);
    $log->out('change binlog format');
    do_sql( $dbname, q{set session binlog_format="statement"} );
    debug_log_db_params($dbname);
    $log->out('change transaction isolation level');
    do_sql($dbname, "SET SESSION TRANSACTION ISOLATION LEVEL REPEATABLE READ");
    debug_log_db_params($dbname);

    do_in_transaction {
        $log->out('process checksum tables');
        debug_log_db_params($dbname);
        # удаляем записи, оставшиеся от предыдущего запуска
        do_sql($dbname, "DELETE FROM $checksum_table WHERE ts < ?", $now);

        # записываем результат выполнения
        do_mass_insert_sql($dbname, "REPLACE INTO $cfg->{db}.checksum_result (name, value) VALUES %s", 
                           [
                            [system_error => join('; ', @system_errors)], 
                            [check_time => strftime("%Y-%m-%d %H:%M:%S", localtime)],
                           ] );
        do_sql($dbname, "
                    REPLACE INTO $cfg->{db}.checksum_result
                            (name, value)
                            SELECT 'errors_cnt', count(*)
                              FROM checksum
                             WHERE this_crc != master_crc
                                OR this_cnt != master_cnt
                        ");
    };
}

juggler_ok(service_suffix => join('.', $ORIGINAL_DB, ($SHARD ? "shard_$SHARD" : ())));

$log->out("finish");

sub get_ddl {
    my ($db, $table) = @_;
    my $ddl = [get_one_line_array_sql($db, "SHOW CREATE TABLE ".get_dbh($db)->quote_identifier($table))]->[1];
    # автоинкременты со стартом могут отсутствовать, если нет строк в таблице и сервер перезапускался
    $ddl =~ s/(\s+AUTO_INCREMENT=)\d+//;
    $ddl =~ s/\s+/ /g;
    return $ddl;
}

sub escape_dsn_param {
    my ($param) = @_;
    $param =~ s/,/\\,/g;
    return $param;
}

=head2 pt_sleep_plugin(times)

    pt-table-checksum не умеет делать паузы между запросами
    плагинная система не гибка, и нормальным образом через плагины нельзя добиться такого поведения
    поэтому грязно манкипатчим MySQLStatusWaiter - проверку количества активных сессий

=cut
sub pt_sleep_plugin {
    my ($times) = @_;
    croak "incorrect times" unless $times =~ /^[0-9]+$/;
    my $file = Path::Tiny->tempfile();
    $file->append_utf8(q%
        package pt_table_checksum_plugin;
        
        require Time::HiRes;
        
        sub new { bless {}; }
        
        my $old_waiter = \&MySQLStatusWaiter::wait;
        my $prev_time;
        *MySQLStatusWaiter::wait = sub {
            if ($prev_time) {
                my $pause = %.$times.q% * (Time::HiRes::time - $prev_time);
                Time::HiRes::sleep($pause);
            }
            $prev_time = Time::HiRes::time;
            goto &$old_waiter;
        };
        
        1;    
    %);
    return $file;
}


=head2

    Для работы, pt-table-checksum переключает репликацию в statement-based, 
    это включает TOI и мастер-нода ждёт завершения всех запросов на репликах.
    Идея - натравливать pt-table-checksum на другую ноду PXC, эксперимент показал, что затупы пропадают
    
    Функция для PXC-базы находит 
    - живую реплику (не мастер и не standby)
    - определяет, если есть extra_port (чтобы обойти portblocker)
    - возвращает ссылку на хэш с данными для коннекта, в стиле DBTools

    - если реплика не нашлась - возвращается undef

=cut
sub get_pxc_replica_config {
    my ($db) = @_;

    my $master_config = get_db_config($db);

    my $vars = get_hash_sql($db, "SHOW VARIABLES");
    if (!$vars->{wsrep_cluster_name}) {
        return undef;
    }
    my $replica_port = $vars->{extra_port} // $vars->{port};
    my $hosts = $vars->{wsrep_cluster_address};
    $hosts =~ s/^gcomm:\/\///;
    $hosts =~ s/\/.*//;
    my @hosts = shuffle uniq grep {!/standby/ && !/^\Q$vars->{wsrep_node_name}\E($|\.)/} map {s/:\d+$//r} split /,/, $hosts;

    my $status = get_hash_sql($db, "SHOW STATUS");
    my $cluster_state_uuid = $status->{wsrep_cluster_state_uuid};

    for my $host (@hosts) {
        my $config = {%$master_config, port => $replica_port, host => $host};
        my $host_state_uuid = eval {
            get_hash_sql(Yandex::DBTools::connect_db($config), "SHOW STATUS LIKE 'wsrep_local_state_uuid'")->{wsrep_local_state_uuid};
        };
        if ($@) {
            warn "Error while probe $host: $@\n";
        }
        if ($host_state_uuid && $host_state_uuid eq $cluster_state_uuid) {
            return $config;
        }
    }

    return undef;
}

sub debug_log_db_params {
    my $db = shift;
    $log->out(get_one_line_sql($db, 'SELECT @@GLOBAL.binlog_format, @@binlog_format, @@GLOBAL.tx_isolation, @@tx_isolation'));
}
