#!/usr/bin/perl

=head1 METADATA

<crontab>
    time: */2 * * * *
    params: --queue=std
    sharded: 1
    <switchman>
        group: scripts-other
        <leases>
            mem: 1100
        </leases>
    </switchman>
    package: scripts-switchman
</crontab>
<crontab>
    time: */10 * * * *
    params: --queue=heavy
    sharded: 1
    <switchman>
        group: scripts-other
        <leases>
            mem: 2300
        </leases>
    </switchman>
    package: scripts-switchman
</crontab>
<crontab>
    time: */10 * * * *
    params: --queue=heavy2
    sharded: 1
    <switchman>
        group: scripts-other
        <leases>
            mem: 2300
        </leases>
    </switchman>
    package: scripts-switchman
</crontab>
<juggler>
    host:   checks_auto.direct.yandex.ru
    raw_events:     scripts.ppcPdfReports.working.$queue.shard_$shard
    sharded:        1
    vars:           queue=std
    vars:           queue<ttl=40m>=heavy,heavy2
    ttl:            20m
    tag: direct_group_internal_systems
</juggler>

<crontab>
    time: */5 * * * *
    params: --queue=std
    sharded: 1
    flock: 1
    <switchman>
        group: scripts-test
    </switchman>
    package: conf-test-scripts
</crontab>
<crontab>
    time: */10 * * * *
    params: --queue=heavy
    sharded: 1
    flock: 1
    <switchman>
        group: scripts-test
    </switchman>
    package: conf-test-scripts
</crontab>

=cut

# $Id$

=head1 NAME

ppcPdfReports.pl - Обработка очереди запросов на создание PDF отчётов.

=cut

=head1 DESCRIPTION

    Опции командной строки:
    --help - вывести справку
    --shard-id - номер обрабатываемого шарда
    --queue - какую очередь обрабатывать (std|heavy)
    --once - выполнить одну итерацию и выйти

=head2 QUEUES

    Условия на тип очереди записаны в %PdfReport::QUEUES
    В "тяжелую" очередь попадают отчеты с одним из следующих условий
        * фраз больше $PdfReport::Queue::HEAVY_PHRASES (60000)
        * период отчета больше $PdfReport::Queue::HEAVY_PERIOD_DAYS дней (100)
        * отчет заказан для количества кампаний, превышающего $PdfReport::Queue::HEAVY_CIDS_COUNT (4)

=head1 RUNNING

     LOG_TEE=1 ./protected/ppcPdfReports.pl --shard-id=1 --queue=std --once

=cut

use Cairo;
use Date::Calc;
use Encode qw/is_utf8 encode/;
use File::Slurp;
use LaTeX::Driver;
use List::MoreUtils qw(uniq any none);
use List::Util qw/min max maxstr minstr/ ;
use POSIX qw/ceil floor/;
use Template;
use List::Util qw/maxstr/;

use Direct::Modern;

use Yandex::Advmon;
use Yandex::DBTools;
use Yandex::HashUtils;
use Yandex::I18n;
use Yandex::ListUtils qw/nsort/;
use Yandex::ProcInfo;
use Yandex::SendMail;
use Yandex::TimeCommon qw/str_round_day/;

use my_inc "..";

use CairoGraph;
use Client;
use Currency::Format;
use geo_regions;
use GeoTools;
use LockTools;
use PdfReport::Queue;
use PdfReport;
use PdfTools;
use PrimitivesIds;
use RBAC2::Extended;
use RBACElementary;
use ScriptHelper 'get_file_lock' => undef,
            script_timer => undef,
            'Yandex::Log' => 'messages',
            sharded => 1;
use Settings;
use Stat::Const;
use Stat::CustomizedArray;
use Stat::Tools;
use TeXEscape qw/tex_escape/;
use User qw(get_user_data);

use open ':std' => ':utf8';

our $DEBUG = 0;
our $PDF_REPORT_DIR = $Settings::ROOT.'/data/pdf_reports/';

=head1 SUBROUTINES/METHODS/VARIABLES

=head2 $MIN_ITERATION_DURATION

    минимальное время итерации
    если итерация закончилась быстрее - делаем sleep

=cut
our $MIN_ITERATION_DURATION //= 10;

my $GRAPHICS_DIR = "/tmp/";
my $TEX_TEMPLATE = $PDF_REPORT_DIR.'yareport05.tex';
my $MAX_WORD_LENGTH = 15;

# максимальный размер файла, по достижению которого процесс перезапустится
my $MAX_PROCESS_SIZE = 1024 * 1024 * 1024;

my ($QUEUE, $ONCE);
extract_script_params(
    'queue=s' => \$QUEUE,
    'once' => \$ONCE,
);

unless ($QUEUE && exists $PdfReport::Queue::QUEUES{ $QUEUE }) {
    die "wrong value for param --queue, should be one of: " . join(', ', keys %PdfReport::Queue::QUEUES);
}

get_file_lock($ONCE ? 0 : 'dont_die', get_script_name() . "_$QUEUE");

$log->msg_prefix("[$QUEUE]");
$log->out("PDF Report Daemon started");

$log->die("No such file [$TEX_TEMPLATE]") unless -e $TEX_TEMPLATE;
my $doc_text = Encode::decode_utf8(read_file($TEX_TEMPLATE)) or $log->die("Can't open file [$TEX_TEMPLATE]");

my $rbac;

while (1) {
    $ScriptHelper::trace = Yandex::Trace->new(service => 'direct.script', method => 'ppcPdfReports', tags => "shard_$SHARD,queue_$QUEUE");

    if (my $reason = smart_check_stop_file()) {
        $log->out("$reason! Let's finish.");
        last;
    }

    my $iter_start_time = time();

    $rbac = eval { RBAC2::Extended->get_singleton(1) }
        or $log->die("Error initialising RBAC: $@");

    # Пробуем взять из очереди отчет и обработать
    my $res = eval { PdfReport::Queue::popReportOrder($QUEUE, {pdf_proc => \&makePdf, shard => $SHARD, log => $log}) };
    my $description;
    if (!defined $res) {
        $log->die('Failed to make pdf_report: ' . ($@ // ''));
    } elsif (ref $res eq 'HASH') {
        $description = "Обработан отчет $res->{id}, поставлен в очередь $res->{create_time}";
    } elsif ($res eq '0E0') {
        $description = "Нет отчетов для обработки (пустая очередь)";
    }

    juggler_ok(service_suffix => $QUEUE, description => $description);

    # Замеряем длительность итерации
    my $ela = time() - $iter_start_time;

    local $Yandex::Advmon::GRAPHITE_PREFIX = sub {[qw/direct_one_min db_configurations/, $Settings::CONFIGURATION]};
    monitor_values({
        flow => { PdfReports => { iteration_duration => { $QUEUE => {"shard_$SHARD" => $ela } } } },
    });

    # проверяем, не сильно ли мы разбухли
    my $size = Yandex::ProcInfo::proc_memory();
    if ($size > $MAX_PROCESS_SIZE) {
        my $msize = int($size / 1024 / 1024);
        $log->out("Finish: process [$$] too big: rss=${msize}Mb");
        last;
    } elsif ($ONCE) {
        $log->out('Finish, one iteration');
        last;
    }

    if ($ela < $MIN_ITERATION_DURATION) {
        my $profile = Yandex::Trace::new_profile('ppcPdfReports:sleep');
        sleep($MIN_ITERATION_DURATION - $ela);
    }
}

release_file_lock();
exit 0;


sub makePdf{
    my $vars = shift;

    $log->out("Making PDF-report for ReportOrder: $vars->{id}; Campaigns: $vars->{cids}; From: ".($vars->{date_from}||'-'x 6)."; To: ".( $vars->{date_to}|| '-'x 6));

    prepareData($vars);

    unless ($vars->{no_data}) {
        preparePlots($vars);
        my $source = makeTemplate($doc_text, $vars);
        $source =~ s/\x{20ac}/e/g; # euro
        $source = is_utf8($source) ? encode('cp1251',$source) : $source;
        write_file($ARGV[0]||'template.tex', $source) if $DEBUG;
        
        $vars->{output} = PdfTools::set_pdf_cs_rgba( getPdf( $source ));
            
        cleanUp();
    }
    #   return real date_from and date_to if undef passed;
    #   hash_merge $opt, hash_cut $vars, qw/date_from date_to/;

    $log->out("ReportOrder: $vars->{id} : Status ready");
    return $vars;
}

sub preparePlots{
    my ($vars)=@_;

    require Cairo;
    require CairoGraph;

    $vars->{plot} = {};

    my $date_from   =   [grep {/^\d+$/} split /\D+/, $vars->{date_from}];
    my $date_to     =   [grep {/^\d+$/} split /\D+/, $vars->{date_to}];
    my $group       =   $vars->{group};

    for my $plotcol (qw/shows clicks ctr sum av_sum adepth aconv agoalcost agoalnum/) {

        my $fname = $GRAPHICS_DIR.$plotcol."$$.pdf";

        $vars->{plot}->{$plotcol} = $fname;

        my $atr = {
            date_from   =>  sprintf( "%04d.%02d.%02d", @$date_from),
            date_to     =>  sprintf( "%04d.%02d.%02d", @$date_to),
            # i18n: в других языках порядок слов может отличаться + могут меняться формы других слов
            title       =>  sprintf(iget("Статистика %s за период с %s по %s по %s"),
                {shows=>iget('показов'),
                 clicks=>iget('кликов'),
                 ctr=>'CTR (%)',
                 sum=>iget('расхода всего'),
                 av_sum=>iget('ср. цены клика'),
                 adepth=>iget('глубины просмотра'),
                 aconv=>iget('процента конверсии'),
                 agoalnum=>iget('количества конверсий'),
                 agoalcost=>iget('цены действия'),}->{$plotcol},
                sprintf( "%02d.%02d.%04d", reverse @$date_from),
                sprintf( "%02d.%02d.%04d", reverse @$date_to),
                {day=>iget('дням'), week=>iget('неделям'), month=>iget('месяцам'), season=>iget('сезонам'), year=>iget('годам')}->{$group}),
            names       => [iget('Все площадки')],
            points      => [map { [ $_->{date}, $_->{$plotcol} ] } @{$vars->{stats}{detail}}]
        };

        $atr->{title} .= ' (' . format_currency($vars->{currency}) . ')' if $plotcol eq 'av_sum';
		# Эмпирический коэффициент масштабирования картинок
        my $scale_factor=0.7;

        my $surface = Cairo::PdfSurface->create ($fname, 634*$scale_factor, 490*$scale_factor);
        my $cr = Cairo::Context->create($surface);
        $cr->scale($scale_factor, $scale_factor);


        $CairoGraph::USE_INT_CRDS = 0;
#        $CairoGraph::SHIFT_BAR_BORDER = 0;
        my @months   = iget_noop("январь", "февраль", "март", "апрель", "май", "июнь", "июль", "август", "сентябрь", "октябрь", "ноябрь", "декабрь");
        my @seasons  = iget_noop("зима", "весна", "лето", "осень");
        my @measures = iget_noop("тыс", "млн", "млрд", "трлн", "млнмлрд");

        unless ( CairoGraph::cr_draw_multi_plot($cr, hash_merge($atr,
            {
                x=>30, y=>444, width=>603, height=>416,
                markup => $group,
                group  => $group,
                month_full_name =>  [map {iget($_)} @months],
                season_name     =>  [map {iget($_)} @seasons],
                number_postfix  =>  [undef, map {iget($_)} @measures]
            }
            ))
            ) {
            die "  CAIRO_GRAPH:>>:EORROS:>>:\n",join("\n", CairoGraph::cr_err_list()),"\n";
        }
        $surface->set_fallback_resolution (196, 96);
        $surface->finish();

    }
}

=head2 _get_currency_options


=cut
sub _get_currency_options {
    my ($currency) = @_;

    return (
        # single_curreny не указываем, т.к. контроллируем, чтобы все заказы в отчёте были в одной валюте
        with_nds => ($currency ne 'YND_FIXED') ? 0 : 1,
        with_discount => 1,
        # если в set_report_parameters не передавать currency,
        # то Stat::CustomizedArray::Streamed::stat_stream_parameters подставит туда YND_FIXED
        currency => $currency,
    );
}

=head2 prepareData

    No variable check: make shure you are providing right data!

    $opt = {
        date_from   =~ /YYYY-mm-dd/ or undef
        date_to     =~ /YYYY-mm-dd/ or undef
        cids         =~ /\D*(\d+\D*)+/
        group       =~ /day|week|month|year/
    }

=cut

sub prepareData{
    my ($opt) = @_;

    my @camps       = grep { m/^\d+$/ } split /\D+/, $opt->{cids};

    # Графики тоже имеют тексты, поэтому нужна инициализация языка.
    Yandex::I18n::init_i18n($opt->{lang}, check_file => 1);
    my $postfix = Yandex::I18n::current_lang() eq 'ru' ? '_ru' : '_en';
    my $vars = hash_merge(
        $opt,
        {
            camps => \@camps,  
            stats => {},
            yalogo => "${PDF_REPORT_DIR}yalogo${postfix}.pdf",
            yalogobig => "${PDF_REPORT_DIR}yalogobig${postfix}.pdf",
        });

    my @orders = @{ get_all_sql( PPC(shard => $SHARD), ["SELECT
                                               c.OrderID
                                             , c.cid
                                             , NOT(ISNULL(sc.master_cid)) AS is_subject
                                             , c.name
                                             , IFNULL(c.AgencyID, 0) AS AgencyID
                                             , uo.ya_counters
                                             , IFNULL(c.currency, 'YND_FIXED') AS currency
                                        FROM campaigns c
                                        LEFT JOIN subcampaigns sc ON c.cid = sc.cid
                                        LEFT JOIN users_options uo ON uo.uid = c.uid",
                                        where => {"c.OrderID__ne" => 0, _OR => {
                                            "c.cid" => \@camps, "sc.master_cid" => \@camps
                                        }}]) || []};

    my @orderids = map { $_->{OrderID} } @orders;
    my @cids = map { $_->{cid} } @orders;

    my @agency_ids = uniq grep { $_ && $_ > 0 } map { $_->{AgencyID} } @orders;
    if (@agency_ids && scalar(@agency_ids) == 1) {
        my $agency_id = $agency_ids[0];
        my $ag_chief_rep_uid = rbac_get_chief_rep_of_agency($agency_id);
        my $user_data = get_user_data($ag_chief_rep_uid, [qw(phone email)]);
        my $agency_data = get_client_data($agency_id, [qw(name)]);
        $vars->{agency} = {
            phone => tex_escape($user_data->{phone} // ''),
            email => tex_escape($user_data->{email} // ''),
            name => tex_escape($agency_data->{name} // ''),
        };
    }

    $vars->{ya_counters} = (any { $_->{ya_counters} } @orders) ? 1 : 0;

    my @currencies = uniq map { $_->{currency} } @orders;
    if (scalar(@currencies) > 1) {
        $log->die("Cannot process multiple currencies");
    }
    my $currency = $currencies[0];
    $log->die('No currency') unless $currency;

    my $truncate_word_list = sub {
        my @res=();
        while(1) {
            $_ = shift;
            last unless defined $_;
                if (  length($_) <= $MAX_WORD_LENGTH ) {
                    push @res, $_;
                } else {
                    push @res, substr($_, 0, $MAX_WORD_LENGTH);
                    push @res, "...";
                    last
                }
            }
            return  @res;
        };

    $vars->{campnames} = join ", ", map {
            s/^\s*|\s*$//g;
            $_ = join '\hspace{0em}',
                map { !/^\s+$/ ? tex_escape($_) : $_ }
                $truncate_word_list->(
                    (split /(?<=[^\w])(?=\w)|(?<=\w)(?=[^\w])/,$_));
            s/(\\hspace\{0em\})?\s(\\hspace\{0em\})?/\\ /g;
            "``$_''"
        } map {$_->{name} || iget("Новая кампания")} grep { !($_->{is_subject}) } @orders;

    $vars->{orders} =  join ",\n", nsort map { $_->{cid} } grep { !($_->{is_subject}) } @orders;

    $vars->{domains_cnt} = get_one_field_sql(PPC(shard => $SHARD), ['SELECT COUNT(DISTINCT domain) FROM banners', WHERE => {BannerID__ne => 0, cid => \@cids, domain__is_not_null => 1, domain__ne => ''}]);


    my ($oid_with_stat, $date_from, $date_to) = Stat::OrderStatDay::get_orderids_with_stat_and_dates(\@orderids);

    $vars->{date_from} ||= $date_from;
    $vars->{date_to} ||= $date_to;
    # раньше BS_STAT_START_DATE статистики в Мастере отчетов нет
    my $stat_start_date = str_round_day(Stat::Const::BS_STAT_START_DATE);
    $vars->{date_from} = maxstr($vars->{date_from}, $stat_start_date);
    $vars->{date_to} = maxstr($vars->{date_to}, $stat_start_date);

    $vars->{active_days} = Stat::OrderStatDay::get_orders_days_num(
            $vars->{date_from}, $vars->{date_to},
            \@orderids,
        );

    $vars->{str_date_from}   =  get_date_str($vars->{date_from});
    $vars->{str_date_to}     =  get_date_str($vars->{date_to});

    unless ($vars->{active_days}) {
        $vars->{no_data} = 1; 
        return $vars;
    }

    $vars->{delta_days} = Date::Calc::Delta_Days(map {int $_} map {m/\D*(\d{4})\D*(\d{1,2})\D*(\d{1,2})\D*/} $vars->{date_from}, $vars->{date_to});

    if ( $vars->{delta_days} > 62*30 ) {
        $vars->{plot_group} = 'year' if $vars->{group} !~ /year/;
    } elsif ( $vars->{delta_days} > 62*7 ) {
        $vars->{plot_group} = 'month' if $vars->{group} !~ /year|month/;
    } elsif ( $vars->{delta_days} > 62 ) {
        $vars->{plot_group} = 'week' if $vars->{group} !~ /year|month|week/;
    } else {
        $vars->{plot_group} = 'day' if $vars->{group} !~ /year|month|week|day/;
    }
    $vars->{group} = $vars->{plot_group} || $vars->{group};

    my @suffixes = ('', '_0', '_1');
    my @fields_to_sum = qw(shows clicks sum asesnum aseslen agoalnum gsum);

    my %orders_stream_refresh_ts;
    my $dates = {};
    my $sum = {};
    my $sum_no_bm = {};
    my $geos = {};
    my $pages = {};
    my %currency_options = _get_currency_options($currency);
    my $translocal_params = { ClientID => get_clientid(uid => $opt->{uid}) };
    my $total_order_cnt = scalar(@orders);
    my $cur_order_num = 0;
    for my $o (@orders) {
        my $order_id = $o->{OrderID};
        $cur_order_num++;
        $log->out("Processing OrderID $order_id [$cur_order_num / $total_order_cnt]");
        next unless $oid_with_stat->{$order_id};

        my $dbstat = Stat::CustomizedArray->new(OrderID => $order_id);

        # Суммарная статистика по заказу (минус статистика по BroadMatch фразам)
        $dbstat->set_report_parameters(
                oid => $order_id
                , start_date => $vars->{date_from}
                , end_date => $vars->{date_to}
                , group_by => ['date', 'contexttype']
                , date_aggregation_by => $vars->{group}
                , options => \%currency_options
                , translocal_params => $translocal_params
            );
        my $st_phrase = $dbstat->generate_with_metrika;
        merge_stat_stream_ts(\%orders_stream_refresh_ts, $order_id, $st_phrase);
        for my $r (@{$st_phrase->{data_array}}) {
            $dates->{$r->{stat_date}} ||= hash_cut($r, qw/stat_date date/);

            for my $suf (@suffixes) {
                for my $k (@fields_to_sum) {
                    $dates->{$r->{stat_date}}{$k.$suf} += $r->{$k.$suf}||0;
                    $sum->{$k.$suf} += ($r->{$k.$suf} || 0);
=head3 about CONTEXT_TYPE_BROADMATCH

    Логика работы с рекламным шардом и расширенными синонимами поисковыми запросами
    реализована в DBStat и Stat::Stream - они представляют такую статистику как обычные ДРФ
    Поэтому здесь логику не дублируем и проверяем только ContextType = 3 (но не 4 или 5)

=cut
                    if ($r->{ContextType} != $Stat::Const::CONTEXT_TYPE_BROADMATCH) {
                        $sum_no_bm->{$k.$suf} += ($r->{$k.$suf} || 0);
                    }
                }
            }
        }

        # Статистика по регионам
        my $st_geo = $dbstat->get_stat_customized( $order_id, $vars->{date_from}, $vars->{date_to}, ['geo' ], undef, undef, %currency_options);
        merge_stat_stream_ts(\%orders_stream_refresh_ts, $order_id, $st_geo);
        for my $r (@{$st_geo->{data}}) {
            $geos->{$r->{region}} ||= hash_cut($r, qw/region region_name name full_name region_not_exactly_defined/);

            for my $suf (@suffixes) {
                for my $k (@fields_to_sum) {
                    $geos->{$r->{region}}{$k.$suf} += $r->{$k.$suf}||0;
                }
            }
        }

        # Статистика по площадкам
        my $st_pages = $dbstat->get_stat_customized( $order_id, $vars->{date_from}, $vars->{date_to}, ['page' ], undef, undef, %currency_options);
        merge_stat_stream_ts(\%orders_stream_refresh_ts, $order_id, $st_pages);
        for my $r (@{$st_pages->{data}}) {
            $pages->{$r->{page_group}||''} ||= hash_cut($r, qw/page_group TargetType name page_sorting sorting/);

            for my $suf (@suffixes) {
                for my $k (@fields_to_sum) {
                    $pages->{$r->{page_group}||''}{$k.$suf} += $r->{$k.$suf}||0;
                }
            }
        }
    }

    for my $suf (@suffixes) {
        map { Stat::Tools::calc_avg($_, $suf) } values %$dates;
        Stat::Tools::calc_avg($sum, $suf);
        Stat::Tools::calc_avg($sum_no_bm, $suf);
    }

$log->out({summary_stats => [map {hash_kgrep {m/shows/} $_}  $sum, $sum_no_bm]});

    $vars->{stats}{detail}=[map {
            {
                date    => $_->{stat_date},
                date_pdf  => Stat::Tools::format_date_pdf($_->{stat_date}, $opt->{date_group}),
                shows   => $_->{shows}||0,
                clicks  => $_->{clicks}||0,
                ctr     => ($_->{ctr} && $_->{ctr} > 100) ? '-' : sprintf( "%.02f", $_->{ctr}||0),
                sum     => sprintf( "%.02f", $_->{sum}||0),
                av_sum  => sprintf( "%.02f", $_->{av_sum}||0),
                adepth  => sprintf( "%.02f", $_->{adepth}||0),
                aconv   => sprintf( "%.02f", $_->{aconv}||0),
                agoalnum   => $_->{agoalnum}||0,
                agoalcost   => sprintf( "%.02f", $_->{agoalcost}||0),
            }
        } sort { $a->{stat_date} cmp $b->{stat_date} } values %{$dates}];

    $vars->{stats}{sum} = {
                shows   => $sum->{shows}||0,
                clicks  => $sum->{clicks}||0,
                ctr     => ($sum->{ctr} && $sum->{ctr} > 100) ? '-' : sprintf( "%.02f", $sum->{ctr}||0),
                sum     => sprintf( "%.02f", $sum->{sum}||0),
                av_sum  => sprintf( "%.02f", $sum->{av_sum}||0),
                agoalnum=> $sum->{agoalnum}||0,
                asesnum => $sum->{asesnum}||0,
            };

    $vars->{stats}{sum_no_bm} = $sum_no_bm;

    my $nice_round = sub {
        ## no critic (Freenode::DollarAB)
        my $a = shift;
        return sprintf( "%.02f", $a) if ( abs $a < 3 );
        return sprintf( "%.02f", 0.5*(floor ( 2*$a + 0.5)) ) if ( abs $a < 5 );
        return sprintf( "%d", floor ( $a + 0.5) );
    };

    $vars->{stats}{avg} = {
                shows   => $nice_round->( $vars->{stats}{sum}{shows} / ($vars->{active_days}||1e6) ),
                clicks  => $nice_round->( $vars->{stats}{sum}{clicks} / ($vars->{active_days}||1e6) ),
                sum     => sprintf( "%.02f", $vars->{stats}{sum}{sum} / ($vars->{active_days}||1e6) ),
            };

    #   Order by clicks, shows
    my @sort_geos = sort {
            ($b->{clicks}||0) <=> ($a->{clicks}||0) ||
            ($b->{shows}||0) <=> ($a->{shows}||0)
        } values %$geos;

    my $others  = { name => iget('прочие'), name_sfx => '$^1$...' };

    # Don't show high level regions while their subregions are present
    for my $chunk (grep { $_->{region_not_exactly_detected} } @sort_geos) {
        for my $key (@fields_to_sum){
            $others->{$key} += $chunk->{$key};
        }
        $chunk->{ignore}=1;
    }
    @sort_geos = grep {!$_->{ignore}} @sort_geos;

    #   Show just 45 most clickable regions or 15 regions at least
    while (scalar @sort_geos > 45 or scalar @sort_geos >= 15 and not $sort_geos[-1]{clicks}) {
        my $chunk = pop @sort_geos;
        for my $key (@fields_to_sum){
            $others->{$key} += $chunk->{$key};
        }
    }
    push @sort_geos, $others if $others->{shows};


    for my $suf (@suffixes) {
        map { Stat::Tools::calc_avg($_, $suf) } @sort_geos;
    }
    my $field = get_geo_name_field($vars->{lang});

    $vars->{stats}{geo}=[map {
            my $geodname = (defined $_->{region} && exists $geo_regions::GEOREG{$_->{region}}) ? $geo_regions::GEOREG{$_->{region}}->{$field} : $_->{name};
            $geodname =~ s/\s*\(.*\)$//;
            # По какой-то причине, в GEOREG почему-то некоторые города имеют подчерк, а это спецсимвол для latex. Принудительно убираем.
            $geodname =~ s/\_/ /;
            {
                geoname => $geodname,
                name_sfx=> $_->{name_sfx} || ($_->{region_not_exactly_detected} ? $_->{original_region_name} : ''),
                shows   => $_->{shows}||0,
                clicks  => $_->{clicks}||0,
                ctr     => ($_->{ctr} && $_->{ctr} > 100) ? '-' : sprintf( "%.02f", $_->{ctr}||0),
                sum     => sprintf( "%.02f", $_->{sum}||0),
                av_sum  => sprintf( "%.02f", $_->{av_sum}||0),
                adepth  => sprintf( "%.02f", $_->{adepth}||0),
                agoalnum => $_->{agoalnum}||0,
                aconv   => sprintf( "%.02f", $_->{aconv}||0),
                agoalcost   => sprintf( "%.02f", $_->{agoalcost}||0)
            }
        } @sort_geos];

    #   Order by clicks, shows
    my @sort_pages = sort {
            ($b->{clicks}||0) <=> ($a->{clicks}||0) ||
            ($b->{shows}||0) <=> ($a->{shows}||0)
        } values %$pages;

    #   Show just 25 most clickable pages or 15 pages at least
    $others  = { name => iget('прочие'), name_sfx => '$^1$...'};
    while (scalar @sort_pages > 45 or scalar @sort_pages >= 15 and not $sort_pages[-1]{clicks}) {
        my $chunk = pop @sort_pages;
        for my $key (@fields_to_sum){
            $others->{$key} += $chunk->{$key};
        }
    }
    push @sort_pages, $others if $others->{shows};

    for my $suf (@suffixes) {
        map { Stat::Tools::calc_avg($_, $suf) } @sort_pages;
    }

    $vars->{stats}{page} = [];
    for my $page (@sort_pages) {
        my $name = $page->{name};

        if (defined($name) && $name ne '') {
            $name = iget($name);
        } else {
            send_alert(sprintf("WARNING: Site name is undef!\nSite statistics for campaigns %s for PDF_report from %s to %s", $opt->{cids}, $vars->{str_date_from}, $vars->{str_date_to}),
                       "PDF report: Site name is undef");
            $name = '';
        }

        push @{$vars->{stats}{page}}, {
            page    => tex_escape($name),
            name_sfx=> $page->{name_sfx}||'',
            shows   => $page->{shows}||0,
            clicks  => $page->{clicks}||0,
            ctr     => ($page->{ctr} && $page->{ctr} > 100) ? '-' : sprintf( "%.02f", $page->{ctr}||0),
            sum     => sprintf( "%.02f", $page->{sum}||0),
            av_sum  => sprintf( "%.02f", $page->{av_sum}||0),
            adepth  => sprintf( "%.02f", $page->{adepth}||0),
            agoalnum => $page->{agoalnum}||0,
            aconv   => sprintf( "%.02f", $page->{aconv}||0),
            agoalcost   => sprintf( "%.02f", $page->{agoalcost}||0)
        };
    }

    $vars->{actual_time} = min grep { defined } values %orders_stream_refresh_ts;

    my @atime = Date::Calc::Localtime( grep {$_} ($vars->{actual_time}) );

    $vars->{actual_human_time} = sprintf "%02d %s %04d (%02d:%02d)", $atime[2], month_name($atime[1]), $atime[0], $atime[3], $atime[4];

    if ( Date::Calc::Delta_Days(
        ($vars->{date_to} =~ m/\D*(\d{4})\D*(\d{2})\D*(\d{2})\D*/),
        Date::Calc::Today) == 0 ) {
        $vars->{is_incomplete} = 1;
    }
    $vars->{is_eng} = $vars->{lang} eq 'en';

    $vars->{currency} = $currency;

    return $vars;
}

sub makeTemplate{
    my ($input, $vars)=@_;
    Yandex::I18n::init_i18n($vars->{lang}, check_file => 1);
    # some useful options (see below for full list)
    my $config = {
        INTERPOLATE  => 1,               # expand "$var" in plain text
        POST_CHOMP   => 1,               # cleanup whitespace
#         PRE_PROCESS  => ’header’,        # prefix each template
        EVAL_PERL    => 1,               # evaluate Perl code blocks
        START_TAG    => quotemeta('{$$'),
        END_TAG      => quotemeta('$$}'),
        PRE_DEFINE   => {
            floor                       => \&floor,
            ceil                        => \&ceil,
            word_pl                     => \&word_pl,
            iget                        => \&iget,

            # функции для работы с валютами. описание см. в Currencies и Currency::Texts
            get_currency_constant      => \&get_currency_constant,
            get_currency_text          => \&get_currency_text,
            format_sum_of_money        => \&format_sum_of_money,
            # conv_unit_explanation($pay_currency)
            conv_unit_explanation      => sub {return conv_unit_explanation('YND_FIXED', @_)},
            format_const               => \&format_const,
            format_currency            => \&format_currency,

            min_table_chunk_length      => 4,   # minimal rows count to be printed on table-tail
        },
    };

    delete $Template::Stash::LIST_OPS->{hash_key};
    $Template::Stash::LIST_OPS->{ hash_key } = sub {
        my $list = shift;
        my $key = shift;
        return [map { (ref $_ eq 'HASH') ? $_->{$key} : undef } @$list];
    };

    # create Template object
    my $template = Template->new($config);

    my $output='';
    # process input template, substituting variables
    $template->process(\$input, $vars, \$output)
        || die $template->error();

    return $output;
}

sub getPdf{
    my ($doc_text) = @_;

    #   Dumn LaTeX::Driver is designed to work with nonunicode symbols
    #   Let's make shure it is proveided with non_utf8 input
    $doc_text = is_utf8($doc_text) ? encode('cp1251',$doc_text) : $doc_text;

    my $output = '';

    my $drv = LaTeX::Driver->new( source  => \$doc_text,
                            output  => \$output,
                            format  => 'pdf',
                            texinputs => $PDF_REPORT_DIR,
                        ) or die "ERROR 0:>>:$!\n";

    $drv->run or die "ERROR 1:>>:$!\n";
    $drv->stats or die "ERROR 2:>>:$!\n";

    return $output;
}

sub cleanUp{
    for my $plotcol (qw/shows clicks ctr sum av_sum adepth agoalnum aconv agoalcost/) {
		unlink "$PDF_REPORT_DIR$plotcol$$.pdf";
	}
}

sub get_date_str{
    my $date = shift;
    return unless $date and $date =~ m/\D*(\d+)\D*(\d+)\D*(\d+)\D*/;

    return sprintf "%d %s %d", $3, month_name($2), $1;
}

sub month_name{
      my @months = iget_noop("января", "февраля", "марта", "апреля", "мая", "июня", "июля", "августа", "сентября", "октября", "ноября", "декабря");
      return iget($months[int $_[0]-1]);
}

=head2 word_pl()

    1     раз   яблоко
    2..4  раза  яблока
    5..20 раза  яблок
    21    раз   яблоко

=cut
sub word_pl{
    my $cnt = shift;

    if ( $cnt != floor $cnt ) {
        return $_[1];
    }
    $cnt = sprintf '%d', $cnt;
    if ( $cnt =~ m/(?:1\d|[05-9])$/) {
        return $_[2]||$_[0];
    }
    if ( $cnt =~ m/(?<=1)[2..4]$/) {
        return $_[1];
    }
    return $_[0];
}

=head2 merge_stat_stream_ts
    
    Собирает в хеш {OrderID => <время обновления статистики>} время обновления стримовой статистики

=cut

sub merge_stat_stream_ts {
    my ($orders_stream_refresh_ts, $OrderID, $st) = @_;
    if (defined $st->{stat_stream_ts} && 
        (!defined $orders_stream_refresh_ts->{$OrderID} || $orders_stream_refresh_ts->{$OrderID} > $st->{stat_stream_ts})) {
        $orders_stream_refresh_ts->{$OrderID} = $st->{stat_stream_ts};
    }
}
