#!/usr/bin/perl

use my_inc "..";

=head1 METADATA

<crontab>
    time: */50 3,4,5 * * *
    sharded: 1
    <switchman>
        group: scripts-other
        <leases>
            mem: 265
        </leases>
    </switchman>
    package: scripts-switchman
</crontab>
<juggler>
    host:   checks_auto.direct.yandex.ru
    sharded: 1
    ttl: 2d2h
    tag: direct_group_internal_systems
</juggler>

=cut

=head1 NAME

    ppcPrepareCurrencyTeaserClients.pl

=head1 DESCRIPTION

    Собирает список пользователей, которым можно показывать тизер конвертации в валюту.

=head1 RUNNING

    Скрипт собирает список пользователей, потенциально удовлетворяющих условиям отображения тизера:
        - валюта одна из биллинга, т.е. есть плательщик в биллинге
        - клиент не входит в список проблемных из https://jira.yandex-team.ru/browse/BALANCE-13536
        - нет проблемы со смешанным НДС, которая возникает если есть остатки в разных валютах

    Можно запускать по конкретным клиентам:
        --login — логин клиента или одного из его представителей, по которому надо получить данные

    ./protected/ppcPrepareCurrencyTeaserClients.pl --shard-id 1
    ./protected/ppcPrepareCurrencyTeaserClients.pl --shard-id=2 --login holodilnikru

=cut

use warnings;
use strict;
use utf8;

use Settings;
use ScriptHelper sharded => 1, 'Yandex::Log' => 'messages';

use Yandex::DBTools;
use Yandex::ListUtils;
use Yandex::ScalarUtils;
use Yandex::Balance qw(balance_get_direct_balance balance_get_all_equal_clients);

use RBAC2::Extended;
use RBACElementary;
use RBACDirect;
use Campaign::Types ();
use Client;
use Primitives;
use PrimitivesIds;
use Client::ConvertToRealMoney ();
use Client::CurrencyTeaserData;
use Property;

use List::MoreUtils qw(any uniq part);
use Parallel::ForkManager;

my $MODIFY_TEASER_SHOW_PERCENT = Property->new($Client::CurrencyTeaserData::MODIFY_TEASER_SHOW_PERCENT_PROPERTY_NAME)->get();
$MODIFY_TEASER_SHOW_PERCENT //= $Client::CurrencyTeaserData::CURRENCY_CONVERT_TEASER_DEFAULT_PERCENT;

# во сколько параллельных потоков проставлять тизер
my $PARALLEL_LEVEL = 10;

my $OPERATOR_UID = 1;

my $logins = [];
extract_script_params(
    'login=s@' => $logins,
);

$log->out('START');
$log->out("MODIFY teaser show percent is $MODIFY_TEASER_SHOW_PERCENT");

my @additional_conds;
if ($logins && @$logins) {
    $log->out('Working only on logins:', $logins);
    my @only_client_ids = map { get_clientid(uid => get_uid_by_login2($_)) } @$logins;
    @additional_conds = ('AND', {'u.ClientID' => \@only_client_ids});
}

$log->out('Fetching client aliases from Balance');
my $balance_aliases = balance_get_all_equal_clients(timeout => 1800);
$log->out('Got ' . scalar(@$balance_aliases) . ' alias pairs');
my @alias_clientids = uniq map { ($_->{CLIENT_ID}, $_->{CLASS_ID}) } @$balance_aliases;
undef $balance_aliases;

my @skip_clientids = uniq(@Client::ConvertToRealMoney::KNOWN_BAD_CLIENTIDS, @alias_clientids);
undef @alias_clientids;
$log->out('Total of ' . scalar(@skip_clientids) . ' excluded ClientIDs');

$log->out('Fetching clients to probably enable currency convert teaser');
my $client_ids = get_one_column_sql(PPC(shard => $SHARD), ['
        SELECT u.ClientID
        FROM users u
        LEFT JOIN clients cl ON u.ClientID = cl.ClientID
        LEFT JOIN clients_to_force_multicurrency_teaser fmt ON u.ClientID = fmt.ClientID
        INNER JOIN campaigns c ON u.uid = c.uid
        WHERE
            IFNULL(cl.work_currency, "YND_FIXED") = "YND_FIXED"
            AND fmt.ClientID IS NULL
            AND c.statusEmpty = "No"
            AND', {
                'u.ClientID__not_in__int' => \@skip_clientids,
                'c.type__not_in' => Campaign::Types::get_camp_kind_types('non_currency_convert'),
            },
            @additional_conds, '
            GROUP BY u.ClientID
            HAVING COUNT(DISTINCT IFNULL(c.AgencyID,0)) <= 1
            ORDER BY u.ClientID
        ']) || [];
undef @skip_clientids;
if (@$client_ids) {
    my $client_ids_cnt = scalar @$client_ids;
    $log->out("Got $client_ids_cnt ClientID's to probably enable currency convert teaser");

    my %is_agid_bad = map { $_ => undef } @Client::ConvertToRealMoney::KNOWN_BAD_AGENCY_CLIENTIDS;

    my $pm = Parallel::ForkManager->new($PARALLEL_LEVEL);

    for my $client_ids_chunk (chunks $client_ids, 1_000) {
        $pm->start and next;
        Yandex::Trace::restart(\$ScriptHelper::trace, tags => "child");
        my $child_msg_prefix = "[child_$$]";
        my $msg_prefix_guard = $log->msg_prefix_guard($child_msg_prefix);
        local $SIG{__DIE__} = sub {
            if (!$^S) {
                $log->out('Child has died');
                $log->die(@_);
            }
        };

        $log->out('Reinitializing RBAC object');
        my $rbac = eval { RBAC2::Extended->get_singleton($OPERATOR_UID) }
            or $log->die("Can't connect to RBAC: $@");

        $log->out("Processing ClientID's chunk:", $client_ids_chunk);

        $log->out('Fetching agencies for clients');
        my $clid2agid = Primitives::mass_get_client_first_agency($client_ids_chunk);
        my $clid2rbac_agencies = rbac_mass_get_agencies_of_clients($rbac, $client_ids_chunk);

        my $skipped_clients;
        ($client_ids_chunk, $skipped_clients) = part {
            my $clid = $_;
            my $first_agid = $clid2agid->{$clid};
            my $is_first_agency_bad = $first_agid && exists $is_agid_bad{ $first_agid };

            my $rbac_agencies = $clid2rbac_agencies->{$clid};
            my $is_any_rbac_agency_bad = $rbac_agencies && any { $is_agid_bad{ $_ } } @$rbac_agencies;

            return ($is_first_agency_bad || $is_any_rbac_agency_bad) ? 1 : 0;
        } @$client_ids_chunk;
        if ($skipped_clients && @$skipped_clients) {
            $log->out('Skipping clients because of known bad agency:', $skipped_clients);
        }
        if (!$client_ids_chunk) {
            $log->out('No more clients to process, skipping chunk');
            $pm->finish;
        }

        my @agency_clientids = uniq grep { !exists $is_agid_bad{$_} } values %$clid2agid;

        if (!$client_ids_chunk) {
            $log->out('No more clients to process, skipping chunk');
            $pm->finish;
        }

        $log->out('Fetching NDS for chunk');
        my %get_NDS_opts = (fetch_missing_from_balance => 1, rbac => $rbac, fetch_for_ynd_fixed_too => 1);
        my @clientids_to_fetch = (@$client_ids_chunk, @agency_clientids);
        my $clients_nds = eval { mass_get_client_NDS(\@clientids_to_fetch, %get_NDS_opts) };
        my %nds_errors;
        # если по всей пачке не выходит забрать, то забираем отдельно по каждому клиенту, чтобы выколоть проблемного, но не потерять остальных
        if (!$clients_nds) {
            $log->out('Error fetching NDS for chunk:', $@);
            $log->out('Fetching NDS for clients alone');
            for my $client_id (@clientids_to_fetch) {
                $log->out("Fetching NDS for ClientID $client_id alone");
                my $client_nds_data = eval { mass_get_client_NDS([$client_id], %get_NDS_opts) };
                if ($client_nds_data && %$client_nds_data) {
                    $log->out("Got NDS for ClientID $client_id alone");
                    $clients_nds->{$client_id} = $client_nds_data->{$client_id};
                } else {
                    $log->out("Got no NDS for ClientID $client_id:", $@);
                    $nds_errors{$client_id} = 1;
                }
            }
        }

        $log->out('Fetching currencies for chunk');
        my $clients_currencies = mass_get_client_currencies($client_ids_chunk);

        $log->out('Fetching chief uids for chunk');
        my $client_chief_uids = rbac_get_chief_reps_of_clients($client_ids_chunk);

        $log->out('Fetching agencies of clients');
        my @all_client_chief_uids = values %$client_chief_uids;
        my $client_chief_uid2agency_uids = rbac_mass_get_agencies_of_clients($rbac, \@all_client_chief_uids);

        $log->out('Fetching current country/currency from DB for fast Russia+RUB skip');
        my $russia_rub_clients = get_hash_sql(PPC(shard => $SHARD), ['
            SELECT cfcc.ClientID
            FROM client_firm_country_currency cfcc
        ',  WHERE => {
                'cfcc.ClientID' => $client_ids_chunk,
            }, '
            GROUP BY cfcc.ClientID
            HAVING COUNT(*) = 1 AND MAX(cfcc.currency) = "RUB"
        ']);

        my @teaser_insert_data;
        CLIENT: for my $client_id (@$client_ids_chunk) {
            my $client_msg_prefix_guard = $log->msg_prefix_guard("$child_msg_prefix\t[ClientID $client_id]");

            my $client_chief_uid = $client_chief_uids->{$client_id};
            if (!$client_chief_uid) {
                $log->out("Skipping, no chief uid found for ClientID $client_id");
                next;
            }

            my $client_nds = $clients_nds->{$client_id};
            if ($nds_errors{$client_id}) {
                $log->out("No NDS fetched before, skipping");
                next;
            }

            my $agency_id = $clid2agid->{$client_id};
            if ($agency_id && !defined $clients_nds->{$agency_id}) {
                $log->out("Skipping, no NDS known for agency with ClientID $agency_id");
                next CLIENT;
            }

            my $can_modify_convert = $client_id % 100 < $MODIFY_TEASER_SHOW_PERCENT;
            $log->out( ($can_modify_convert) ? "Client CAN be converted without copying" : "Client CANNOT be converted without copying" );

            if (!$can_modify_convert && defined $client_nds && $client_nds == 100*$Settings::NDS_RU_DEPRECATED && exists $russia_rub_clients->{$client_id}) {
                $log->out("Skipping, can be converted without copying [fast check]");
                next;
            }

            $log->out("Fetching country/currency data");
            my $country_currency_data;
            my $country_currency_success = eval {
                Client::CurrencyTeaserData::fetch_client_multicurrency_teaser_data($client_id, $agency_id, firm_country_currency_data_ref => \$country_currency_data, balance_timeout => 310);
                return 1;
            };
            if (!$country_currency_success) {
                $log->out("Skipping, error refreshing country/currency data: ".str($@));
                next;
            }

            my $balance_modify_convert_allowed = Client::CurrencyTeaserData::can_be_modify_converted($country_currency_data);
            undef $country_currency_data;
            $log->out('Client is ' . ( $balance_modify_convert_allowed ? '' : 'NOT ' ) . 'allowed to be modify converted by Balance');

            $log->out('Checking whether we can convert client');
            my @country_currencies;
            my $can_convert_error = Client::can_convert_to_real_money(
                ClientID => $client_id,
                NDS => $clients_nds->{$client_id},
                client_currencies => $clients_currencies->{$client_id},
                client_chief_uid => $client_chief_uid,
                country_currency_ref => \@country_currencies,
                ignore_checkbox => 1,
                ignore_nds_absence => 1,
                agency_id => $agency_id,
            );
            if ($can_convert_error) {
                $log->out("Skipping, can_convert_to_real_money returned $can_convert_error");
                next;
            }

            if (!@country_currencies) {
                $log->out("Skipping, no available country/currency pairs");
                next;
            }

            if (!$can_modify_convert && $balance_modify_convert_allowed && $client_nds && $client_nds == 100*$Settings::NDS_RU_DEPRECATED && scalar(@country_currencies) == 1 && any { $_->{currency} eq 'RUB' } @country_currencies) {
                $log->out("Skipping, can be converted without copying");
                next;
            }

            my $agency_uids = $client_chief_uid2agency_uids->{$client_chief_uid};
            if ($agency_uids && @$agency_uids) {
                my @subclient_agency_clientids = @{ get_clientids(uid => $agency_uids) };
                if (scalar(@subclient_agency_clientids) > 1) {
                    $log->out('Skipping, subclient has multiple agencies in RBAC');
                    next;
                }

                my $relations = get_agency_clients_relations_data([$client_id]);
                if ($relations && scalar(@$relations) > 1) {
                    $log->out('Skipping, have relations with multiple agencies');
                    next;
                }
            }


            # Balance.GetDirectBalance падает на клиентах со смешенным НДС
            # от валюты эта способность не зависит, поэтому используем первую доступную
            my $first_currency = $country_currencies[0]->{currency};
            $log->out("Checking for mixed nds [$first_currency]");
            my $get_direct_budget_success = eval {
                balance_get_direct_balance(ClientID => $client_id, CurrencyCode => $first_currency);
                return 1;
            };
            if (!$get_direct_budget_success) {
                $log->out("Skipping, error fetching campaigns balance in $first_currency: ".str($@));
                next;
            }

            # если не проскипали по условиям выше, включаем тизер
            $log->out('Adding client to enable teaser');
            push @teaser_insert_data, [ $client_id, ($balance_modify_convert_allowed ? 1 : 0) ];
        }

        if (@teaser_insert_data) {
            $log->out('Enabling teaser for clients:', \@teaser_insert_data);
            do_mass_insert_sql(PPC(shard => $SHARD), 'INSERT IGNORE INTO clients_to_force_multicurrency_teaser (ClientID, modify_convert_allowed) VALUES %s', \@teaser_insert_data);
        } else {
            $log->out('No ClientIDs in chunk found to enable teaser');
        }

        $log->out('Exitting child');
        $pm->finish;
    }

    $log->out('Waiting for children');
    $pm->wait_all_children;
} else {
        $log->out('No ClientIDs found to probably enable currency convert teaser');
}

juggler_ok();

$log->out('FINISH');
