#!/usr/bin/perl

=encoding utf8

=head1 NAME

    ppcProcessAutoPayments.pl

=head1 SYNOPSIS

    Параметры запуска:
        --help          - вывести справку
        --shard-id=N    - какой шард базы данных обрабатывать
        --wallet-cid XXXX - обрабатывать только конкретные кампании-кошельки (можно указать несколько раз)
        --force-create-camp-balance - отправлять кампанию-кошелек в баланс перед каждой оплатой (для автотестов)
        --dont-wait-notification - не ждать нотификацию после успешной транзакции (для тестов)

=head1 DESCRIPTION

    Инициирует автоплатежи по кошелькам где выполняется условие для платежа, а также отслеживает статусы созданных платежей

=head1 METADATA

<crontab>
    sharded: 1
    time: */10 * * * *
    <switchman>
        group: scripts-other
    </switchman>
    package: scripts-switchman
</crontab>

<juggler>
    host:   checks_auto.direct.yandex.ru
    sharded:    1
    ttl:        60m
    tag: direct_group_internal_systems
    <notification>
        template: on_status_change
        status: OK
        status: CRIT
        method: telegram
        login: DISMonitoring
    </notification>
</juggler>

=cut

use Direct::Modern;

use Yandex::Advmon qw/monitor_values/;
use Yandex::DBTools;

use my_inc "..";

use ScriptHelper sharded => 1, 'Yandex::Log' => 'messages';
use Settings;
use Direct::Wallets::Payment;
use Campaign::Types;

my (@ONLY_CIDS, $force_create_camp_balance, $dont_wait_notification);
extract_script_params(
    'wallet-cid=i@' => \@ONLY_CIDS,
    'force-create-camp-balance' => \$force_create_camp_balance,
    'dont-wait-notification' => \$dont_wait_notification,
);

my $short_script_name = get_script_name(short => 1);

=head1 VARS

=head2 $SQL_LOCK_TIMEOUT

    время ожидания (в секундах) при получении лока в MySQL

=cut

my $SQL_LOCK_TIMEOUT = 5;

local $Yandex::Advmon::GRAPHITE_PREFIX = sub {[qw/direct_one_min db_configurations/, $Settings::CONFIGURATION]};

$log->out('START');

my $sql_lock_guard = eval { sql_lock_guard(PPC(shard => $SHARD), $short_script_name, $SQL_LOCK_TIMEOUT) };
if (!$sql_lock_guard) {
    $log->die(sprintf(q!Can't obtain MySQL lock '%s' with timeout '%d': %s!, $short_script_name, $SQL_LOCK_TIMEOUT, $@));
}

$log->out("Checking statuses for incompleted transactions");
my ($processed_trans_qty, $processed_trans_errors_qty) = (0, 0);
for my $t ( @{ Direct::Wallets::Payment::get_incompleted_transactions($SHARD, \@ONLY_CIDS) }) {
    my $payment_id_prefix;
    if ($t->{trust_payment_id}) {
        $payment_id_prefix = "trust_payment_id=$t->{trust_payment_id}";
    } elsif ($t->{request_id}) {
        $payment_id_prefix = "request_id=$t->{request_id}";
    }
    my $msg_prefix_quard = $log->msg_prefix_guard("[wallet_cid=$t->{wallet_cid},$payment_id_prefix]");
    eval { Direct::Wallets::Payment::get_and_process_transaction_status($t, log => $log); };
    my $error = $@;
    $log->out($error) if $error;
    $processed_trans_errors_qty++ if $error;
    $processed_trans_qty++;
}
$log->out("Total transactions qty: checked status - $processed_trans_qty, checking status errors - $processed_trans_errors_qty");

$log->out("Processing wallets with enabled autopayment");
my ($processed_camps_qty, $processed_camps_errors_qty, $payed_camps_qty) = (0, 0, 0);
for my $cid (@{ Direct::Wallets::Payment::get_wallet_autopay_cids($SHARD, \@ONLY_CIDS) }) {
    my $msg_prefix_quard = $log->msg_prefix_guard("[wallet_cid=$cid]");

    my $num_camps_to_pay = get_one_field_sql(PPC(cid => $cid), [
            'SELECT count(c.cid)
            FROM campaigns wc
            JOIN campaigns c USING(uid)',
            WHERE => {
                'wc.cid' => $cid,
                'c.type' => get_camp_kind_types('under_wallet'),
                'c.statusNoPay__ne' => 'Yes',
                'c.archived' => 'No',
                'c.statusModerate' => 'Yes',
            },
        ]);

    if (!$num_camps_to_pay) {
        $log->out("No attached campaigns to pay, skipping");
        next;
    }

    my ($is_payed, $error) = eval { Direct::Wallets::Payment::check_and_process_wallet_autopay(
                                                        $cid, log => $log, 
                                                        force_create_camp_balance => $force_create_camp_balance,
                                                        dont_wait_notification => $dont_wait_notification, ); };

    if ($@) {
        $error = $@;
        $log->out($error);
    }
    
    $processed_camps_qty ++;
    $payed_camps_qty += $is_payed ? 1 : 0;
    $processed_camps_errors_qty ++ if $error;
}
$log->out("Total wallets qty: processed - $processed_camps_qty, payed - $payed_camps_qty, process errors - $processed_camps_errors_qty");

monitor_values({ flow => { autopay => { "processed_camps_qty.shard_$SHARD" => $processed_camps_qty, 
                                        "processed_camps_errors_qty.shard_$SHARD" => $processed_camps_errors_qty, 
                                        "payed_camps_qty.shard_$SHARD"     => $payed_camps_qty,
                                        "checked_trans_qty.shard_$SHARD"   => $processed_trans_qty,
                                        "checked_trans_errors_qty.shard_$SHARD"   => $processed_trans_errors_qty,} } });
juggler_ok();

$log->out("FINISH");
