#!/usr/bin/perl

use my_inc "..";



=head1 METADATA

<crontab>
    time: */5 * * * *
    sharded: 1
    params: --uniq 1
    <switchman>
        group: scripts-other
        <leases>
            mem: 170
        </leases>
    </switchman>
    package: scripts-switchman
</crontab>
<crontab>
    time: */5 * * * *
    sharded: 1
    params: --uniq 2
    <switchman>
        group: scripts-other
        <leases>
            mem: 170
        </leases>
    </switchman>
    package: scripts-switchman
</crontab>
<crontab>
    time: */5 * * * *
    sharded: 1
    params: --uniq 3
    <switchman>
        group: scripts-other
        <leases>
            mem: 170
        </leases>
    </switchman>
    package: scripts-switchman
</crontab>
<juggler>
    host:   checks_auto.direct.yandex.ru
    raw_events: scripts.ppcProcessImageQueue.working.uniq_$uniq.shard_$shard
    sharded: 1
    vars:   uniq=1,2,3
    ttl: 25m
    tag: direct_group_internal_systems
</juggler>

<crontab>
    time: */10 * * * *
    params: --uniq 1
    <switchman>
        group: scripts-test
    </switchman>
    package: conf-test-scripts
    sharded: 1
</crontab>

=cut

=head1 DESCRIPTION

    Скрипт для обработки очереди картинок на скачивание

=cut

use Direct::Modern;

use Settings;

use Time::HiRes qw /time gettimeofday tv_interval/;
use List::Util qw/min max/;
use List::MoreUtils qw/all part/;

use Yandex::HTTP;
use Yandex::DBQueue;
use Yandex::DBTools;
use Yandex::I18n qw/iget/;
use Yandex::Advmon;
use Yandex::ListUtils qw/chunks/;
use Yandex::HashUtils qw/hash_copy/;
use Yandex::Log;
use Yandex::TVM2;
use Yandex::URL;
use Yandex::IDN;
use JSON;
use MIME::Base64;

use BannerStorage;
use EnvTools qw/is_production/;
use Direct::Validation::Image;
use Direct::Model::ImageFormat;
use Direct::Model::ImageFormat::Manager;
use Direct::Model::ImagePool;
use Direct::Model::ImagePool::Manager;
use Direct::Campaigns;
use Direct::CanvasCreatives;
use Direct::Creatives::Tools;
use Direct::AdGroups2;
use Direct::Model::BannerImageAd;
use Direct::Banners::ImageAd;
use Direct::ImageFormats;
use Direct::Validation::AdGroupsText qw/validate_add_banners_text_adgroup validate_update_banners_text_adgroup/;
use Direct::Validation::AdGroupsMobileContent qw/validate_add_banners_mobile_adgroup validate_update_banners_mobile_adgroup/;


use PrimitivesIds qw/get_uid/;
use LockTools qw/get_file_lock release_file_lock/;
use Tools qw/parse_image_name_from_url/;
use TTTools qw();
use BannerImages qw/banner_prepare_save_image banner_assign_image/;
use BannerImages::Pool;
use Models::Banner qw/check_banner_ignore_moderate/;

use ScriptHelper sharded => 1, script_timer => undef, get_file_lock => undef, 'Yandex::Log' => 'messages';

# workaround for: Can't call method "_put_session" on an undefined value at /usr/lib/perl5/AnyEvent/Handle.pm
# it's a race condition in the object destroying order
use Carp::Always;

our @RETRY_INTERVALS = (60, 5*60);   # интервалы перед повторной попыткой скачивания
our $MAX_TRIES_QTY = 3;
our $MAX_REQUEST_PER_DOMAIN = 2;

our $IMAGES_GRAB_DURATION = 30 * 60;
our $IMAGES_PER_TASK = 10;          # количество картинок для одновременного скачивания

my $UNIQ;
my $ONCE;
my $CLIENT_ID;
my $FAKE_CREATIVES;
extract_script_params(
    "uniq=s" => \$UNIQ,
    "once" => \$ONCE,
    "clientid=i" => \$CLIENT_ID,
    "fake-creatives-base64=s" => \$FAKE_CREATIVES,
);
die '--uniq param is mandatory' unless defined $UNIQ;

my $SCRIPT_NAME = get_script_name();
if ( !get_file_lock('dont_die', "$SCRIPT_NAME.$UNIQ") ) {
    exit 0;
}

$log->msg_prefix("[shard_$SHARD,uniq_$UNIQ]");

$log->out("start  $SCRIPT_NAME ...");

$BannerImages::Queue::log->{tee} = $log->{tee};

if ($FAKE_CREATIVES && !is_production){
    #В целях тестестирования, позволяем передать предопределенный список canvas-креативов
    eval {
        $FAKE_CREATIVES = decode_json(decode_base64($FAKE_CREATIVES));
        1;
    } || die sprintf 'Invalid JSON: %s', $@;
    {
        no warnings 'redefine';
        *BannerStorage::banner_storage_canvas_call = sub {
            return {content => $FAKE_CREATIVES}
        }
    };
}

my $sleep_time = 0;
for (1 .. $Settings::ProcessImageQueue_MAX_ITERATIONS) {
    my $trace = Yandex::Trace->new(service => 'direct.script', method => "ppcProcessImageQueue", tags => "shard_$SHARD");
    
    # стоп-файл и сон в начале цикла для того, чтобы не было никакой возможности обойти этот шаг (через next в середине цикла, например)
    if (my $reason = smart_check_stop_file()) {
        $log->out("smart_check_stop_file: $reason. Let's finish.");
        last;
    }
    if( $sleep_time > 0.001 ){
        $log->out(sprintf "Sleep %.3f seconds before next task", $sleep_time);
        Time::HiRes::sleep($sleep_time);
    }

    $log->out("iteration start");

    $sleep_time = $Settings::ProcessImageQueue_MIN_ITERATION_DURATION;
    my $t1 = Time::HiRes::time();

    # полезная работа
    process($SHARD);
    clean($SHARD);

    my $work_duration = Time::HiRes::time() - $t1;
    $work_duration = int($work_duration * 100)/ 100;
    # спим так, чтобы итерации не повторялись бы слишком часто
    $sleep_time = max($Settings::ProcessImageQueue_MIN_ITERATION_DURATION - $work_duration, 0);
    juggler_ok(service_suffix => "uniq_$UNIQ");
    $log->out("iteration end, duration $work_duration, to sleep $sleep_time sec");

    if ($ONCE) {
        $log->out("exit after first iteration");
        last;
    }
}

release_file_lock();

$log->out("finish");



=head2 process

    обработка текущей очереди

=cut

sub process {
    my ($shard) = @_;
    $log->out('start processing');
    
    $Yandex::DBQueue::LOG = $log;
    
    my $queue = Yandex::DBQueue->new(PPC(shard => $shard), 'banner_images');

    my @jobs = $queue->grab_jobs(grab_for => $IMAGES_GRAB_DURATION, limit => 100, filters => {
        ( $CLIENT_ID ? ( ClientID => $CLIENT_ID ) : () ),
    });

    my $queue_size = 0;
    for my $jobs_chunk (chunks \@jobs, $IMAGES_PER_TASK) {
        $queue_size += scalar @$jobs_chunk;
        download_images($shard, $jobs_chunk);
    }

    local $Yandex::Advmon::GRAPHITE_PREFIX = sub {[qw/direct_one_min db_configurations/, $Settings::CONFIGURATION]};
    monitor_values({ flow => { BannerImagesQueue => { queue_size => { "shard$shard" => $queue_size } } } });

    $log->out('done');
}

sub _gozora_tvm_ticket {
    return eval{Yandex::TVM2::get_ticket($Settings::GOZORA_TVM2_ID)} or $log->die("Cannot get ticket for $Settings::GOZORA_TVM2_ID: $@");
}

=head2 download_images

download_images($shard, \@jobs)

Скачивание картинок

=cut

sub download_images($$)
{
    my ($shard, $jobs) = @_;

    $Yandex::DBQueue::LOG = $log;

    return unless $jobs && @$jobs;

    my ($jobs_ok, $jobs_failed) = part { $_->trycount > $MAX_TRIES_QTY ? 1 : 0 } @$jobs;
    $jobs_ok //= [];
    $jobs_failed //= [];
    for my $failed_job (@$jobs_failed) {
        $log->out("job failed", $failed_job);
        $failed_job->mark_failed_permanently({ error => iget("Неизвестная ошибка")});
    }

    unless (@$jobs_ok) {
        $log->out("No jobs to process");
        return;
    }

    $log->out("processing pack of image queue items: " . join ',', map { $_->job_id } @$jobs_ok);
    
    my ($simple_images, $canvas_creatives) = part {$_->args->{url} =~ qr(^creative\://) ? 1 : 0} @$jobs_ok;
    $_ //= [] foreach ($simple_images, $canvas_creatives);
    
    my $t0 = [gettimeofday];

    my $result = {};
    if (@$simple_images) {
        my %reqs;
        for my $job (@$simple_images) {
            my %request;

            my $url = $job->args->{url};
            my $protocol = get_protocol($url);
            $request{url} = $Settings::GOZORA_PROXY;
            $request{headers} = { 'X-Ya-Dest-Url' => Yandex::IDN::idn_to_ascii($url) };
            $reqs{ $job->job_id } = \%request;
        }

        my ($proxy, $headers);
        $headers = {
            %{$Settings::GOZORA_PROXY_HEADERS},
            'X-Ya-Follow-Redirects' => 'true',
            'X-Ya-Ignore-Certs' => 'true',
            'User-Agent' => "Mozilla/5.0 (compatible; YaDirectFetcher/1.0; Dyatel; +http://yandex.com/bots)",
            'X-Ya-Service-Ticket' => _gozora_tvm_ticket(),
        };


        $result = Yandex::HTTP::http_parallel_request(
            GET => \%reqs,
            $proxy ? ( proxy => $proxy ) : (),
            headers => $headers,
            max_req => $MAX_REQUEST_PER_DOMAIN,
            response_size_limit => $BannerImages::MAX_IMAGE_FILE_SIZE,
            timeout => 10,
        );
        $log->out(sprintf('time elapsed while processing items pack: %.4f', tv_interval ( $t0 )));
    }

    _touch_canvas_creatives($canvas_creatives, $result) if @$canvas_creatives;
    
    my @jobs_to_remove;
    for my $job (@$simple_images, @$canvas_creatives) {        
        my $res = $result->{$job->job_id};
        my ($error, $error_log, $job_result);
        if (defined $res->{error}) {
            ($error, $error_log) = @$res{qw/error error_log/};
        }
        else {
            ($error, $error_log, $job_result) = $res->{creative_id}
                ? process_canvas_creative_job_result($job, $res)
                : process_image_job_result($job, $res);
        }
        unless ($job->is_failed()){
            if ($error) {
                $log->out(sprintf "# %d / %s / %.4f / error: %s", $job->job_id, $job->args->{url}, $res->{elapsed} // 0, ($error_log // $error));
                if ($job->trycount >= $MAX_TRIES_QTY || $job_result->{image_hash}) {
                    $job->mark_failed_permanently({ error => $error });
                } else {
                    $job->mark_failed_once();
                }
            } else {
                $job->mark_finished($job_result);
                $log->out(sprintf "# %d / %s / %.4f / success", $job->job_id, $job->args->{url}, $res->{elapsed} // 0);
                push @jobs_to_remove, $job->job_id;
            }
        }
    }
    if (@jobs_to_remove) {
        my $removed = int do_delete_from_table(PPC(shard => $shard), 'banner_images_process_queue_bid', where => { job_id => \@jobs_to_remove });
        $log->out("removed $removed job_ids from banner_images_process_queue_bid");
    }
}

sub _touch_canvas_creatives {
    my ($jobs_canvas_creatives, $result) = @_;
    
    my (@all_creatives, %unknown_creatives);
    foreach my $job (@$jobs_canvas_creatives){
        $result->{$job->job_id} //= {};
        my $res = $result->{$job->job_id};
        my ($creative_id) = $job->args->{url} =~ /(\d+)/;
        unless($creative_id){
            @$res{qw/error error_log/} = iget('Неверный идентификатор креатива: %s', $job->args->{url});
            next;
        }
        $res->{creative_id} = $creative_id;
        push @all_creatives, $creative_id;
    }
    my $existing_creatives = Direct::Creatives::Tools->get_existing(undef, \@all_creatives, 'canvas');
        
    if (@all_creatives > scalar keys %$existing_creatives) {
        foreach my $job (@$jobs_canvas_creatives){
            my $res = $result->{$job->job_id};
            next if $res->{error};
            if ($existing_creatives->{$res->{creative_id}}) {
                $res->{is_success} = 1;
            }
            else {
                $unknown_creatives{$res->{creative_id}} //= [];
                push @{$unknown_creatives{$res->{creative_id}}}, $job;
            }
        }

        my %unknown_creatives_by_client_id;
        foreach my $creative_id (keys %unknown_creatives){
            foreach my $job (@{$unknown_creatives{$creative_id}}){
                my $client_id = $job->ClientID;
                $unknown_creatives_by_client_id{$client_id} //= [];
                push @{$unknown_creatives_by_client_id{$client_id}}, $creative_id;
            }
        }
        # проверяем есть ли в BS отбранные креативы
        my $ok_creatives = [];
        foreach my $client_id ( keys %unknown_creatives_by_client_id ){
            my $bs_creatives = BannerStorage::receive_canvas_creatives($client_id => $unknown_creatives_by_client_id{$client_id});
        
            my ($local_ok_creatives, $missed_creatives) = part {$_->{ok} ? 0 : 1} @$bs_creatives;
            if ($missed_creatives) {
                #Если часть id - неправильная или не принадлежит нужному клиенту, относящиеся к ним задания сразу завершим
                foreach my $creative (@$missed_creatives) {
                    my ($ok_jobs, $failed_jobs) = part {$_->ClientID eq $client_id} @{$unknown_creatives{$creative->{creativeId}}};
                    foreach my $job (@{$failed_jobs // []}) {
                        $job->mark_failed_permanently({ error => iget('Canvas-креатив с идентификатором %s не существует.', $creative->{creativeId}) });
                        $log->out(sprintf "# %d / %s / %.4f / error: %s", $job->job_id, $job->args->{url}, 0, $creative->{message});
                    }
                    if ($ok_jobs) {
                        $unknown_creatives{$creative->{creativeId}} = $ok_jobs;
                    }
                    else {
                        delete $unknown_creatives{$creative->{creativeId}};
                    }
                }
            }
            push @$ok_creatives, @$local_ok_creatives if $local_ok_creatives;
        }
        
        #для тех, которые есть в BS запустим создание через INTAPI
        my %creatives_by_operator;

        foreach my $creative (@$ok_creatives){
            my $jobs = $unknown_creatives{$creative->{creativeId}};
            next unless $jobs;
            foreach my $job (@$jobs) {
                my ($ClientID, $operator_uid) = map { $job->$_ } (qw/ClientID uid/);
                $creatives_by_operator{$operator_uid}->{$ClientID}->{$creative->{creativeId}} = $creative->{creative};
            }
        }
        
        foreach my $operator_uid (keys %creatives_by_operator){
            foreach my $ClientID (keys %{$creatives_by_operator{$operator_uid}}){
                my ($push_result, $push_error) = Direct::CanvasCreatives->push_creatives(
                    $operator_uid,
                    $ClientID,
                    [values %{$creatives_by_operator{$operator_uid}->{$ClientID}}]
                );
                if ($push_error) {
                    $log->out(sprintf 'intapi push error: operator_uid %s, client_id %s, creatives: %s - %s',
                              $operator_uid,
                              $ClientID,
                              join( ', ', keys %{$creatives_by_operator{$operator_uid}->{$ClientID}}),
                              $push_error
                    );
                    next;
                }
                foreach my $creative_result (@$push_result){
                    if (lc($creative_result->{status}) eq 'error') {
                        my $jobs = delete $unknown_creatives{$creative_result->{creative_id}};
                        foreach my $job (@$jobs){
                            $job->mark_failed_permanently({error => iget('Ошибка обработки canvas-креатива %s.', $creative_result->{creative_id})});
                            $log->out(sprintf "# %d / %s / %.4f / error: %s", $job->job_id, $job->args->{url}, 0, $creative_result->{message});
                        }
                    }
                }
            }
        }
        #Теперь еще раз проверим отсутствовавшие креативы в базе
        $existing_creatives = Direct::Creatives::Tools->get_existing(undef, [keys %unknown_creatives], 'canvas');
        foreach my $job(map { @$_ } values %unknown_creatives){
            my $res = $result->{$job->job_id};
            $res->{is_success} = 1 if $existing_creatives->{$res->{creative_id}}; 
        }
    }
    
    return $result;
}


sub process_canvas_creative_job_result {
    my ($job, $res) = @_;

    unless ($res && $res->{is_success}) {
        return (@$res{qw/error error_log/}, undef) if $res->{error};
        return (iget('Креатив %s отсутствует в Директе', $res->{creative_id}));
   }

    my $error = '';
    my $error_log = '';
    my $image_hash;

    my $pid = $job->args->{pid};

    my $adgroup = Direct::AdGroups2->get_by(adgroup_id => $pid, adgroup_type => [qw/base mobile_content/], extended => 1)->items->[0];

    unless ($adgroup) {
        $error = iget("Не удалось привязать баннер к группе");
        $error_log = "adgroup pid = $pid not found";
        return ($error, $error_log, undef);
    }

    my $banner;    
    my $bid = get_one_field_sql(PPC(shard => $SHARD), ["select bid from banner_images_process_queue_bid", where => {
        job_id => $job->job_id
    }]);
    
    if ($bid) {
        #Обновляем существующий баннер
        $banner = Direct::Banners->get_by(banner_id => $bid)->items->[0];
        unless ($banner) {
            return (iget('Не удалось изменить изображение'), sprintf('banner bid = %s not found', $bid), undef);
        }
        if ($banner->has_image_ad){
            return (iget('Невозможно изменить тип изображения'), sprintf('can`t change creative to image_ad for bid=%s, $bid'), undef);
        }
        $banner->old($banner->clone);
        return (iget('Невозможно изменить изображение - предыдущий креатив не найден', sprintf('banner bid = %s - previous creative not found' ), undef))
            unless $banner->old->creative;
    }
    else{
        #Создаем новый
         $banner = Direct::Model::BannerImageAd->new(
            id => 0, client_id => $job->ClientID, campaign_id => $adgroup->campaign_id, adgroup_id => $adgroup->id,
        );
    }
    
    $banner->adgroup($adgroup);
    $adgroup->campaign(Direct::Campaigns->get_by(campaign_id => $adgroup->campaign_id)->items->[0]);
    my $uid = get_uid(cid => $adgroup->campaign_id);
    MailNotification::save_UID_host($uid);
    
    my $creative = Direct::CanvasCreatives->get_by(creative_id => [$res->{creative_id}], $uid)->items->[0];
    if (!$banner->has_creative) {
        my $canvas_creative = Direct::Model::BannerCreative->new(
            campaign_id => $adgroup->campaign_id,
            adgroup_id => $adgroup->id,
            banner_id => $banner->id,
            creative_id => $res->{creative_id},
            creative => $creative,
        );

        $banner->creative($canvas_creative);
    }
    else {
        $banner->creative->creative_id($res->{creative_id});
        $banner->creative->creative($creative);
    }
    $banner->href($job->args->{href}) if $job->args->{href};
    
        my %validate_banner = (
        $adgroup->adgroup_type eq 'mobile_content'
        ? (
            add => \&validate_add_banners_mobile_adgroup,
            update => \&validate_update_banners_mobile_adgroup,
        )
        : (
            add => \&validate_add_banners_text_adgroup,
            update => \&validate_update_banners_text_adgroup,
        )
    );

    if ($bid) {
        
        my $vr = $validate_banner{update}->([$banner], $adgroup);
        unless ($vr->is_valid) {
            $error = $vr->get_first_error_description();
            $error_log = "Banner validation failed: ".$error;
            return ($error, $error_log, undef);
        }

        Direct::Banners::ImageAd->new([$banner])->update($uid);
    }
    else {
        my $vr = $validate_banner{add}->([$banner], $adgroup);
        unless ($vr->is_valid) {
            $error = $vr->get_first_error_description();
            $error_log = "Banner validation failed ".$error;
            return ($error, $error_log, undef);
        }
        $banner->href(undef) unless $banner->has_href;
        do_in_transaction {
            Direct::Banners::ImageAd->new([$banner])->create($uid);
        }
    }

    return ($error, $error_log, {image_hash => $image_hash});
}

=head2 process_image_job_result

Обработка одной скачанной картинки
$job - объект Y::DBQueue::Job
$res - ответ из http_parallel_request для этой $job

Возвращает список из трех элементов:
    - сообщение об ошибке для пользователя
    - сообщение об ошибке для записи в лог
    - image_hash

=cut

sub process_image_job_result
{
    my ($job, $res) = @_;
    my $error = '';
    my $error_log = '';
    my $image_hash;
    
    unless ($res && $res->{is_success}) {
        return (@$res{qw/error error_log/}, undef) if $res->{error};
        # не сумели скачать картинку
        my $reason = join ' ', map { $res->{headers}->{$_} } grep { defined $res->{headers}->{$_} } qw/Status Reason/;
        utf8::decode($reason) unless utf8::is_utf8($reason);
        # скрываем статусы зоры
        $reason =~ s/\bzora:.*/internal spider error/xms;
        $error ||= iget('Ошибка при загрузке файла') . ($reason ? " ($reason)" : '');
        my $status_line = "$res->{headers}->{Status} $res->{headers}->{Reason}";
        utf8::decode($status_line) if !utf8::is_utf8($status_line);
        $error_log = "File download error / $status_line";
        return ($error, $error_log, undef);
    }

    if ($job->args->{pid} && ($job->args->{ad_type}//'') eq 'image_ad') {
        return process_imagead_job_result($job, $res);
    }
    
    my $check = banner_prepare_save_image($res->{content}, { ClientID => $job->ClientID });
    $image_hash = $check->{md5};
    unless ($image_hash) {
        $error = iget('Ошибка сохранения изображения');
        $error_log = 'Image saving error';
        return ($error, $error_log, undef);
    }

    my $job_result = {image_hash => $image_hash, image_type => $check->{image_type}};
    if ($check->{error}) {
        # не сумели сохранить картинку (проблемы с аватарницей или картинка неверного формата)
        $error = $check->{error};
        $error_log = "Image processing error / $error";
        return ($error, $error_log, $job_result);
    }

    hash_copy $job_result, $check, qw/namespace mds_group_id/;

    my $fname = $job->args->{name} // parse_image_name_from_url($job->args->{url});

    my $banners = get_all_sql(PPC(ClientID => $job->ClientID), 
        "select qb.bid, b.cid, c.type, c.statusEmpty, b.statusModerate
        from banner_images_process_queue_bid qb
        inner join banners b using(bid)
        inner join campaigns c using(cid)
        where qb.job_id = ?", $job->job_id);

    # картинку добавляем в пул, если список баннеров пустой (т.е это вызов из апи)
    # или все баннеры из подходящего типа кампаний
    unless (all { BannerImages::is_image_type_allowed_for_campaign($_->{type}, $check->{image_type}) } @$banners) {
        $error = iget('Размер картинки не соответствует типу кампании');
        $error_log = 'Image assigning error';
        return ($error, $error_log, $job_result);
    }
    
    BannerImages::Pool::add_items([{ ClientID => $job->ClientID, image_hash => $image_hash, name => $fname }]);
    
    $log->out("assing image $image_hash to banners", $banners);
    foreach my $banner (@$banners) {
        my $ignore_moderate = check_banner_ignore_moderate($banner);
        banner_assign_image($banner->{cid}, $banner->{bid}, $image_hash,
            ($ignore_moderate ? 'New' : 'Ready'),
            {name => $fname, skip_pool => 1},
        );
    }
    return (undef, undef, $job_result);
}

=head2 process_imagead_job_result

=cut

sub process_imagead_job_result
{
    my ($job, $res) = @_;
    my $error = '';
    my $error_log = '';
    my $image_hash;

    if (length($res->{content}) > $BannerImages::MAX_IMAGEAD_FILE_SIZE) {
        $error = iget("Размер файла больше допустимого (%s)", TTTools::format_file_size($BannerImages::MAX_IMAGEAD_FILE_SIZE));
        $error_log = "File too big (@{[length($res->{content})]}";
        return ($error, $error_log, undef);
    }
    
    my $validation = Direct::Validation::Image::validate_images([$res->{content}], type => 'image_ad');
    unless ($validation->is_valid()) {
        $error = $validation->get_first_error_description();
        $error_log = 'Image validation failed: '.$error;
        return ($error, $error_log, undef);
    }
    
    my $fname = $job->args->{name} // parse_image_name_from_url($job->args->{url});

    my $format = Direct::Model::ImageFormat->new(image => $res->{content}, name => $fname, image_type => 'image_ad');
    my $format_manager = Direct::Model::ImageFormat::Manager->new(items => [$format]);
    my $errors = $format_manager->save(shard => $SHARD, namespace => 'direct-picture');
    if (%$errors) {
        $error = iget('Ошибка при сохранении файла');
        $error_log = "Format Manager / Avatars error: ".(to_json($errors));
        return ($error, $error_log, undef);
    }

    $image_hash = $format->hash;

    my $pool = Direct::Model::ImagePool->new(client_id => $job->ClientID, name => $fname, hash => $image_hash);
    my $pool_manager = Direct::Model::ImagePool::Manager->new(items => [$pool]);
    $pool_manager->create();

    my $pid = $job->args->{pid};

    my $adgroup = Direct::AdGroups2->get_by(adgroup_id => $pid, adgroup_type => [qw/base mobile_content/], extended => 1)->items->[0];

    my $job_result = {
        image_hash => $image_hash,
        mds_group_id => $format->mds_group_id,
        namespace => 'direct-picture',
    };

    unless ($adgroup) {
        $error = iget("Не удалось привязать баннер к группе");
        $error_log = "adgroup pid = $pid not found";
        return ($error, $error_log, $job_result);
    }

    my $bid = get_one_field_sql(PPC(shard => $SHARD), ["select bid from banner_images_process_queue_bid", where => {
        job_id => $job->job_id
    }]);

    my $banner;

    if ($bid) {
        # меняем картинку в существующем баннере
        $banner = Direct::Banners->get_by(banner_id => $bid)->items->[0];
        unless ($banner) {
            $error = iget("Не удалось изменить изображение");
            $error_log = "banner bid = $bid not found";
            return ($error, $error_log, $job_result);
        }
        if ($banner->has_creative){
            return ('Невозможно изменить тип изображения', sprintf('can`t change image_ad to creative for bid=%s, $bid'), undef);
        }

        $banner->old($banner->clone);
        my $prev_format = Direct::ImageFormats->get_by(banner_id => $bid, filter => {image_type => 'image_ad'})->items->[0];
        unless ($prev_format) {
            $error = iget("Не удалось изменить изображение - предыдущее изображение не найдено");
            $error_log = "image format for bid = $bid not found";
            return ($error, $error_log, $job_result);
        }
        $banner->old->image_ad->format($prev_format);
    }
    else {
        # создаем новый баннер
        $banner = Direct::Model::BannerImageAd->new(
            id => 0, client_id => $job->ClientID, campaign_id => $adgroup->campaign_id, adgroup_id => $adgroup->id,
        );
    }
    $banner->adgroup($adgroup);
    $adgroup->campaign(Direct::Campaigns->get_by(campaign_id => $adgroup->campaign_id)->items->[0]);

    if (!$banner->has_image_ad) {
        my $image_ad = Direct::Model::Image->new(hash => $image_hash);
        $image_ad->campaign_id($banner->campaign_id);
        $image_ad->adgroup_id($banner->adgroup_id);
        $banner->image_ad($image_ad);
    }
    else {
        $banner->image_ad->hash($image_hash);
        $banner->image_ad->format($format);
    }
    $banner->href($job->args->{href}) if $job->args->{href};

    my $uid = get_uid(cid => $adgroup->campaign_id);
    MailNotification::save_UID_host($uid);

    my %validate_banner = (
        $adgroup->adgroup_type eq 'mobile_content'
        ? (
            add => \&validate_add_banners_mobile_adgroup,
            update => \&validate_update_banners_mobile_adgroup,
        )
        : (
            add => \&validate_add_banners_text_adgroup,
            update => \&validate_update_banners_text_adgroup,
        )
    );


    if ($bid) {
        
        my $vr = $validate_banner{update}->([$banner], $adgroup);
        unless ($vr->is_valid) {
            $error = $vr->get_first_error_description();
            $error_log = "Banner validation failed: ".$error;
            return ($error, $error_log, $job_result);
        }

        Direct::Banners::ImageAd->new([$banner])->update($uid);
    }
    else {
        my $vr = $validate_banner{add}->([$banner], $adgroup);
        unless ($vr->is_valid) {
            $error = $vr->get_first_error_description();
            $error_log = "Banner validation failed ".$error;
            return ($error, $error_log, $job_result);
        }
        $banner->href(undef) unless $banner->has_href;
        do_in_transaction {
            Direct::Banners::ImageAd->new([$banner])->create($uid);
        };
    }

    return (undef, undef, $job_result);
}

=head2 clean

=cut

sub clean
{
    my $shard = shift;
    my $queue = Yandex::DBQueue->new(PPC(shard => $shard), 'banner_images');
    # при таком status удаление будет только из таблицы dbqueue_job_archive. Удаление из dbqueue_jobs будет выполнено мнгновенно, т.к. mysql увидит impossible where
    my $deleted = $queue->delete_old_jobs(24 * 60 * 60, status => ['Finished','Failed','Revoked']);
    $log->out("deleted $deleted old jobs");
}


