#!/usr/bin/perl

=pod

=encoding utf8

=head1 METADATA

<crontab>
    time: * * * * *
    sharded: 1
    <switchman>
        group: scripts-other
        <leases>
            mem: 230
        </leases>
    </switchman>
    package: scripts-switchman
</crontab>
<juggler>
    host:   checks_auto.direct.yandex.ru
    sharded: 1
    ttl: 20m
    tag: direct_group_internal_systems
</juggler>
<monrun>
    juggler_host:   checks_auto.direct.yandex.ru
    name: push.push_queue_length
    warn: 300000
    expression: 'movingAverage(sumSeries(one_min.direct.production.push.push_queue_length.shard_*),"5min")'
    tag: direct_group_internal_systems
</monrun>
<monrun>
    juggler_host:   checks_auto.direct.yandex.ru
    name: push.push_queue_errors
    crit: 2
    expression: 'sumSeries(one_min.direct.production.push.push_queue_errors.shard_*)'
    tag: direct_group_internal_systems
</monrun>

=cut

=head1 NAME

ppcPushNotifications.pl - отправка push-уведомлений

=head1 SYNOPSIS

./protected/ppcPushNotifications.pl --shard-id=12 --once

=head1 DESCRIPTION

$Id$

Скрипт для сбора и отправки push-уведомлений

Скрипт собирает события из Eventlog для пользователей, подписавшихся на них,
по таблицам users_notifications и users_notifications_details, складывает из
них личные списки событий в таблицу push_notifications_process, затем собирает
эти списки в готовые сообшения для отправки на конечные устройства через Xiva.

Параметры:

    --help — показать справку из описания,
    --shard-id=N — обязательный идентификатор шарда,
    --debug — включить синхронную отправку, дающую подробный ответ,
    --once — завершиться после одного цикла отправки.

=cut

use warnings;
use strict;
use utf8;

use JSON;
use List::MoreUtils qw/firstval/;

use Yandex::Advmon;
use Yandex::DateTime qw/now/;
use Yandex::DBShards qw/SHARD_IDS/;
use Yandex::DBTools;
use Yandex::I18n;
use Yandex::Xiva;

use my_inc "..";

use Settings;
use ScriptHelper 'get_file_lock' => ['dont_die'],
    'Yandex::Log' => 'messages',
    'sharded' => 1;

use Currency::Format;
use EventLog qw/is_event_for_object push_notifications_enrich_payload INFORMATIONAL_WITH_LINK_TYPE/;
use LockTools;
use PrimitivesIds;
use Property;
use TextTools;


my $TIMEOUT = 60;#5 * 60; #seconds

my $MAX_RANK = 10; #times

my %push_messages = (
	# при изменениях нужно актуализировать https://wiki.yandex-team.ru/direct/Push-uvedomlenija-dlja-mobilnogo-prilozhenija/Tipy-sobytijj-pushejj/
	money_out 					=> iget_noop('На кампании "%s" закончились деньги'),
	money_out_wallet 			=> iget_noop('На общем счете закончились деньги'),

	money_warning 				=> iget_noop('На кампании "%s" израсходовано 80%% средств'),
	money_warning_wallet 		=> iget_noop('На общем счете израсходовано 80%% средств'),
    money_warning_one_day_remain_wallet => iget_noop('При текущем уровне расходов средств на счёте хватит на 1 день'),
	money_warning_three_days_remain_wallet => iget_noop('При текущем уровне расходов средств на счёте хватит на 3 дня'),

	money_in 					=> iget_noop('На кампанию "%s" поступило %s'),
	money_in_wallet 			=> iget_noop('На общий счет поступило %s'),

	camp_finished 				=> iget_noop('Кампания "%s" остановлена, так как наступила дата окончания показов'),

	paused_by_day_budget 		=> iget_noop('Кампания "%s" остановлена по дневному бюджету'),
    paused_by_day_budget_wallet => iget_noop('Кампании остановлены по дневному бюджету'),

	warn_place 					=> iget_noop('У фразы "%s" изменился объём трафика'),

	banner_moderated_accepted 	=> iget_noop('Объявление "%s" прошло модерацию'),
	banner_moderated_declined 	=> iget_noop('Объявление "%s" отклонено на модерации'),
	banner_moderated_declined_partly => iget_noop('Объявление "%s" частично отклонено на модерации'),
);

my %push_messages_multi = (
	money_out_multi => [
		iget_noop('На %s кампании закончились деньги'),
		iget_noop('На %s кампаниях закончились деньги'),
		iget_noop('На %s кампаниях закончились деньги')
	],
	money_warning_multi => [
		iget_noop('На %s кампании израсходовано 80%% средств'),
		iget_noop('На %s кампаниях израсходовано 80%% средств'),
		iget_noop('На %s кампаниях израсходовано 80%% средств')
	],
	money_in_multi => [
		iget_noop('На %s кампанию поступила оплата'),
		iget_noop('На %s кампании поступила оплата'),
		iget_noop('На %s кампаний поступила оплата')
	],
	camp_finished_multi => [
		iget_noop('%s кампания остановлена, так как наступила дата окончания показов'),
		iget_noop('%s кампании остановлены, так как наступила дата окончания показов'),
		iget_noop('%s кампаний остановлено, так как наступила дата окончания показов')
	],
	paused_by_day_budget_multi => [
		iget_noop('%s кампания остановлена по дневному бюджету'),
		iget_noop('%s кампании остановлены по дневному бюджету'),
		iget_noop('%s кампаний остановлено по дневному бюджету')
	],
	warn_place_multi => [
		iget_noop('У %s фразы изменился объём трафика'),
		iget_noop('У %s фраз изменился объём трафика'),
		iget_noop('У %s фраз изменился объём трафика')
	],

	banner_moderated_accepted_multi => [
		iget_noop('%s объявление прошло модерацию'),
		iget_noop('%s объявления прошли модерацию'),
		iget_noop('%s объявлений прошли модерацию')
	],
	banner_moderated_declined_multi => [
		iget_noop('%s объявление отклонено на модерации'),
		iget_noop('%s объявления отклонены на модерации'),
		iget_noop('%s объявлений отклонены на модерации')
	],
	banner_moderated_declined_partly_multi => [
		iget_noop('%s объявление частично отклонено на модерации'),
		iget_noop('%s объявления частично отклонены на модерации'),
		iget_noop('%s объявлений частично отклонены на модерации')
	],

	campaign_multi => [
		iget_noop('В кампании "%s" произошло %s событие'),
		iget_noop('В кампании "%s" произошло %s события'),
		iget_noop('В кампании "%s" произошло %s событий')
	],
	campaign_multi_wallet => [
		iget_noop('На общем счете произошло %s событие'),
		iget_noop('На общем счете произошли %s события'),
		iget_noop('На общем счете произошло %s событий')
	],

	many_camps => [
		iget_noop('Произошло %s событие'),
		iget_noop('Произошло %s события'),
		iget_noop('Произошло %s событий')
	],
);

my $ID_TO_EVENT = EventLog::get_type2slug();

$log->out("Start daemon ...");

my ($ONCE, $DEBUG);
extract_script_params(
	help => sub {system("podselect -section NAME -section DESCRIPTION $0 | pod2text >&2"); exit 0;},
    once => \$ONCE,
    debug => \$DEBUG
);
if ($DEBUG) {
    $log->out("Debug send mode enabled");
    $Yandex::Xiva::DEBUG_SYNC_SEND_MODE = 1;
}

# Ошибки при отправке пушей
my $errors;

while (1) {
    if (my $reason = smart_check_stop_file()) {
        $log->out("$reason!");
        last;
    }

    $errors = 0;

    cleanup_erroneous_notifications();

	fill_push_queue();

	compile_and_send_pushes();

	my $cnt = get_one_field_sql(PPC(shard => $SHARD), 'select count(*) from push_notifications_process');
	# пишем данные для нового мониторинга
    monitor_values({
        "push.push_queue_length.shard_$SHARD" => $cnt,
        "push.push_queue_errors.shard_$SHARD" => $errors,
    });
    $log->out(qq!monitor_values: "push.push_queue_length.shard_$SHARD" => $cnt, "push.push_queue_errors.shard_$SHARD" => $errors!);

    juggler_ok();
    last if $ONCE;
    $log->out("sleep for $TIMEOUT");

	sleep $TIMEOUT;
    Yandex::Trace::restart(\$ScriptHelper::trace);
}

$log->out("Stop daemon ...");

=head1 METHODS

=head2 cleanup_erroneous_notifications

    удаляем записи из очереди, если много раз не получилось их запушить

=cut

sub cleanup_erroneous_notifications {
    my $notification_ids_to_remove = get_one_column_sql(PPC(shard => $SHARD),
        "SELECT notification_id FROM push_notifications_process WHERE rank > $MAX_RANK"
    );
    if (@$notification_ids_to_remove) {
        # логируем
        my $data = get_all_sql(PPC(shard => $SHARD), ['
                                    SELECT notification_id, uid, type, device_id, tag
                                      FROM users_notifications',
                                     WHERE => {notification_id => $notification_ids_to_remove}
                                    ]);
        $log->out("Notifications to be removed due to lots of errors while sending pushes", $data);

        # удаляем
        do_delete_from_table(PPC(shard => $SHARD), 'users_notifications',
            where => {notification_id => $notification_ids_to_remove});

        do_delete_from_table(PPC(shard => $SHARD), 'users_notifications_details',
            where => {notification_id => $notification_ids_to_remove});

        do_delete_from_table(PPC(shard => $SHARD), 'push_notifications_process',
            where => {notification_id => $notification_ids_to_remove});
    }

    # заодно удаляем те записи, которые не сможем отправить потому, что связанная запись удалена
    my $notification_ids_without_a_subscription = get_one_column_sql(PPC(shard => $SHARD),
        ["SELECT pnp.notification_id FROM push_notifications_process pnp LEFT JOIN users_notifications un ON pnp.notification_id = un.notification_id",
            where => {'un.notification_id__is_null' => 1}
        ]
    );
    if (@$notification_ids_without_a_subscription) {
        $log->out("Notifications to be removed because they are missing in users_notifications", $notification_ids_without_a_subscription);
        do_delete_from_table(PPC(shard => $SHARD), 'push_notifications_process',
            where => {notification_id => $notification_ids_without_a_subscription});
    }
}

=head2 fill_push_queue

	Фаза 1: собираем произошедшие события в спец. табличку

=cut

sub fill_push_queue {
	my $last_id_prop = Property->new("ppcPushNotification_last_shard_$SHARD");

	# Получаем последний извлеченный id события
	my $last_id = $last_id_prop->get() || 0;

	# чтобы не доставать много лишнего в случае падения скрипта, ограничиваем выборку часовым интервалом
    my $time_border = $last_id ? now()->add(hours => -1) : now()->add(minutes => -5);
	my $where_clause = {
        id__gt => $last_id,
        eventtime__gt => "$time_border",
    };

	# достаем текущий последний id события, запоминаем его и добавляем в условие выборки для консистентности
	my $next_id = get_one_field_sql(PPC(shard => $SHARD), ['select max(id) from eventlog', where => $where_clause]);
    return unless $next_id;

	$where_clause->{id__le} =  $next_id;

	# Выбираем всех клиентов, для которых произошли события
	my $client_ids = get_one_column_sql(PPC(shard => $SHARD), ['select distinct ClientID from eventlog', where => $where_clause]);
	$log->out("Clients with events " . (scalar @$client_ids));

	# фильтруем клиентов, выбираем те uid'ы, которые заказывали пуши
	my $uids2clientid = get_uid2clientid(ClientID => $client_ids);

	my $uids_data = get_all_sql(PPC(shard => $SHARD), [ "
			SELECT un.uid, un.notification_id, und.event_type
			  FROM users_notifications un
                   LEFT JOIN users_notifications_details und using (notification_id)",
             WHERE => {'un.uid' => [keys %$uids2clientid]}
        ]);
	# Собираем структуру: кому что отправлять, вида:
	#	ClientID => {
	#					notification_id1 => {
	#							event1 => 1,
	#							event2 => 1
	#							},
	#					notification_id2 => {
	#							event1 => 1,
	#							event2 => 1
	#							},
	#					},
	#				}
	my %push_data;
    foreach my $item (@$uids_data) {
        my $notification = $push_data{$uids2clientid->{$item->{uid}}}{$item->{notification_id}} ||= {};
        $notification->{$item->{event_type}} = 1;
        $notification->{money_out_wallet} = 1 if $item->{event_type} eq 'money_out';
        $notification->{money_in_wallet} = 1 if $item->{event_type} eq 'money_in';
        $notification->{money_warning_wallet} = 1 if $item->{event_type} eq 'money_warning';
    }

	$where_clause->{ClientID} = [keys %push_data];

	# where_clause here:
	# id > prev_last_id, id < current_last_id, ClientID in (clients who have events and need to be pushed),
	# eventtime > one hour ago
	my $events = get_all_sql(PPC(shard => $SHARD), ['select id, ClientID, type from eventlog', where => $where_clause]);
	$log->out("Events to compile " . (scalar @$events));
	# Сохраняем в табличку данные по произошедшим событиям на notification_id
	my @data_to_proc_table;
	foreach my $event (@$events) {
		foreach my $notification_id (keys %{$push_data{$event->{ClientID}}}) {
			if ($push_data{$event->{ClientID}}{$notification_id}{ $ID_TO_EVENT->{$event->{type}} }) {
				push @data_to_proc_table, [$notification_id, $event->{id}];
			}
		}
	}
    $log->out("Going to insert " . (scalar @data_to_proc_table) . " records into push_notifications_process");
	my $cnt = do_mass_insert_sql(PPC(shard => $SHARD), "insert into push_notifications_process(notification_id, event_id) values %s", \@data_to_proc_table);
    $log->out("Inserted $cnt records into push_notifications_process");

	$last_id_prop->set($next_id);
}


=head2 compile_and_send_pushes

	Фаза 2: рассылаем пуши по спец. табличке

=cut

sub compile_and_send_pushes {
    my $notification_ids_with_events = get_one_column_sql(PPC(shard => $SHARD),
        'select distinct notification_id from push_notifications_process'
    );
    $log->out("Selected " . (scalar @$notification_ids_with_events) . " potential notifications to send");
    my $notification_ids_to_push = get_hashes_hash_sql(PPC(shard => $SHARD), [
        'select un.notification_id, un.locale, un.uid, un.tag, un.device_id from users_notifications un ',
        'where un.last_sent < now() - interval un.timeout minute and '
        , {'un.notification_id' => $notification_ids_with_events}
    ]);
    $log->out("Selected " . (scalar keys %$notification_ids_to_push) . " notifications to send");

	my %pushes;
	my $text_descrs;
	if (scalar keys %$notification_ids_to_push) {
		my $events = get_all_sql(PPC(shard => $SHARD), [
			'select pn.notification_id, el.*
			from push_notifications_process pn
			left join eventlog el on pn.event_id = el.id',
			where => { notification_id => [keys %$notification_ids_to_push] }]);

		$text_descrs = EventLog::get_events_text_descr($events);

		do_sql(PPC(shard => $SHARD), ['update push_notifications_process set rank = rank + 1', where=>{notification_id => [keys %$notification_ids_to_push]}]);

		# список протухших событий в очереди на отправку (например, фраза удалилась, а пришло событие на неё)
		my @events_to_remove;

		foreach my $event (@$events) {
			if (!$text_descrs->{$event->{id}}) {
				push @events_to_remove, $event->{id};
				next;
			}
			my $type = $ID_TO_EVENT->{$event->{type}};
			if ($type eq 'banner_moderated') {
				my $mod_data = JSON::from_json($event->{params});
				$type .= '_'.$mod_data->{results}{global};
			}

            my $push = $pushes{$event->{notification_id}} ||= {
                notification_id => $event->{notification_id},
                cids => {},
                event_types => {},
                events => [],
            };
			$push->{cids}{$event->{cid}} = 1;
			$push->{event_types}{$type} = 1;
			push @{$push->{events}}, $event;
		}
		if (@events_to_remove) {
			do_delete_from_table(PPC(shard => $SHARD), 'push_notifications_process', where => {event_id => \@events_to_remove});
		}
	}

    # собираем пуш-нотификации (не более одной на notification_id за исключением типа custom_message_with_link, который идёт обособленно вне зачёта)
    my $compiled_pushes = compile_pushes([values %pushes], $notification_ids_to_push, $text_descrs);
    if ($compiled_pushes->{messages} && @{$compiled_pushes->{messages}}) {
        $log->out("Sending " . (scalar @{$compiled_pushes->{messages}}) . " pushes by Xiva");
        my $results_array = xiva_send_messages($compiled_pushes->{messages});
        my $sent_count = 0;
        for (my $i = 0; $i < @$results_array; $i++) {
            my $notification_id = $compiled_pushes->{notification_ids}->[$i];
            my $code = $results_array->[$i];
            if ($code == 200) {
                $sent_count++;
            } else {
                $log->out("Xiva $code error");
                $errors++;
                delete $notification_ids_to_push->{$notification_id};
            }
        }
        $log->out("Sent ". $sent_count . " pushes by Xiva");
    }

	# чистим очередь от отправленных пушей, обновляем last_sent
	if ($notification_ids_to_push && scalar keys %$notification_ids_to_push) {
		do_delete_from_table(PPC(shard => $SHARD), 'push_notifications_process', where => {notification_id => [ keys %$notification_ids_to_push ]});
		do_sql(PPC(shard => $SHARD), ['update users_notifications set last_sent = now()', where => {notification_id => [keys %$notification_ids_to_push]}])
	}
}

sub _enrich_payload {
    my $link_error = push_notifications_enrich_payload(@_);
    if ($link_error) {
        my (undef, $msg_type, $first_event) = @_;
        $log->out("couldn't parametrize a link $link_error in type $msg_type event $first_event->{id}");
    }
}

sub compile_pushes {
    my $CHANGE_NOTIFICATION_MESSAGES_FOR_DAYS_WARNING_WALLET = Property->new('CHANGE_NOTIFICATION_MESSAGES_FOR_DAYS_WARNING_WALLET');
 
    my ($pushes, $notification_ids_to_push, $text_descrs) = @_;

    my %pushes_compiled;
    for my $push (@$pushes) {
        my $notification_id = $push->{notification_id};
		my $device_id = $notification_ids_to_push->{$notification_id}->{device_id};
		my $tag;
		if (!$device_id) {
			$tag = $notification_ids_to_push->{$notification_id}->{tag};
		}
        my @subscription_line = $device_id ? (device_id => $device_id) : (tag => $tag);
		my @cids = keys %{$push->{cids}};

		my $wallet_cids = get_hashes_hash_sql(PPC(cid => \@cids),
				["select cid from campaigns", where => {cid => SHARD_IDS, statusEmpty => 'No', type__in => ['wallet']}] );

        my $uid = $notification_ids_to_push->{$notification_id}->{uid};

        my $has_custom_message = delete $push->{event_types}->{custom_message_with_link};
        # считаем, что мы пуши с заранее заданной ссылкой отправляем нечасто, и не надо брать в отправку более одного события такого типа
        if ($has_custom_message) {
            my $custom_message_event = firstval {$_->{type} == INFORMATIONAL_WITH_LINK_TYPE} @{$push->{events}};
            if ($text_descrs->{$custom_message_event->{id}}->{descr}) {
                push @{$pushes_compiled{messages}}, {
                    uid => $uid,
                    message => $text_descrs->{$custom_message_event->{id}}->{descr},
                    payload => {link => $text_descrs->{$custom_message_event->{id}}->{link}, category => 'LINK', T => INFORMATIONAL_WITH_LINK_TYPE},
                    @subscription_line,
                };
            }

            $push->{events} = [grep {$_->{type} != INFORMATIONAL_WITH_LINK_TYPE} @{$push->{events}}];
            if (!@{$push->{events}}) {
                # больше ничего не осталось
                push @{$pushes_compiled{notification_ids}}, $notification_id;
                next;
            }
        }

		my @event_types = keys %{$push->{event_types}};
		my $camps_num = scalar @cids;
		my $event_types_num = scalar @event_types;
		my $events_num = scalar @{$push->{events}};
		my $locale = $notification_ids_to_push->{$notification_id}{locale};

		my $first_event = $push->{events}[0];
		my $first_event_obj = $EventLog::EVENTS{$ID_TO_EVENT->{$first_event->{type}}}{object};

		# инициализируем локаль пушей
		Yandex::I18n::init_i18n($locale, check_file => 1);

		my $msg;
		my $payload = {};

        if ($events_num == 1) {
            my $msg_type = $event_types[0];
            if ( is_event_for_object($msg_type, 'campaign') ) {
                if ($msg_type =~ m/_wallet$/) {
                    $payload = {'A' => $first_event->{cid} };
                } else {
                    $payload = {'C' => $first_event->{cid} };
                }
                if ($msg_type =~ m/^money_in/) {
                    my $pay_data = JSON::from_json($first_event->{params});
                    my $sum = sprintf("%0.2f", $pay_data->{sum_payed})." ".format_currency($pay_data->{currency});
                    if ($msg_type =~ m/_wallet$/) {
                        $msg = iget($push_messages{$msg_type}, $sum);
                    } else {
                        $msg = iget($push_messages{$msg_type}, $text_descrs->{$first_event->{id}}{descr}, $sum);
                    }
                } else {
                    my $msg_new_type = $msg_type;
                	if ($msg_type eq 'money_warning_wallet') {
                		my $notification_data = JSON::from_json($first_event->{params});
                		if (defined $notification_data->{days_left}
								&& $CHANGE_NOTIFICATION_MESSAGES_FOR_DAYS_WARNING_WALLET->get(60)) {
							my $days_param = $notification_data->{days_left};
							if ($days_param eq 'ONE_DAY_REMAIN') {
								$msg_new_type = 'money_warning_one_day_remain_wallet';
							} elsif ($days_param eq 'THREE_DAYS_REMAIN') {
								$msg_new_type = 'money_warning_three_days_remain_wallet';
							}
						}
                	}
                    $msg = iget($push_messages{$msg_new_type}, $text_descrs->{$first_event->{id}}{descr});
                }
            } elsif (is_event_for_object($msg_type, 'phrase')) {
                $msg = iget($push_messages{$msg_type}, $text_descrs->{$first_event->{id}}{descr});
                $payload = {'P' => $first_event->{bids_id}};
            } elsif ($msg_type =~ m/^(banner_moderated_accepted|banner_moderated_declined|banner_moderated_declined_partly)$/) {
                $msg = iget($push_messages{$msg_type}, $text_descrs->{$first_event->{id}}{descr});
                $payload = {'B' => $first_event->{bid}};
                $payload->{'T'} = $EventLog::EVENTS{'banner_moderated'}{type};
            }
            $payload->{'T'} //= $EventLog::EVENTS{$msg_type}{type};
            if ($payload->{'T'} == $EventLog::EVENTS{'banner_moderated'}->{type}) {
                _enrich_payload($payload, 'banner_moderated', $first_event);
            } else {
                _enrich_payload($payload, $msg_type, $first_event);
            }

		} elsif ($event_types_num == 1 && ($first_event_obj ne 'campaign' || $camps_num > 1)) {
			my $msg_type = $event_types[0]."_multi";
			if ($msg_type =~ m/^(money_out_multi|money_warning_multi|camp_finished_multi|paused_by_day_budget_multi|money_in_multi)$/ ) {
				$msg = iget( get_word_for_digit($camps_num, @{$push_messages_multi{$msg_type}}), $camps_num);
			} elsif ($msg_type =~ m/^(warn_place_multi|banner_moderated_accepted_multi|banner_moderated_declined_multi|banner_moderated_declined_partly_multi)$/) {
				$msg = iget( get_word_for_digit($events_num, @{$push_messages_multi{$msg_type}}), $events_num);
			}
			if ($event_types[0] !~ m/banner_moderated/) {
				$payload->{'T'} = $EventLog::EVENTS{$event_types[0]}{type};
			} else {
				$payload->{'T'} = $EventLog::EVENTS{'banner_moderated'}{type};
			}
            if ($msg_type =~ m/^(banner_moderated_accepted_multi|banner_moderated_declined_multi|banner_moderated_declined_partly_multi)$/) {
                # добавим диплинк на кампанию (или на список кампаний, если баннеры более чем в одной кампании)
                if ($camps_num > 1) {
                    _enrich_payload($payload, 'banner_moderated_multicamp', $first_event);
                } else {
                    _enrich_payload($payload, 'banner_moderated_multi', $first_event);
                }
            }
		} elsif ($camps_num == 1) {

			if (exists $wallet_cids->{$first_event->{cid}}) {
				$msg = iget( get_word_for_digit($events_num, @{$push_messages_multi{campaign_multi_wallet}}), $events_num);

				$payload = {'A' => $first_event->{cid} };
			}else{
				$msg = iget( get_word_for_digit($events_num, @{$push_messages_multi{campaign_multi}}), $text_descrs->{$first_event->{id}}{camp_descr}, $events_num);
				$payload = {'C' => $first_event->{cid} };
			}
		} else {
			$msg = iget( get_word_for_digit($events_num, @{$push_messages_multi{many_camps}}), $events_num);
		}

        push @{$pushes_compiled{messages}}, {
            uid => $uid,
            message => $msg,
            payload => $payload,
            @subscription_line,
        };
        push @{$pushes_compiled{notification_ids}}, $notification_id;
    }
    return \%pushes_compiled;
}
