#!/usr/bin/perl

use my_inc "..";

=pod

=encoding utf-8

=head1 METADATA

<crontab>
    time: * * * * *
    sharded: 1
    <switchman>
        group: scripts-other
        <leases>
            mem: 70
        </leases>
    </switchman>
    package: scripts-switchman
</crontab>
<juggler>
    host:   checks_auto.direct.yandex.ru
    sharded: 1
    ttl: 20m
    tag: direct_group_internal_systems
</juggler>
<monrun>
    juggler_host:   checks_auto.direct.yandex.ru
    name: push.register_queue_length
    warn: 70
    crit: 120
    sharded: 1
    expression: 'movingAverage(direct_one_min.db_configurations.production.flow.push_token.register_queue_length.shard_$shard,"5min")'
    tag: direct_group_internal_systems
</monrun>
<monrun>
    juggler_host:   checks_auto.direct.yandex.ru
    name: push.register_queue_errors
    crit: 2
    sharded: 1
    expression: 'direct_one_min.db_configurations.production.flow.push_token.register_queue_errors.shard_$shard'
    tag: direct_group_internal_systems
</monrun>

<crontab>
    time: */10 * * * *
    sharded: 1
    flock: 1
    <switchman>
        group: scripts-test
    </switchman>
    package: conf-test-scripts
</crontab>

$Id$

=head1 NAME

ppcPushNotificationsQueue.pl - регистрация и удаление push-токенов

=head1 SYNOPSIS

    ppcPushNotificationsQueue.pl --shard-id 2

=head1 DESCRIPTION

    Скрипт для регистрации и удаления push-токенов

=cut

use warnings;
use strict;

use JSON;

use Yandex::Advmon;
use Yandex::DBShards;
use Yandex::DBTools;
use Yandex::ListUtils qw/xminus nsort/;
use Yandex::Xiva;

use Settings;
use ScriptHelper 'get_file_lock' => ['dont_die'], 'Yandex::Log' => 'messages', sharded => 1;

use HashingTools qw//;
use LockTools;

$Yandex::Advmon::GRAPHITE_PREFIX = sub {[qw/direct_one_min db_configurations/, $Settings::CONFIGURATION]};

my $TIMEOUT = 30; #seconds
my $LIMIT = 50;
my $DEFAULT_PUSH_TIMEOUT = 10;
my $DEFAULT_PUSH_LOCALE = 'ru';

my $ONCE;
extract_script_params(
    once => \$ONCE,
);

$log->out("Start daemon ...");

=head2 %subscription_type2platform

Соответствие типа подписки из API платформе в Xiva

=cut

my %subscription_type2platform = (
    APNS => 'apns',
    GCM => 'gcm'
);

while (1) {
    restart_tracing();
    if (my $reason = smart_check_stop_file()) {
        $log->out("$reason! Let's finish.");
        last;
    }

    my $cnt = get_one_field_sql(PPC(shard => $SHARD), 'select count(*) from users_notifications_register_queue');
    my $actions = get_all_sql(PPC(shard => $SHARD), "select  id
                                                            ,type
                                                            ,data
                                                            ,action
                                                            ,uid
                                                            ,resource
                                                    from users_notifications_register_queue
                                                    order by id limit $LIMIT");
	my @action_ids_to_delete;

	my $errors = 0;

	if ($actions && scalar @$actions) {
		foreach my $action (@$actions) {
            my $log_data = {%$action};
            if ($log_data->{resource}) {
                $log_data->{resource} = HashingTools::push_token_hash($log_data->{resource});
            }
			$log->out("Try", $log_data);
			if ($action->{action} eq 'add') {
                my $platform = $subscription_type2platform{$action->{type}};
                if (!$platform) {
                    $log->out("Unknown platform");
                    push @action_ids_to_delete, $action->{id};
                    next;
                }
                my $add_data = from_json($action->{data});
                my $device_id = $add_data->{DeviceId};
                my $tag = $add_data->{Tag};

                my $notification_id = _deduplicate_notification_ids(
                    uid => $action->{uid},
                    type => $action->{type},
                    tag => $tag,
                    device_id => $device_id,
                );

                my @subscription_line = $device_id ? (device_id => $device_id) : (tag => $tag);
                my $return_code = xiva_subscribe(
                    uid        => $action->{uid},
                    client     => $add_data->{Options},
                    uuid       => $add_data->{Uuid},
                    platform   => $platform,
                    push_token => $action->{resource},
                    @subscription_line,
                );

                if ($return_code != 200) {
                    $errors++;
                    if ($return_code == 0 || $return_code >= 500) { # что-то не то в коммуникации или на той стороне
                        $log->out("Xiva error. Sleep");
                        last;
                    }
                    if ($return_code == 429) {
                        $log->out("Xiva asked to slow down. Sleep");
                        last;
                    }
                    # все неучтённые выше проблемы
                    do_delete_from_table(PPC(shard => $SHARD), 'users_notifications_register_queue', where => {id => $action->{id}});
                    $log->out("Xiva call yielded a $return_code return code. Removed from queue");
                    next;
                }

                my $data = {
                    uid => $action->{uid}
                    , timeout => defined $add_data->{Timeout} ? $add_data->{Timeout} : $DEFAULT_PUSH_TIMEOUT
                    , locale => $add_data->{Locale} || $DEFAULT_PUSH_LOCALE
                    , options => $add_data->{Options}
                    , type => $action->{type}
                };
                if ($device_id) {
                    $data->{device_id} = $device_id;
                    $data->{tag} = undef;
                } else {
                    $data->{device_id} = undef;
                    $data->{tag} = $tag;
                }

                if ($notification_id) {
                    $data->{last_change__dont_quote} = 'NOW()';
                    do_update_table(PPC(shard => $SHARD), 'users_notifications', $data, where => {notification_id => $notification_id});
                } else {
                    $notification_id = $data->{notification_id} = get_new_id('users_notification_id');
                    do_insert_into_table(PPC(shard => $SHARD), 'users_notifications', $data);
                }

                my @event_types = @{$add_data->{EventTypes} || []};

                my $old_events = get_one_column_sql(PPC(shard => $SHARD), ['select event_type from users_notifications_details',
                                                    where => {notification_id => $notification_id}]) || [];

                my $events_to_insert = xminus \@event_types, $old_events;

                my @detail_update;

                foreach my $event (@{ $events_to_insert || [] }) {
                    push @detail_update, [$notification_id, $event];
                }

                if (scalar @detail_update) {
                    do_mass_insert_sql(PPC(shard => $SHARD), "insert into users_notifications_details(`notification_id`, `event_type`) values %s", \@detail_update);
                }

                do_delete_from_table(PPC(shard => $SHARD), 'users_notifications_details', where => { notification_id => $notification_id, event_type__not_in =>\@event_types });

                $log->out("Success");
                push @action_ids_to_delete, $action->{id};
			} elsif ($action->{action} eq 'remove') {

                my $data      = from_json($action->{data});
                my $uuid      = $data->{Uuid};
                my $tag       = $data->{Tag};
                my $device_id = $data->{DeviceId};

                my $notification_id = _deduplicate_notification_ids(
                    uid => $action->{uid},
                    type => $action->{type},
                    tag => $tag,
                    device_id => $device_id,
                );

                if ($uuid && $notification_id) {
                    my $return_code = xiva_unsubscribe(
                        uid  => $action->{uid},
                        uuid => $uuid
                    );
                    if ($return_code != 200) {
                        $errors++;
                        if ($return_code == 0 || $return_code >= 500) { # что-то не то в коммуникации или на той стороне
                            $log->out("Xiva error. Sleep");
                            last;
                        }
                        if ($return_code == 429) {
                            $log->out("Xiva asked to slow down. Sleep");
                            last;
                        }
                        # все неучтённые выше проблемы
                        do_delete_from_table(PPC(shard => $SHARD), 'users_notifications_register_queue', where => {id => $action->{id}});
                        $log->out("Xiva call yielded a $return_code return code. Removed from queue");
                        next;
                    }

                    _clear_notification_ids([$notification_id]);
                    $log->out("Success");
                }

			    push @action_ids_to_delete, $action->{id};
			}
		}
	}

	if (scalar @action_ids_to_delete) {
		do_delete_from_table(PPC(shard => $SHARD), 'users_notifications_register_queue', where => {id => \@action_ids_to_delete});
	}

	# пишем данные для нового мониторинга
    monitor_values({
        "flow.push_token.register_queue_length.shard_$SHARD" => $cnt,
        "flow.push_token.register_queue_errors.shard_$SHARD" => $errors,
    });

    juggler_ok();
    last if $ONCE;

    $log->out("sleep for $TIMEOUT");
	sleep $TIMEOUT;
}

$log->out("Stop daemon ...");


=head2 _deduplicate_notification_ids

На входе именованные параметры: uid, type, (device_id, и/или tag).
Находим все записи с совпадающими uid, type, (device_id, и/или tag), удаляем все, кроме самой последней (по notification_id),
На выходе notification_id оставшейся записи или undef, если её нет.

=cut

sub _deduplicate_notification_ids {
    my %O = @_;
    my @conditions;
    @conditions = (device_id => $O{device_id}) if $O{device_id};
    push @conditions, (tag => $O{tag}) if $O{tag};
    if (!@conditions) {
        $log->out(["Wrong _deduplicate_notification_ids invocation", \@_]);
        return undef;
    }

    my $notification_data = get_one_column_sql(PPC(shard => $SHARD), [
        'select notification_id from users_notifications',
        where => {
            uid => $O{uid},
            type => $O{type},
            _OR => \@conditions,
        }
    ]);

    my $ret_id;
    if ($notification_data && @$notification_data) {
        my @notification_ids = nsort @$notification_data;
        $ret_id = pop @notification_ids;

        _clear_notification_ids(\@notification_ids);
    }

    return $ret_id;
}

=head2 _clear_notification_ids

Удалить из базы все упоминания переданных notification_id

=cut

sub _clear_notification_ids {
    my $notification_ids = shift;
    if (@$notification_ids) {
        do_delete_from_table(PPC(shard => $SHARD), 'users_notifications_details', where => {notification_id => $notification_ids});
        do_delete_from_table(PPC(shard => $SHARD), 'users_notifications', where => {notification_id => $notification_ids});
        do_delete_from_table(PPC(shard => $SHARD), 'push_notifications_process', where => {notification_id => $notification_ids});
    }
}
