#!/usr/bin/perl

use my_inc "..";

=head1 METADATA

<crontab>
    time: */5 * * * *
    <switchman>
        group: scripts-other
    </switchman>
    package: scripts-switchman
</crontab>

<juggler>
    host:   checks_auto.direct.yandex.ru
    ttl:        30m
    tag: direct_group_internal_systems
</juggler>


=cut

=head1 DESCRIPTION

Скрипт для добавления в очередь на переотправку в БК кампаний с relevance_match

=cut

use Direct::Modern;

use Yandex::DBQueue;
use Yandex::DBShards;
use Yandex::DBTools;
use Yandex::HashUtils qw/hash_cut/;
use Yandex::Retry qw/relaxed_guard/;
use Yandex::I18n;
use Try::Tiny qw/try catch/;

use Settings;
use ScriptHelper;
use ShardingTools qw/ppc_shards/;
use Yandex::DBTools qw/get_all_sql get_one_column_sql get_hash_sql/;
use Client qw//;
use Campaign qw//;
use BS::ResyncQueue;


my $TRYCOUNT = 5;
my $SLEEP_COEF = 1;
my $MAX_ITERATIONS = 1;
my $CHUNK_SIZE = 500;

$log->out("START");

$Yandex::DBQueue::LOG = $log;
my $queue = Yandex::DBQueue->new(PPCDICT, 'bs_resync_relevance_match');

foreach my $iteration (1 .. $MAX_ITERATIONS) {
    my $rg = relaxed_guard times => $SLEEP_COEF;

    if (my $job = $queue->grab_job()) {

        if ($job->trycount > $TRYCOUNT) {
            $job->mark_failed_permanently({ error => 'Unknown error' });
            next;
        }

        my $resync_all_clients = $job->args->{resync_all_clients} ? 1 : 0;
        my @client_ids = $resync_all_clients ? () : @{ $job->args->{client_ids} // [] };

        unless ($resync_all_clients || @client_ids) {
            $job->mark_failed_permanently({ error => 'Clients not found' });
            $log->out('Clients not found');
            next;
        }
 
        eval {
            process_job($resync_all_clients, @client_ids);
        };
        if ($@) {
            $log->out("job failed once: $@");
            $job->mark_failed_once();
        } else {
            $job->mark_finished({});
        }
    }
}

sub process_job {
    my ($resync_all_clients, @client_ids) = @_;

    for my $shard (ppc_shards()) {
        $log->msg_prefix("[shard_$shard]");

        my $LIMIT = $CHUNK_SIZE;
        my $last_client_id = 0;
        my $data_by_client_id = {};
        while(1) {
            $log->out("processing $LIMIT entries with client_id > $last_client_id");

            my $clients = get_all_sql(PPC(shard => $shard),
                    [ qq/SELECT 
                            c.ClientID,
                            FIND_IN_SET('feature_context_relevance_match_allowed', co.client_flags) > 0 as has_a,
                            FIND_IN_SET('feature_context_relevance_match_interface_only', co.client_flags) > 0 as has_b,
                            isNull(co.ClientID) as has_not_clients_options_record
                        FROM clients c LEFT JOIN clients_options co ON (c.ClientID = co.ClientID)/,
                        WHERE => {
                            'c.ClientID__gt' => $last_client_id,
                            (@client_ids ? ('c.ClientID' => \@client_ids) : ())
                        },
                        'ORDER BY' => 'c.ClientID',
                        'LIMIT' => $CHUNK_SIZE
                    ]);
            last unless @$clients;
         
            $last_client_id = $clients->[-1]->{ClientID};
            my $new_feature_values;

            my %old_feature_state = map {$_->{ClientID} => hash_cut($_, qw/has_a has_b/)} @$clients;
            my @clients_without_clients_options_record = map {$_->{ClientID}} grep {$_->{has_not_clients_options_record}} @$clients;
            undef $clients;
            try {
                # находим клиентов, у которых изменилось значение фичи context_relevance_match_allowed
                # текущее значение флагов context_relevance_match_allowed и context_relevance_match_interface_only получаем из java intapi
                # _is_feature_allowed_for_client_ids берет из intapi все фичи клиентов и кеширует результат, поэтому можно вытащить фичи последовательно

                my $current_feature_a_state = Client::ClientFeatures::_is_feature_allowed_for_client_ids([keys %old_feature_state], 'context_relevance_match_allowed');
                my $current_feature_b_state = Client::ClientFeatures::_is_feature_allowed_for_client_ids([keys %old_feature_state], 'context_relevance_match_interface_only');

                foreach my $client_id (keys %old_feature_state){
                    $current_feature_a_state->{$client_id} ||= 0;
                    $current_feature_b_state->{$client_id} ||= 0;
                    #Если значения обоих фич клиента не изменились, либо у клиента не изменилась включенная context_relevance_match_allowed - ничего обновлять не нужно
                    next if ( 
                                ($old_feature_state{$client_id}->{has_a} || 0) == $current_feature_a_state->{$client_id} 
                                && ($old_feature_state{$client_id}->{has_b} || 0) == $current_feature_b_state->{$client_id}
                                    ||
                                $old_feature_state{$client_id}->{has_a} && $current_feature_a_state->{$client_id}
                           );
                    $new_feature_values->{$client_id} = {
                                    flags => [$current_feature_a_state->{$client_id}, $current_feature_b_state->{$client_id}],
                                    val  => $current_feature_a_state->{$client_id} + $current_feature_b_state->{$client_id}
                    };
                    $log->out(sprintf('feature values changed for client %s: ctx_rm_allowed  %s -> %s, ctx_rm_interface_only %s -> %s', $client_id,
                                ($old_feature_state{$client_id}->{has_a} || 0), $new_feature_values->{$client_id}->{flags}->[0],
                                ($old_feature_state{$client_id}->{has_b} || 0), $new_feature_values->{$client_id}->{flags}->[1])
                    );
                }
                
                if (keys %$new_feature_values){
                    _update_clients_options($shard, $new_feature_values, \@clients_without_clients_options_record);
                    _update_campaigns_for_clients($shard, $new_feature_values);
                }
            }
            catch {
                my $error = shift;
                $log->out("Got error: ".$error);
                if ($resync_all_clients) {
                    my $job = Yandex::DBQueue->new(PPCDICT, 'bs_resync_relevance_match')->insert_job({
                            job_id => get_new_id('job_id'),
                            ClientID => 0,
                            args => {client_ids => [keys %old_feature_state]},
                    });
                    $log->out("New job added into DBQueue: ".$job->job_id);
                } else {
                    die $error;
                }
            };
        }

        $log->msg_prefix("");
    }
}

sub _update_campaigns_for_clients {
    my ($shard, $new_feature_values) = @_;

    my $campaigns = get_all_sql(PPC(shard => $shard), [
            qq/SELECT DISTINCT c.cid, c.ClientID, c.OrderID
                FROM campaigns c
                JOIN bids_base b ON (c.cid = b.cid)
            /,
            WHERE => {
                'c.archived' => 'No',
                'c.ClientID' => [keys %$new_feature_values],
                'b.bid_type' => [qw/relevance_match_search relevance_match/],
                _TEXT => "NOT FIND_IN_SET('suspended', b.opts) AND NOT FIND_IN_SET('deleted', b.opts)"
            }]);

    #С учетом стратегии выставим ставки на сеть всем кампаниям, для которых включена фича
    my $cids_to_update_prices = [map {$_->{cid}} grep {$new_feature_values->{$_->{ClientID}}->{val} > 0} @$campaigns];
    update_prices($shard, $cids_to_update_prices);
    
    #Переотправляем в БК все кампании, которые имеют OrderID
    my $cids_to_bs_resync = [map {$_->{cid}} grep {$_->{OrderID} > 0} @$campaigns];
    $log->out(sprintf('BS resync for cids: [%s]', join(', ', @$cids_to_bs_resync)));
    BS::ResyncQueue::bs_resync_camps($cids_to_bs_resync, priority => BS::ResyncQueue::PRIORITY_INTERNAL_REPORTS_LAZY_RESYNC);
    
    return;
}
    
sub _update_clients_options {
    my ($shard, $new_feature_values, $clients_without_clients_options_record) = @_;

    my @client_ids_for_insert = map {[$_]} grep {exists $new_feature_values->{$_}} @$clients_without_clients_options_record;
    if (@client_ids_for_insert){
        do_mass_insert_sql(PPC(shard => $shard), 'insert ignore into clients_options (ClientID) values %s',
                \@client_ids_for_insert)
    }

    my %case_values = map {
        $_ => Yandex::DBTools::sql_set_mod('client_flags', {
                feature_context_relevance_match_allowed => $new_feature_values->{$_}->{flags}->[0],
                feature_context_relevance_match_interface_only => $new_feature_values->{$_}->{flags}->[1],
        })
    } keys %$new_feature_values;
    do_update_table(PPC(shard => $shard), 'clients_options', {
            client_flags => sql_case(ClientID => \%case_values, default__dont_quote => 'client_flags', dont_quote_value => 1)
        },
        where => {ClientID => [keys %$new_feature_values]},
        dont_quote => ['client_flags']
    ) if %case_values;

    $log->out(sprintf('feature_context_relevance_match_allowed, feature_context_relevance_match_interface_only updated for clients: %s',
                join(', ', map {$_.': ['.$new_feature_values->{$_}->{flags}->[0].', '.$new_feature_values->{$_}->{flags}->[1].']'} keys %$new_feature_values)));
}


sub update_prices {
    my ($shard, $cids) = @_;
    return unless @$cids;
    $log->out("update_prices: campaigns count: " . scalar(@$cids));

    my $camps_strategies = Campaign::mass_campaign_strategy($cids);

    foreach my $cid (keys %$camps_strategies) {
        # для обновления price_context используем _relevance_match_bids_resync_and_fix
        # поэтому будем вызывать ее только в том случае когда она будет обновлять price_context
        # то есть когда has_extended_relevance_match = 1 (включенность фичи проверяется выше) 
        #             и is_ctx_price_correction_allowed = 1  

        my $is_ctx_price_correction_allowed = Campaign::is_ctx_price_correction_allowed($camps_strategies->{$cid});
        $log->out(sprintf('Campaign: %s, ctx_price_correction_allowed: %s', $cid, $is_ctx_price_correction_allowed));
        if ($is_ctx_price_correction_allowed) {
            #TODO get_camp_info можно брать массово
            my $camp = Campaign::get_camp_info($cid);
            my $is_cpm_campaign = Campaign::is_cpm_campaign($camp->{mediaType});
            my $currency = $camp->{currency};
            Campaign::_relevance_match_bids_resync_and_fix($cid, $currency, $is_cpm_campaign, 1, 1, $camp->{ContextPriceCoef}, rewrite_price_context => 1);
            $log->out(sprintf('Campaign: %s - prices updated', $cid));
        }
        else {
            #Если ставка на сеть для автотаргетинга недоступна - сбросим ее на 0
            my $updated = do_update_table(PPC(shard => $shard), 'bids_base', {price_context => 0},
                where => {cid => $cid, price_context__gt => 0, bid_type => 'relevance_match'},
            );
            $log->out("purging price_context for cid:$cid, $updated bids purged");
        }
    }
}

$queue->delete_old_jobs(24 * 60 * 60);

juggler_ok();

$log->out("FINISH");
