#!/usr/bin/perl

use my_inc "..";

=encoding utf8

=head1 METADATA

<crontab>
    time: */59 2-22/2 * * *
    sharded: 1
    package: scripts-switchman
    <switchman>
        group:  scripts-other
        <leases>
            mem: 300
        </leases>
    </switchman>
</crontab>

<juggler>
    host:   checks_auto.direct.yandex.ru
    sharded: 1
    ttl: 6h
    tag: direct_group_internal_systems
</juggler>

=cut

=head1 DESCRIPTION

    Проверка живости целей метрики для ретаргетинга
    https://st.yandex-team.ru/DIRECT-15193

    Запускается каждые несколько часов, но реально работает только раз в сутки.

    Параметры:
    --shard-id=X        - номер шарда, клиенты которого будут обработаны
    --clientid=YYYYY    - обработать только указанных клиентов (из шарда X) (можно указывать несколько раз)

=cut

use Direct::Modern;

use List::MoreUtils qw/any/;

use Yandex::Advmon;
use Yandex::DBShards;
use Yandex::DBTools;
use Yandex::HashUtils qw/hash_merge/;
use Yandex::ListUtils qw/chunks nsort/;
use Yandex::TimeCommon qw/today/;

use Settings;
use ScriptHelper sharded => 1;

use BS::ResyncQueue;
use JavaIntapi::GetClientsWithExplicitFeature;
use Monitor;
use Notification;
use Property;
use RBAC2::Extended;
use RBACElementary;
use Retargeting;
use ShardingTools;

=head1 SUBROUTINES/METHODS/VARIABLES

=cut

# за сколько приемов проверяем все цели (разбиваем по ClientID клиента)
our $MAX_CHUNKS = 10;

# --------------------------------------------------------------------

#Поле name в таблице ppclog.monitor_targets
my $TARGET_NAME_FOR_GOALS_NOT_ACCESSIBLE = "$Retargeting::BASE_TARGET_NAME_FOR_GOALS_NOT_ACCESSIBLE" . $SHARD;

my @debug_client_id;
extract_script_params(
    'clientid=i@' => \@debug_client_id,
);

# Клиенты и агентства, которым можно таргетироваться на чужие цели
# здесь должны быть только внутренние клиенты, заводящие кампании BrandLift
# Хранятся в виде хешика, чтобы легче проверять наличие ClientID в этом наборе.
my $FULL_ACCESS_INTERNAL_CLIENTS = _find_full_access_internal_clients();

$log->out('START');
my $prop_name = "RETARGETING_CHECK_GOALS_SHARD_$SHARD";
my $prop = Property->new($prop_name);
my $today = today();

# валидируем шард
unless ($SHARD && any { $_ eq $SHARD } ppc_shards()) {
    $log->die('shard-id is not specified or has invalid value');
}

my $rbac;
my $skip_work_today;

if (@debug_client_id) {
    $log->out('Processing only ClientIDs: ' . join (', ', @debug_client_id));
    $MAX_CHUNKS = 1;
} elsif (($prop->get() // '') eq $today) {
    $skip_work_today = 'Already worked today';
    $log->out($skip_work_today);
    juggler_ok(description => $skip_work_today);
}

local $Yandex::Advmon::GRAPHITE_PREFIX = sub {[qw/direct_one_min db_configurations/, $Settings::CONFIGURATION]};
my $goals_made_not_accessible_count = 0;
my $goals_made_accessible_count = 0;

for my $CHUNK_NO (0 .. $MAX_CHUNKS - 1) {
    last if $skip_work_today;

    my $msg_prefix_guard = $log->msg_prefix_guard(sprintf('[chunk: %d / %d]', ($CHUNK_NO + 1), $MAX_CHUNKS));
    $log->out("start processing chunk");
    _get_rbac_singleton_or_die();

    # выбираем все цели c ClientID и типами  Яндекс.Метрики
    my %where_cond = (
        'rc.is_deleted' => 0,
        'rc.properties__scheck' => {interest => 0},
        'rg.goal_type' => ['goal', 'audience', 'segment', 'ab_segment']
    );
    if (@debug_client_id) {
        $where_cond{'rc.ClientID'} = \@debug_client_id;
    } elsif ($MAX_CHUNKS > 1) {
        $where_cond{_TEXT} = "rc.ClientID mod $MAX_CHUNKS = $CHUNK_NO";
    }
    my $all_clients_goals = get_all_sql(PPC(shard => $SHARD), [
                                            'SELECT rc.ClientID, rc.condition_name, rg.ret_cond_id, rg.goal_id, rg.is_accessible, rg.goal_type',
                                            'FROM retargeting_conditions rc',
                                            'JOIN retargeting_goals rg USING(ret_cond_id)',
                                            WHERE => \%where_cond,
                                        ]);

    next unless @$all_clients_goals;

    my %goals_by_clients;
    for my $row (@$all_clients_goals) {
        push @{$goals_by_clients{$row->{ClientID}}}, $row;
    }

    my $chiefs_uids = rbac_get_chief_reps_of_clients([keys %goals_by_clients]);

    for my $ClientID (keys %goals_by_clients) {
        my ($goals_made_not_accessible, $goals_made_accessible) = _check_for_one_clientid($ClientID, $chiefs_uids->{$ClientID}, $goals_by_clients{$ClientID});
        $goals_made_not_accessible_count += $goals_made_not_accessible;
        $goals_made_accessible_count += $goals_made_accessible;
    }
}

monitor_values({
    'flow.'.$TARGET_NAME_FOR_GOALS_NOT_ACCESSIBLE => $goals_made_not_accessible_count,
    'flow.ppcRetargetingCheckGoals.goals_made_accessible_shard_'.$SHARD => $goals_made_accessible_count,
});

unless (@debug_client_id || $skip_work_today) {
    $log->out("setup property '$prop_name' value: $today");
    $prop->set($today);
    juggler_ok();
}

$log->out('FINISH');

exit;

=head3 _get_rbac_singleton_or_die()

    for my $chunk (...) {
        _get_rbac_singleton_or_die();
        ... # do something with $chunk and $rbac object
    }

    Получение объекта RBAC2::Extended (и запись в глобальную $rbac).
    Умирает с записью в лог (глобальную $log), если не получилось.

=cut
sub _get_rbac_singleton_or_die {
    $rbac = eval { RBAC2::Extended->get_singleton(1) } or $log->die("Can't connect to RBAC: $@");
}

sub _check_for_one_clientid {
    # используем глобальные объекты $log, $rbac, $SHARD
    my ($ClientID, $chief_uid, $goals) = @_;

    my $is_internal_ad_product = _is_internal_ad_product($ClientID);

    my $client_uids = rbac_get_client_uids_by_clientid($ClientID);

    my %goal_id_to_goals;
    for my $goal (@$goals) {
        push @{$goal_id_to_goals{$goal->{goal_id}} ||= []}, $goal;
    }
    my $hidden_brand_lift_ret_conditions = _find_hidden_brand_lift_ret_conditions($ClientID);

    my %cond_for_client_vars;
    my %changed_ret_cond_id;
    my $goals_made_not_accessible = 0;
    my $goals_made_accessible = 0;

    for my $chunk (chunks([keys %goal_id_to_goals], 100)) {
        my $actual_goals = {
            map {$_ => {goal_id => $_}}
            map {@$_}
                values %{Retargeting::check_metrika_goals_access_by_uid($client_uids, $chunk, log => $log, timeout => 30, retry_fails => 1)}
        };
        $log->out({
            ClientID => $ClientID,
            chief_uid => $chief_uid,
            actual_goals => $actual_goals,
            hidden_brand_lift_ret_conditions => [keys $hidden_brand_lift_ret_conditions]
        });
        for my $goal_id (@$chunk) {
            for my $row (@{$goal_id_to_goals{$goal_id}}) {
                $row->{goal_type} //= "";

                my $do_make_accessible = 0;
                if (!exists $actual_goals->{ $row->{goal_id} } && $is_internal_ad_product) {
                    # для внутренней рекламы нет ограничений на видимость целей
                    $log->out("Allowing full access internal ad client $ClientID" .
                        " to use inaccessible goal_id: $row->{goal_id}, ret_cond_id: $row->{ret_cond_id}"
                    );

                    if (!$row->{is_accessible}) {
                        $do_make_accessible = 1;
                    }
                }
                elsif (!exists $actual_goals->{ $row->{goal_id} } && $FULL_ACCESS_INTERNAL_CLIENTS->{$ClientID}) {

                    my $cl_info = $FULL_ACCESS_INTERNAL_CLIENTS->{$ClientID};
                    $log->out("Allowing full access client $ClientID" .
                        " with login $cl_info->{login}, role $cl_info->{role}" .
                        " to use inaccessible goal_id: $row->{goal_id}, ret_cond_id: $row->{ret_cond_id}"
                    );

                    if (!$row->{is_accessible}) {
                        $do_make_accessible = 1;
                    }
                }
                elsif (!exists $actual_goals->{ $row->{goal_id} } && exists $hidden_brand_lift_ret_conditions->{$row->{ret_cond_id}}) {
                    $log->out("Allowing access client $ClientID" .
                        " to use inaccessible goal from hidden Brand Lift retargeting condition -" .
                        " goal_id: $row->{goal_id}, ret_cond_id: $row->{ret_cond_id}"
                    );
                    if (!$row->{is_accessible}) {
                        $do_make_accessible = 1;
                    }
                }
                elsif ($row->{is_accessible} && !exists $actual_goals->{ $row->{goal_id} }) {

                    # если цель была доступна, а теперь нет, проставляем новый статус и отсылаем письмо
                    Retargeting::update_condition_goals_accessibility(
                        0, $row->{ret_cond_id}, $row->{goal_id}, $row->{goal_type}
                    );

                    $cond_for_client_vars{ $row->{ret_cond_id} } = {
                        ret_cond_id    => $row->{ret_cond_id},
                        condition_name => $row->{condition_name},
                    };
                    $changed_ret_cond_id{ $row->{ret_cond_id} } = undef;
                    $log->out("Now NOT accessible: goal_id: $row->{goal_id} ret_cond_id: $row->{ret_cond_id}");
                    ++$goals_made_not_accessible;

                }
                elsif (!$row->{is_accessible} && exists $actual_goals->{ $row->{goal_id} }) {

                    # если цель не была доступна, а теперь стала - просто проставляем статус, и тип цели
                    $do_make_accessible = 1;

                }
                else {
                    $log->out("Nothing change for: goal_id: $row->{goal_id} ret_cond_id: $row->{ret_cond_id}");
                }

                if ($do_make_accessible) {
                    Retargeting::update_condition_goals_accessibility(1, $row->{ret_cond_id}, $row->{goal_id});

                    $changed_ret_cond_id{ $row->{ret_cond_id} } = undef;
                    $log->out("Now accessible: goal_id: $row->{goal_id} ret_cond_id: $row->{ret_cond_id}");
                    ++$goals_made_accessible;
                }
            }
        }
    }

    # если были отключенные цели, то посылаем уведомление
    if (%cond_for_client_vars) {
        my $vars = {
            ClientID => $ClientID,
            uid => $chief_uid,
            retargetings => [values %cond_for_client_vars],
        };
        add_notification($rbac, 'retargeting_goals_check', $vars);
    }

    if (%changed_ret_cond_id) {
        # переотправляем в БК через ленивую очередь данные, использующие условия ретаргетинга в которых изменилась "доступность"
        my @resync_data;
        # хеш соответствий pid => cid
        my %resync_pids;

        my $changed_ids = [keys %changed_ret_cond_id];
        my $sth_bids = exec_sql(PPC(shard => $SHARD), [
                                    'SELECT p.cid, p.pid',
                                    'FROM bids_retargeting br',
                                    'JOIN phrases p ON p.pid = br.pid',
                                    WHERE => {
                                        'br.ret_cond_id__int' => $changed_ids,
                                    },
                                    'GROUP BY cid, pid',
                                ]);
        while (my ($cid, $pid) = $sth_bids->fetchrow_array()) {
            $resync_pids{$pid} //= $cid;
        }
        $sth_bids->finish();

        my $hmids = get_one_column_sql(PPC(shard => $SHARD), [
                                            'SELECT hierarchical_multiplier_id',
                                            'FROM retargeting_multiplier_values',
                                            WHERE => {
                                                ret_cond_id__int => $changed_ids,
                                            },
                                       ]);
        if (@$hmids) {
            for my $hmids_chunk (chunks($hmids, 10_000)) {
                my $sth_multipliers = exec_sql(PPC(shard => $SHARD), [
                                                   'SELECT cid, pid',
                                                   'FROM hierarchical_multipliers',
                                                   WHERE => {
                                                        hierarchical_multiplier_id__int => $hmids_chunk,
                                                        type => 'retargeting_multiplier',
                                                   },
                                               ]);
                while (my ($cid, $pid) = $sth_multipliers->fetchrow_array()) {
                    if ($pid) {
                        # группа
                        $resync_pids{$pid} //= $cid;
                    } else {
                        # кампания
                        push @resync_data, {
                            cid => $cid,
                            priority => BS::ResyncQueue::PRIORITY_ON_CHANGED_RETARGETING_ACCESSIBILITY,
                        };
                    }
                }
                $sth_multipliers->finish();
            }
        }

        # resync performance filters
        my $sth_perf_filters = exec_sql(PPC(shard => $SHARD), [
                                    'SELECT p.cid, p.pid
                                    FROM bids_performance bp
                                    JOIN phrases p USING(pid)',
                                    WHERE => {'bp.ret_cond_id__int' => $changed_ids},
                                    'GROUP BY p.cid, p.pid'
                                ]);
        while (my ($cid, $pid) = $sth_perf_filters->fetchrow_array()) {
            $resync_pids{$pid} //= $cid;
        }
        $sth_perf_filters->finish();

        for my $pid (keys(%resync_pids)) {
            push @resync_data, {
                pid => $pid,
                cid => $resync_pids{$pid},
                priority => BS::ResyncQueue::PRIORITY_ON_CHANGED_RETARGETING_ACCESSIBILITY,
            };
        }

        # останавливаем показывающиеся кампании, у которых нет доступа к аб-сегментам в таргетинге
        my @campaign_ids_to_stop;

        my $ab_segment_ret_conditions_to_stop = [grep { !exists $hidden_brand_lift_ret_conditions->{$_} } @$changed_ids];
        my $sth_camps_ab_segments = exec_sql(PPC(shard => $SHARD), [
                'SELECT c.cid
                FROM campaigns c',
                WHERE => {'c.statusShow' => 'Yes', 'c.ab_segment_ret_cond_id__int' => $ab_segment_ret_conditions_to_stop},
            ]);
        while (my ($cid) = $sth_camps_ab_segments->fetchrow_array()) {
            push @campaign_ids_to_stop, $cid;

        }
        $sth_camps_ab_segments->finish();

        %resync_pids = ();

        if ($goals_made_not_accessible > 0) {
            #Сохраняем кол-во недоступных целей в ppclog
            Monitor::accumulate_values({
                $TARGET_NAME_FOR_GOALS_NOT_ACCESSIBLE => {
                    value => $goals_made_not_accessible,
                    description => 'Количество недоступных целей',
                },
            });
        }

        if (@resync_data) {
            my $affected_rows_cnt = bs_resync(\@resync_data);
            $log->out(sprintf('Added %d rows to bs_resync_queue, affected %d rows', scalar(@resync_data), $affected_rows_cnt));
        }
        if (@campaign_ids_to_stop) {
            my %update_camp_hash = map {$_ => {
                statusShow             => 'No',
                statusBsSynced         => 'No',
                start_time => 'start_time',
                LastChange => 'now()' }} @campaign_ids_to_stop;
            do_mass_update_sql(PPC(shard => $SHARD), 'campaigns', 'cid', \%update_camp_hash,
                byfield_options =>  {start_time => {dont_quote_value => 1}, LastChange => {dont_quote_value => 1}});
            my %update_camp_options_hash = map {$_ => { stopTime => 'now()' }} @campaign_ids_to_stop;
            do_mass_update_sql(PPC(shard => $SHARD), 'camp_options', 'cid', \%update_camp_options_hash,
                byfield_options =>  {stopTime => {dont_quote_value => 1}});
        }
    }

    return ($goals_made_not_accessible, $goals_made_accessible);
}

=head2 _is_internal_ad_product($client_id)

    Проверяет, относится ли клиент ко внутренней рекламе (продукт вн. рекламы)

    Продукт внутренней рекламы, это клиент, в котором размещаются рекламные материалы для одного
    из продуктов внутренних сервисов. Например, "Поисковое приложение" или "Афиша"

=cut

sub _is_internal_ad_product {
    my ($client_id) = @_;
    my $perminfo = Rbac::get_perminfo( ClientID => $client_id );
    if ( Rbac::has_perm( $perminfo, 'internal_ad_product' ) ) {
        return 1;
    } else {
        return 0;
    }
}

=head2 _find_full_access_internal_clients

    Определяет список клиентов, которым разрешено таргетироваться на чужие цели по наличию фичи
    skip_goal_existence_for_agency
    Если фичу выставили на агентство, все клиенты этого агентства также добавляются в возвращаемый набор.

    Пишет о них информацию в лог и возвращает HashRef
    (
        ClientID1 => { login => 'login1', role => 'client' },
        ClientID2 => { login => 'login2', role => 'client of agency login3' },,
        ...
    )

=cut
sub _find_full_access_internal_clients {
    my $client_ids = JavaIntapi::GetClientsWithExplicitFeature->new(
            feature_name => 'skip_goal_existence_for_agency'
        )->call();

    if (!@$client_ids) {
        return;
    }

    my $clients_info = get_hashes_hash_sql(PPC(ClientID => $client_ids), ["
        SELECT cl.ClientID, cl.role, u.login
        FROM clients cl
            JOIN users u ON u.uid = cl.chief_uid",
        WHERE => [
            'cl.ClientID' => SHARD_IDS
        ]
    ]);

    my @agency_ids = map { $_->{ClientID} } grep { $_->{role} eq 'agency' } values %$clients_info;
    if (@agency_ids) {
        my $subclients_info = get_hashes_hash_sql(PPC(shard => 'all'), ["
            SELECT cl.ClientID, cl.agency_client_id, cl.role, u.login
            FROM clients cl
                JOIN users u ON u.uid = cl.chief_uid",
            WHERE => [
                'cl.agency_client_id' => \@agency_ids
            ]
        ]);
        for my $v (values %$subclients_info) {
            $v->{role} = 'client of agency '.$clients_info->{$v->{agency_client_id}}{login};
        }
        hash_merge($clients_info, $subclients_info);
    }

    @$client_ids = nsort keys %$clients_info;
    for my $client_id (@$client_ids) {
        my $cl = $clients_info->{$client_id};
        $log->out("Found client with full access: ClientID $cl->{ClientID}, role $cl->{role}, login $cl->{login}");
    }

    return $clients_info;
}

=head2 _find_hidden_brand_lift_ret_conditions

    Находим условия ретаргентинга клиента, которые предназначаются для скрытого проведения Brand Lift'а.
    Для таких условий не проверяем доступность целей, так как они заведены на других логинах.

=cut
sub _find_hidden_brand_lift_ret_conditions {
    my ($ClientID) = @_;

    return get_hash_sql(PPC(shard => $SHARD), [
        'SELECT ab_segment_ret_cond_id',
        'FROM campaigns',
        WHERE => {
            'ClientID' => $ClientID,
            'ab_segment_ret_cond_id__is_not_null' => 1,
            '_TEXT' => 'find_in_set("is_brand_lift_hidden", opts)'
        }
    ]);
}
