#!/usr/bin/perl

use my_inc "..";

=head1 METADATA

<crontab>
    time: */59 * * * *
    package: scripts-switchman
    <switchman>
        group:  scripts-other
        <leases>
            mem: 300
        </leases>
    </switchman>
</crontab>

<juggler>
    host:   checks_auto.direct.yandex.ru
    ttl: 3h
    tag: direct_group_internal_systems
</juggler>
<juggler_check>
    host:   checks_auto.direct.yandex.ru
    name:           direct.retargeting_goals.daily_not_accessible_count
    raw_events:     direct.retargeting_goals.daily_not_accessible_count.production
    ttl:            3h
    tag: direct_group_internal_systems
</juggler_check>
<juggler_check>
    host:   checks_auto.direct.yandex.ru
    name:           direct.retargeting_goals.weekly_not_accessible_count
    raw_events:     direct.retargeting_goals.weekly_not_accessible_count.production
    ttl:            3h
    tag: direct_group_internal_systems
</juggler_check>

=cut

=head1 DESCRIPTION

    Мониторинг количества недоступных целей метрики для ретаргетинга по всем шардам, ставших недоступными в течение дня и недели
    Juggler-проверки недоступных целей метрики делаются при каждом запуске скрипта

=cut

use Direct::Modern;

use Yandex::DBTools;

use Settings;
use ScriptHelper;
use Retargeting;


$log->out('START');

my $target_name = $Retargeting::BASE_TARGET_NAME_FOR_GOALS_NOT_ACCESSIBLE;
my $daily_goals_not_accessible_count = get_goals_not_accessible_count_by_last_day(1, $target_name);
$log->out("Total daily not accessible goals: $daily_goals_not_accessible_count");
my $weekly_goals_not_accessible_count = get_goals_not_accessible_count_by_last_day(7, $target_name);
$log->out("Total weekly not accessible goals: $weekly_goals_not_accessible_count");

juggler_check(service => 'direct.retargeting_goals.daily_not_accessible_count',
    description       => 'Количество ставших недоступными целей за день',
    value             => $daily_goals_not_accessible_count,
    warn              => 1_000,
    crit              => 10_000,
);
juggler_check(service => 'direct.retargeting_goals.weekly_not_accessible_count',
    description       => 'Количество ставших недоступными целей за неделю',
    value             => $weekly_goals_not_accessible_count,
    warn              => 5_000,
    crit              => 20_000,
);
juggler_ok();

$log->out('FINISH');


=head2 get_goals_not_accessible_count_by_last_day

    Возвращает количество недоступных целей по всем шардам за последние n дней
    На вход получает n - количество дней и название поля

=cut

sub get_goals_not_accessible_count_by_last_day {
    my ($day, $target_name) = @_;
    return get_one_field_sql(MONITOR, ['
        SELECT sum(value)
        FROM monitor_targets t
        JOIN monitor_values_sec s ON s.target_id=t.target_id',
        where => {'t.name__like' => $target_name.'%'},
        "AND DATEDIFF(NOW(), measure_time) < $day"
        ]) || 0;
}
