#!/usr/bin/perl

use my_inc "..";


=head1 METADATA

<crontab>
    params:
    time: */5 * * * *
    sharded: 1
    <switchman>
        group: scripts-other
        <leases>
            mem: 180
        </leases>
    </switchman>
    package: scripts-switchman
</crontab>
<crontab>
    params: --clear
    time: 0 4 * * *
    sharded: 1
    <switchman>
        group: scripts-other
    </switchman>
    package: scripts-switchman
</crontab>
<juggler>
    host:   checks_auto.direct.yandex.ru
    raw_events:     scripts.ppcSearchPhrasesDoubles.$mode.shard_$shard
    sharded:        1
    vars:           mode=searching,clearing<ttl=25h>
    ttl:            1h
    tag: direct_group_internal_systems
</juggler>

<crontab>
    time: */10 8-23 * * *
    <switchman>
        group: scripts-test
    </switchman>
    package: conf-test-scripts
    sharded: 1
    flock: 1
</crontab>

=head1 NAME

    ppcSearchPhrasesDoubles.pl

=head1 SYNOPSIS

    ppcSearchPhrasesDoubles.pl --shard-id 2

    ppcSearchPhrasesDoubles.pl --shard-id 2 --clear

=head1 DESCRIPTION

    Обрабатывает очередь заявок на проверку фраз на дубли с кампаниями

=head1 COMMENTS


=cut

use strict;
use warnings;

use Yandex::DBTools;

use Settings;
use ScriptHelper 'get_file_lock' => undef, sharded => 1, 'Yandex::Log' => 'messages';

use LockTools;
use PhraseDoubles;
use Tools;

# Количество заявок, которое в один присест выполнять.
my $CHUNK_COUNT = 5;
# Продолжительность жизни запроса (в месяцах)
my $REQUEST_LIFE_LENGTH = 1;
# Интервал между проверками на новые запросы.
my $SLEEP_TIMEOUT = 10;
# Количество попыток, которые позволяем сделать, а затем переводим в status=Failed
my $MAX_TRIES_COUNT = 3;

my $clear = 0;
extract_script_params(
    '--clear' => \$clear,
);

if ($clear) {
    get_file_lock(undef, get_script_name().'_clear');
    delete_old_requests();
    release_file_lock();
    juggler_ok(service => "scripts.ppcSearchPhrasesDoubles.clearing.shard_$SHARD");
    exit;
}

my $script_name = get_script_name();

get_file_lock('dont_die', $script_name);
$log->out("start searchDoubles");

while (1) {
    if (my $reason = smart_check_stop_file()) {
        $log->out("$reason! Let's finish.");
        last;
    }

    Yandex::Trace::restart(\$ScriptHelper::trace, tags => "shard_$SHARD");

    #Отбросить тяжелые заявки
    check_tries_counter();
    #Собственно поиск дублей.
    searchDoubles();

    juggler_ok(service => "scripts.ppcSearchPhrasesDoubles.searching.shard_$SHARD");
    sleep($SLEEP_TIMEOUT);
}
release_file_lock();
$log->out("finish searchDoubles");

exit;

=head2 searchDoubles

    Разбирает запросы на поиск дублей и выполняет их.

=cut
sub  searchDoubles {
    my $requests = get_all_sql(PPC(shard => $SHARD), "SELECT request_id, params FROM phrases_doubles_queue WHERE status='New' ORDER BY request_id LIMIT ?", $CHUNK_COUNT);
    foreach my $request (@$requests) {

        my $params = decode_json_and_uncompress($request->{params});
        $log->out(sprintf("start request processing: request_id: %d, cids: %s, words: %s", $request->{request_id}, (join ",", @{$params->{cids}}), (join ",", @{$params->{words}})));
        # увеличиваем счетчики попыток
        do_update_table(PPC(shard => $SHARD), "phrases_doubles_queue", {tries_counter__dont_quote=>'tries_counter+1'}, where=>{request_id => $request->{request_id}});

        eval {
            my $result = PhraseDoubles::search_phrase_doubles_for_request($params);

            # Записываем результат в БД
            do_update_table(PPC(shard => $SHARD), "phrases_doubles_queue", {result=>encode_json_and_compress($result), 
                                                           status => 'Done', 
                                                           done_time__dont_quote=>'NOW()'}, 
                                                   where=>{request_id => $request->{request_id}});
        };

        $log->out("There was a problem: $@") if ($@);

        $log->out("finish the request processing");

    }
}

=head2 check_tries_counter

    Переводит статус "плохих"(которые не смогли обработаться за $MAX_TRIES_COUNT раз) заявок в статус Failed

=cut
sub check_tries_counter {
    do_update_table(PPC(shard => $SHARD), "phrases_doubles_queue", {status => 'Failed'}, where => {status =>'New', tries_counter => $MAX_TRIES_COUNT});
}
=head2 delete_old_requests

	Удаляет все заявки на поиск дублей старше REQUEST_LIFE_LENGHT месяцев

=cut
sub delete_old_requests{
    $log->out("start all requests clearing");	

    my $count = do_delete_from_table(PPC(shard => $SHARD), 'phrases_doubles_queue', where => {create_time__lt__dont_quote => "DATE_SUB(NOW(), INTERVAL $REQUEST_LIFE_LENGTH MONTH)"});

    $log->out(sprintf("finish clearing: deleted %d requests", $count));	
}
