#!/usr/bin/perl

use my_inc "..";

=encoding utf-8

=head1 METADATA

<crontab>
    time: */8 * * * *
    ulimit: -v 28000000
    params: --report-type=dynamic
    <switchman>
        group:  scripts-other
        <leases>
            mem: 28000
        </leases>
    </switchman>
    package: scripts-switchman
</crontab>
<crontab>
   time: */8 * * * *
   ulimit: -v 20000000
   params: --report-type=performance
   <switchman>
       group:  scripts-other
       <leases>
           mem: 20000
       </leases>
   </switchman>
   package: scripts-switchman
</crontab>

<juggler>
    host:   checks_auto.direct.yandex.ru
    raw_events:     scripts.ppcSearchQueryStatus.working.$type
    vars:           type=dynamic<ttl=80m>
    vars:           type=performance<ttl=40m>
    tag: direct_group_internal_systems
</juggler>

=cut

=head1 NAME

ppcSearchQueryStatus.pl - обработчик очереди api_queue_search_query

=head1 SYNOPSIS

  Посмотреть этот текст:
    ppcSearchQueryStatus.pl --help

  Запустить генерацию всех отчетов соответствующего типа:
    ppcSearchQueryStatus.pl --report-type dynamic
    ppcSearchQueryStatus.pl --report-type performance

  Запустить генерацию только одного отчета:
    ppcSearchQueryStatus.pl --report-type performance --report-id 1234

=head1 DESCRIPTION

    Генерация оффлайн-отчетов по поисковым фразам, ДТО-кампаниям и Смарт-баннерам.

    Скрипт который обрабатывает очередь для генерации отчетов search query (таблица ppcdict.api_queue_search_query).
    Каждый отчет может быть обработан не чаще, чем раз в $Stat::SearchQuery::Queue::HOLD_TIME минут (сейчас это 4 минуты).
    Шаги обработки:
    1. Получает все отчеты, которые нужно удалить из очереди. Подходящие условия:
      статус Abandoned
      статусы New, Process, Done и отчет создан больше, чем $Stat::SearchQuery::Queue::REPORT_TTL{ report-type } секунд назад
      статус Error и отчёт последний раз обрабатывался больше, чем $Stat::SearchQuery::Queue::ERRONEOUS_REPORT_HOLD_TIME секунд назад
    2. Удаляет эти отчеты (в том числе файлы из хранилища, если они есть)
    3. Ставит статус Error для отчётов, достигших rank 5
    4. Получает отчеты для проверки их статуса в БК. условия выборки (объединяются через "И"):
      rank < $Stat::SearchQuery::Queue::MAX_RANK
      статус отчета - один из Stat::SearchQuery::Queue::REPORT_STATUS_TO_CHECK{ report-type } (Process/Done или только Process)
    5. Для выбраннных отчетов запрашиваем у БК их готовность.
      для отчетов типа search - обновляется статус на Done
      для ДТО/ДМО - происходит скачивание данных и генерация xls/csv/tsv/etc... это долгая и тяжелая операция, повышающая rank!
    6. Получает новые отчеты (в статусе New)
    7. Делает в БК запросы на генерацию этих отчетов (и получает для них taskid, по которому потом проверяется статус),
      в базе это поле хранится как bs_id,
      при неудаче повышает rank, при успешной постановке задачи в БК ставит rank 0.
    8. Выбирает удаленные отчеты (в статусе Deleted) и удаляет их из очереди. Также удаляет файлы из хранилища, если были.

    Параметры командной строки:
        --help          - вывести этот текст
        --report-type   - тип отчетов для обработки (обязательный параметр). возможные значения
                          записаны в %Stat::SearchQuery::Queue::REPORT_TYPES.
                          сейчас это dynamic / performance
        --report-id     - id отчета для обработки. скрипт будет обрабатывать только указанный отчет.
                          если тип отчета не совпадает с указанным параметром report-type - он не будет обработан

=cut

use strict;
use warnings;
use utf8;

use List::MoreUtils qw/any/;

use LockTools qw/get_file_lock/;
use ScriptHelper 'get_file_lock' => undef;
use Stat::SearchQuery::Report;
use Stat::SearchQuery::Queue qw/
    get_old_tasks_ids
    get_tasks_for_check_status
    get_new_reports_options
    calculate_error_status
    clear_queue
/;
use Reports::Offline::Dynamic;
use Reports::Offline::Performance;

use Settings;
use PrimitivesIds;
use Yandex::Advmon qw//;
use Yandex::DateTime qw/now/;

=head1 SUBROUTINES/METHODS/VARIABLES

=cut

my ($REPORT_ID, $REPORT_TYPE);
extract_script_params(
    'report-id:i' => \$REPORT_ID,
    'report-type=s' => \$REPORT_TYPE,
);

if (!defined $REPORT_TYPE) {
    die "--report-type is not specified";
} elsif (! exists $Stat::SearchQuery::Queue::REPORT_TYPES{ $REPORT_TYPE }) {
    die "--report-type '$REPORT_TYPE' is not valid";
}


my $log_prefix_guard = $log->msg_prefix_guard("[$REPORT_TYPE]");

get_file_lock($REPORT_ID ? 0 : 'dont_die', get_script_name() . "_$REPORT_TYPE");

$log->out('START');

my $report = Stat::SearchQuery::Report->new(log_object => $log);

ScriptHelper::attach_log_any($log, 'Reports::Offline::Performance', 'trace');
ScriptHelper::attach_log_any($log, 'Reports::Offline::Dynamic', 'trace');

$log->out('Deleting old reports and reports marked for deletion.');
my $old_ids = get_old_tasks_ids(id => $REPORT_ID, type => $REPORT_TYPE);
if (scalar @$old_ids) {
    $report->delete($old_ids);
}

$log->out('Updating statuses.');
my $error_count = calculate_error_status(id => $REPORT_ID, type => $REPORT_TYPE);
if ($error_count) {
    $log->out("$error_count records reached max rank and were marked as erroroneous.");
}
my $tasks = get_tasks_for_check_status(id => $REPORT_ID, type => $REPORT_TYPE);
for my $task (@$tasks) {
    $log->out("Checking id=$task->{id} ($task->{type})");
    eval {
        $report->update([$task],
            on_ready => sub {
                if (any { $task->{type} eq $_ } qw/dynamic performance/) {
                    $log->out("Generating xls report id=$task->{id}");
                }

                if ($task->{type} eq 'dynamic') {
                    Reports::Offline::Dynamic->new()->generate_report($task->{uid}, $task->{id});
                } elsif ($task->{type} eq 'performance') {
                    Reports::Offline::Performance->new()->generate_report($task->{uid}, $task->{id});
                } else {
                    $log->warn("Unknown task type '$task->{type}' for report id $task->{id}");
                }
            });
        1;
    }
    or $log->warn("Failed id=$task->{id}: $@");
}

$log->out('Requesting new reports.');
my $new_tasks = get_new_reports_options(id => $REPORT_ID, type => $REPORT_TYPE);
if (scalar @$new_tasks) {
    my $bs_params = _prepare_bs_params($new_tasks);
    $report->create($bs_params);
}

$log->out('Removing old records on deleted reports.');
# вместе с очисткой очереди надо удалить файлы из монго;
# делаем через колбек, чтобы не переворачивать всю логику
my $task_cb = sub {
    my $rep = shift;
    if ($rep->{type} eq 'dynamic') {
        for my $type (values %Reports::Offline::Dynamic::FILE_TYPE) {
            my $key = $type->{key};
            next if !$key;
            next if !$rep->{options}->{$key};
            $log->out("Removing report file for id=$rep->{id} ($key)");
            Reports::Offline::Dynamic->delete_report_file($rep->{options}->{$key}, $rep->{uid});
        }
    } elsif ($rep->{type} eq 'performance') {
        for my $key (values %Reports::Offline::Performance::FILE_TYPE) {
            next if !$key;
            next if !$rep->{options}->{$key};
            $log->out("Removing report file for id=$rep->{id} ($key)");
            Reports::Offline::Performance->delete_report_file($rep->{options}->{$key}, $rep->{uid});
        }
    } else {
        return;
    }
};

clear_queue(log => $log, task_cb => $task_cb, id => $REPORT_ID, type => $REPORT_TYPE);

local $Yandex::Advmon::GRAPHITE_PREFIX = sub {[qw/direct_one_min db_configurations/, $Settings::CONFIGURATION, qw/flow SearchQueryStatus/, $REPORT_TYPE]};
Yandex::Advmon::monitor_values({
    "deleted" => scalar @$old_ids,
    "checked" => scalar @$tasks,
    "created" => scalar @$new_tasks,
});

juggler_ok(service_suffix => $REPORT_TYPE);

$log->out('FINISH');

=head2 _prepare_bs_params($options)

    Преобразование запроса к API в запрос к БК

=cut

sub _prepare_bs_params {
    my $tasks = shift;
    my @results;
    for my $task (@$tasks) {
        my $options = $task->{options};
        $options->{StartDate} ||= now()->subtract(months => 3)->strftime("%Y%m%d");
        $options->{StartDate} =~ s/\-//g;
        $options->{EndDate} ||= now()->strftime("%Y%m%d");
        $options->{EndDate} =~ s/\-//g;
        my $order_ids = get_orderids(cid => $options->{CampaignIds});
        my $bs_params = {
            orderlist => join(',', @$order_ids),
            startdate => $options->{StartDate},
            stopdate => $options->{EndDate},
        };
        if ($options->{ClicksThreshold}) {
            $bs_params->{cliksthreshold} = $options->{ClicksThreshold};
        }
        if ($options->{ImpressionsThreshold}) {
            $bs_params->{showsthreshold} = $options->{ImpressionsThreshold};
        }

        if (any { $task->{type} eq $_ } qw/dynamic performance/) {
            $bs_params->{bannerdesc} = 1;
            $bs_params->{metricdesc} = 1;
            $bs_params->{incemptyqueries} = 1;
        }

        $task->{options} = $bs_params;
    }
    return $tasks;
}
