#!/usr/bin/perl

=head1 METADATA

<crontab>
    time: */5 * * * *
    sharded: 1
    <switchman>
        group: scripts-other
        <leases>
            mem: 250
        </leases>
    </switchman>
    package: scripts-switchman
</crontab>
<juggler>
    host:   checks_auto.direct.yandex.ru
    sharded: 1
    ttl: 40m
    tag: direct_group_internal_systems
</juggler>

<crontab>
    time: */10 * * * *
    <switchman>
        group: scripts-test
    </switchman>
    package: conf-test-scripts
    sharded: 1
</crontab>

=head1 DESCRIPTION

    Send changes info for campaigns and clients to balance
    exec from cron

=head1 RUNNING

    ./protected/ppcSendBalanceInfoChanges.pl --shard-id=2

=cut

use Direct::Modern;

use Yandex::Advmon;
use Yandex::ListUtils;
use Yandex::DBTools;
use Yandex::TimeCommon;

use my_inc "..";

use Campaign;
use Client;
use GeoTools;
use RBAC2::Extended;
use RBACElementary;
use ScriptHelper sharded => 1, 'Yandex::Log' => 'messages';
use Settings;
use User;

# Сколько записей выбираем из очереди за один раз
# в среднем за 10 минут обрабатывается 3-4 тысячи записей
my $QUEUE_LIMIT = 3_000;
# Размер обрабатываемой пачки (обновление статуса обработки происходит после обработки)
my $CHUNK_SIZE = 100;

# Передаем в Баланс данные про город клиента (SUBREGION_ID) только для клиентов, созданных после 2016-04-01
my $SUBREGION_ID_BORDER_DATE = '2016-04-01';

#-----------------------------------------------------------
my ($wait_to_error, $error_to_send);
sub _init_stats {
    $wait_to_error = 0;
    $error_to_send = 0;
}
# считаем количество записей, меняющих свой статус Wait->Error или Error->Send
sub _calc_stats {
    my ($row, $status) = @_;
    if ($row->{send_status} eq 'Wait' && $status eq 'Error') {
        $wait_to_error++;
    } elsif ($row->{send_status} eq 'Error' && $status eq 'Send') {
        $error_to_send++;
    }
}
#-----------------------------------------------------------

sub main
{
    extract_script_params();

    $log->out('START');

    # помечаем данные на отправку для исключения гонок
    # Error оставляем как есть, и выбираем на переотправку чтоб по базе можно было увидеть что эта запись стабильно застряла на ошибке
    do_sql(PPC(shard => $SHARD), "
        UPDATE balance_info_queue
           SET send_status = 'Sending'
         WHERE send_status = 'Wait'
      ORDER BY priority DESC, id
         LIMIT $QUEUE_LIMIT");

    my $queue = get_all_sql(PPC(shard => $SHARD), "
        SELECT id
             , operator_uid
             , obj_type, cid_or_uid
             , add_time
             , send_status
             , priority
          FROM balance_info_queue
         WHERE send_status IN ('Sending', 'Error')
      ORDER BY priority DESC, id
         LIMIT $QUEUE_LIMIT
    ") || [];

    _init_stats();

    my (@sent_ids, @error_ids);
    for my $chunk (chunks($queue, $CHUNK_SIZE)) {
        my $rbac = eval { RBAC2::Extended->get_singleton(1) }
            or $log->die("Error initialising RBAC: $@");

        # обрабатываем пачку
        for my $row (@$chunk) {
            $log->out('Processing new task:', $row);

            my $id = $row->{id};
            my $operator_uid = $row->{operator_uid};

            my ($error, $res, $error_res);
            if ($row->{obj_type} eq 'cid') {
                my $cid = $row->{cid_or_uid};

                if (! get_one_field_sql(PPC(shard => $SHARD), "select count(*) from campaigns where cid = ?", $cid)) {
                    # табличка с очередью - не решардится, поэтому запись может быть удалена по причине того, что клиента перенесли в другой шард
                    $log->out("Deleting from queue because of unexisting campaign row with id $id");
                    do_delete_from_table(PPC(shard => $SHARD), "balance_info_queue", where => {id => $id});
                    next;
                }

                eval {
                    ($res) = create_campaigns_balance($rbac, $operator_uid, [$cid])->{balance_res};
                };
                $error = $@;
                $error_res = !$res;
            } elsif ($row->{obj_type} eq 'uid') {
                my $uid = $row->{cid_or_uid};
                my $chief_uid;
                my $role = rbac_who_is($rbac, $uid);
    
                # получаем шефов
                if ($role eq 'agency') {
                    $chief_uid = rbac_get_chief_rep_of_agency_rep($uid);
                } elsif ($role eq 'client') {
                    $chief_uid = rbac_get_chief_rep_of_client_rep($uid);
                } else {
                    # например если это менеджер
                    $chief_uid = $uid;
                }

                my $client_info = get_user_data($chief_uid, [qw/ClientID fio phone email geo_id/]);

                if (!$client_info || ref($client_info) ne 'HASH' || ! $client_info->{ClientID}) {
                    # табличка с очередью - не решардится, поэтому запись может быть удалена по причине того, что клиента перенесли в другой шард
                    $log->out("Deleting from queue because of unexisting user row with id $id");
                    do_delete_from_table(PPC(shard => $SHARD), 'balance_info_queue', where => {id => $id});
                    next;
                }
    
                my $client_id = $client_info->{ClientID};
                my $client_data = get_client_data($client_id, [qw/name agency_url country_region_id subregion_id create_date/]);
                my $client_name = $role eq 'agency' ? $client_data->{name} : $client_info->{fio};
                my $agency_url = $client_data->{agency_url};
                my $new_info = {
                    NAME  => $client_name,
                    PHONE => $client_info->{phone},
                    EMAIL => $client_info->{email},
                };
                $new_info->{URL} = $agency_url if $agency_url;
    
                if (defined $client_info->{geo_id} && $client_info->{geo_id} != 0) {
                    my $city_name = get_cityname_by_geoid($client_info->{geo_id});
                    $new_info->{CITY} = $city_name if $city_name;
                }

                if ($client_data && $client_data->{country_region_id}) {
                    $new_info->{REGION_ID} = $client_data->{country_region_id};
                }

                my $client_create_date_ts = mysql2unix($client_data->{create_date} || 0);
                if ($client_data && $client_data->{subregion_id} && ($client_create_date_ts > mysql2unix($SUBREGION_ID_BORDER_DATE))) {
                    # данные про город клиента (SUBREGION_ID) передаются только для новых клиентов, созданных после $SUBREGION_ID_BORDER_DATE
                    $new_info->{SUBREGION_ID} = $client_data->{subregion_id};
                }

                # В текущем виде не переотправляются поля: FAX, IS_AGENCY, AGENCY_ID

                my $res = eval { update_client_id($operator_uid, $client_id, $new_info) };
                $error_res = $res;
                $error = $@;
            }

            my $status = ($error || $error_res) ? 'Error' : 'Send';
            _calc_stats($row, $status);
            if ($status eq 'Error') {
                $log->out("Error resending record with id $id:", {error => $error, res => $res});
                push @error_ids, $id;
            } elsif ($status eq 'Send') {
                $log->out("Successfully resent record with id $id");
                push @sent_ids, $id;
            }
        }
        # сохраняем результат обработки: отправленные удаляем, ошибки записываем
        if (@sent_ids) {
            do_delete_from_table(PPC(shard => $SHARD), 'balance_info_queue', where => {id => \@sent_ids});
        }
        if (@error_ids) {
            do_update_table(PPC(shard => $SHARD), 'balance_info_queue', {send_status => 'Error'}, where => {id => \@error_ids});
        }
    }

    # отмечаемся в графите, что все ок.
    local $Yandex::Advmon::GRAPHITE_PREFIX = sub {[qw/direct_one_min db_configurations/, $Settings::CONFIGURATION]};
    monitor_values({
        flow              => { balance_info_queue => {
                                                       processed_rows    => { "shard_$SHARD" => scalar(@$queue) },
                                                       wait_to_error_cnt => { "shard_$SHARD" => $wait_to_error },
                                                       error_to_send_cnt => { "shard_$SHARD" => $error_to_send },
                                                     }
                             },
    });

    juggler_ok();

    $log->out('FINISH');
}

main();
