#!/usr/bin/perl

=head1 METADATA

<crontab>
    time: 51 */12 * * *
    sharded: 1
    <switchman>
        group: scripts-other
        <leases>
            mem: 250
        </leases>
    </switchman>
    package: scripts-switchman
</crontab>
<juggler>
    host:   checks_auto.direct.yandex.ru
    sharded: 1
    ttl: 1d40m
    tag: direct_group_internal_systems
</juggler>

<crontab>
    time: 51 */12 * * *
    <switchman>
        group: scripts-test
    </switchman>
    package: conf-test-scripts
    sharded: 1
</crontab>

=head1 DESCRIPTION

    Отправляет в Баланс информацию о том, что у клиента навсегда заблокирована
    возможность отключения общего счета

=head1 RUNNING

    ./protected/ppcSendBalanceWalletLock.pl --shard-id=2  [--client-id=123456]

=cut

use my_inc "..";

use Direct::Modern;

use Yandex::DBTools;
use Yandex::DateTime;
use DateTime::Format::MySQL;
use RBAC2::Extended;
use RBACDirect;
use RBACElementary;

use Campaign;
use Campaign::Types;
use ScriptHelper sharded => 1, 'Yandex::Log' => 'messages';
use Settings;

# Общее количество записей которые мы обрабатываем за один запуск
my $QUEUE_LIMIT = 1_000;
# Размер обрабатываемой пачки
my $CHUNK_SIZE = 100;
# Период в днях, после достижения которого, информация о блокировке отправляется в Баланс
my $LOCAL_LOCK_INTERVAL = 7;
my $OPERATOR_UID = 1;

my $rbac = RBAC2::Extended->get_singleton($OPERATOR_UID);


=head2 check_and_fix_servised_clients($client_id)

    Если логин сервисируемый - проверить, что его кампании сервисируются под тем же менеджером, что и ОС,
    и поправить положение, если это не так. См. https://st.yandex-team.ru/DIRECT-52243

    Возвращает cid общего счета

=cut


sub check_and_fix_servised_clients {
    my ($client_id, $work_currency) = @_;

    $log->out("Checking serviced campaings of client $client_id");
    my $chief_uid = rbac_get_chief_rep_of_client($client_id);
    my $campaigns = get_all_sql(PPC(shard => $SHARD),
        ["SELECT cid, ManagerUID, type, IFNULL(currency, 'YND_FIXED') as currency FROM campaigns",
         WHERE => {uid => $chief_uid, type => get_camp_kind_types('text_campaign_in_balance', 'billing_aggregate', 'cpm')}]
    );

    my ($wallet_cid, $wallet_manager, %manager2cids, @unserviced_cids);
    for my $campaign (@$campaigns) {
        if ($campaign->{type} eq 'wallet' && $work_currency eq $campaign->{currency}) {
            $wallet_manager = $campaign->{ManagerUID};
            $wallet_cid = $campaign->{cid};
        }
        if (defined $campaign->{ManagerUID}) {
            push @{$manager2cids{$campaign->{ManagerUID}}}, $campaign->{cid};
        } else {
            push @unserviced_cids, $campaign->{cid};
        }
    }

    # Если логин не сервисируемый - едем дальше, если не нашли общий счет - тоже едем дальше.
    return $wallet_cid if (scalar keys %manager2cids == 0 || ! defined $wallet_cid);

    $log->out("Serviced campaings of client $client_id found and will be checked for consistency");
    my $manager_uid;
    if (defined $wallet_manager) {
        # Если у общего счета есть менеджер - проставить его всем кампаниям
        $manager_uid = $wallet_manager;
        delete $manager2cids{$wallet_manager};
    } else {
        # Если у счета менеджера нет, но он есть у кампаний и он один - проставить его счету и остальным;
        # Если у счета менеджера нет, а у кампаний их несколько - выбрать того, у которого больше кампаний.
        # Если кампаний одинаковое количество - берем менеджера с наименьшим uid.
        $log->out("Wallet has no manager, so we will set it");
        $manager_uid = ${keys(%manager2cids)}[0];

        if (scalar keys %manager2cids > 1) {
            my $max_score = scalar @{$manager2cids{$manager_uid}};

            for my $m_uid (keys %manager2cids) {
                my $score = scalar @{$manager2cids{$m_uid}};
                if ($score > $max_score || (($score == $max_score) && ($m_uid < $manager_uid))) {
                    $manager_uid = $m_uid;
                }
            }
        }
        delete $manager2cids{$wallet_manager};
    }

    # Переводим кампании от одного менеджера другому
    for my $old_uid (keys %manager2cids) {
        $log->out("Switching campaigns of manager $old_uid to $manager_uid");
        for my $cid (@{$manager2cids{$old_uid}}) {
            $log->out("Processing campaign $cid");
            my $errorcode = rbac_change_manager($rbac, $old_uid, $manager_uid, $cid);
            if ($errorcode) {
                $log->out("Error $errorcode when switching campaign $cid to manager $manager_uid");
                return;
            }

            campaign_manager_changed($rbac, $chief_uid, $cid, $manager_uid);
        }
    }

    # Сервисируем кампании, у которых нет менеджера
    for my $cid (@unserviced_cids) {
        $log->out("Servicing campaign $cid to manager $manager_uid");
        my $errorcode = rbac_move_nscamp_to_scamp($rbac, $cid, $manager_uid);
        if ($errorcode) {
            $log->out("Error $errorcode when servicing campaign $cid to manager $manager_uid");
            return;
        }

    }
    BalanceQueue::add_to_balance_info_queue($chief_uid, cid => \@unserviced_cids, BalanceQueue::PRIORITY_CAMP_ON_SERVICED);

    return $wallet_cid;
}


sub main
{
    my $client_id;
    extract_script_params("client-id=i" => \$client_id);

    $log->out('START');

    my $where_clause = '';
    if (defined $client_id) {
        $where_clause = "w.ClientID = $client_id AND";
        $log->out("Working only with clientid $client_id");
    }

    my $entries_processed = 0;
    while ($entries_processed < $QUEUE_LIMIT) {
        $log->out("Starting processing of chunk with offset $entries_processed");
        my $chunk = get_all_sql(PPC(shard => $SHARD), "
            SELECT w.ClientID, IFNULL(cl.work_currency, 'YND_FIXED') as currency
              FROM wallet_campaigns_lock w
                LEFT JOIN clients cl USING(ClientID)
             WHERE $where_clause w.state IN ('created', 'sending', 'error')
               AND w.lock_set_time <= NOW() - INTERVAL $LOCAL_LOCK_INTERVAL DAY
          ORDER BY w.ClientID
             LIMIT $CHUNK_SIZE
        ") || [];
        if (scalar @$chunk == 0) {
            $log->out("No more entries to process");
            last;
        };
        $entries_processed += scalar @$chunk;

        do_update_table(PPC(shard => $SHARD), 'wallet_campaigns_lock', {state => 'sending'},
            where => {ClientID => [map {$_->{ClientID}} @$chunk]});

        my (@error_clients, @sent_clients);
        for my $row (@$chunk) {
            $log->out("Processing common wallet for client $row->{ClientID}");

            my $wallet_cid = check_and_fix_servised_clients($row->{ClientID}, $row->{currency});
            if (!defined $wallet_cid) {
                $log->out("Client $row->{ClientID} doesn't have a wallet, skipping");
                push @error_clients, $row->{ClientID};
                next;
            }
            my $result = create_campaigns_balance($rbac, $OPERATOR_UID, [$wallet_cid], is_lock_wallet => 1);

            if (defined $result->{error}) {
                $log->out("Error when sending client $row->{ClientID} balance");
                push @error_clients, $row->{ClientID};
            } else {
                $log->out("Succesfully sent client $row->{ClientID} balance");
                push @sent_clients, $row->{ClientID};
            }
        }

        $log->out("Updating chunk states in database");
        if (scalar @error_clients) {
            do_update_table(PPC(shard => $SHARD), 'wallet_campaigns_lock', {state => 'error'}, where => {ClientID => \@error_clients});
        }
        if (scalar @sent_clients) {
            do_update_table(PPC(shard => $SHARD), 'wallet_campaigns_lock', {state => 'sent'}, where => {ClientID => \@sent_clients});
        }
        $log->out("Finished processing chunk. $entries_processed of $QUEUE_LIMIT done");
    }

    juggler_ok();

    $log->out('FINISH');
}

main();
