#!/usr/bin/perl

use my_inc "..";

=head1 METADATA

<crontab>
    time: */5 * * * *
    sharded: 1
    <switchman>
        group: scripts-other
        <leases>
            mem: 1500
        </leases>
    </switchman>
    package: scripts-switchman
</crontab>
<juggler>
    host:   checks_auto.direct.yandex.ru
    sharded:        1
    ttl:            1h30m
    tag: direct_group_internal_systems
    <notification>
            template: on_status_change
            status: OK
            status: CRIT
            method: telegram
            login: DISMonitoring
    </notification>
</juggler>

<crontab>
    time: */5 * * * *
    sharded: 1
    only_shards: 1
    package: scripts-sandbox
</crontab>
<juggler>
    host:   checks_auto.direct.yandex.ru
    name:           scripts.ppcSendOrderWarnings.working.sandbox
    raw_host:       CGROUP%direct_sandbox
    raw_events:     scripts.ppcSendOrderWarnings.working.sandbox.shard_$shard
    vars:           shard=1
    ttl:            1h30m
    tag: direct_group_internal_systems
</juggler>

=cut

=head1 NAME

$Id$

=head1 DESCRIPTION

Отсылка клиентам предупреждений о заканчивающихся или закончившихся кампаниях
Вынесен из bsClientActiveOrders.pl

=cut

use warnings;
use strict;
use feature 'state';

use Time::Local;

use ScriptHelper sharded => 1, 'Yandex::Log' => 'messages';
use Client::ClientFeatures;
use Currencies;
use Notification;
use JavaIntapi::SendMoneyOutWarning;
use Property;
use RBAC2::Extended;
use RBACElementary;
use Settings;
use Stat::OrderStatDay;
use User;
use Client qw/is_new_wallet_warnings_client/;
use WalletUtils;

use Yandex::DBTools;
use Yandex::DateTime qw/now/;

# Событие, которое случилось с клиентом
# В зависимости от этого менеджерам клиентов отправляются разные нотификации
use constant {
    # Деньги закончились, и порог отключения тоже достигнут (был автоовердрафт)
    MONEY_OUT_AND_AUTO_OVERDRAFT_OUT => 1,
    # Деньги закончились, но порог отключения ещё не достигнут (был автоовердрафт)
    MONEY_OUT_AND_AUTO_OVERDRAFT_ACTIVE => 2,
    # Деньги закончились, и автоовердрафта не было
    MONEY_OUT_AND_NO_AUTO_OVERDRAFT => 3,
};

state $use_intapi_for_active_orders_money_out = Property->new('use_intapi_for_active_orders_money_out');
=head2

    Обрабатывает тех клиентов, которые пользуются автоовердрафтом и у которых был достигнут порог отключения

=cut
sub process_out_of_money_with_auto_overdrafts {
    my ($rbac, $and_cid_in_list, $uids_with_money_out) = @_;

    $log->out("Started process_out_of_money_with_auto_overdrafts");

    # Ищем кошельки со включённым автоовердрафтом, о которых мы уже сообщали клиентам то, что по основному балансу
    # деньги закончились, но ещё не сообщали о том, что деньги закончились даже с учётом автоовердрафта
    my $sth = exec_sql(PPC(shard => $SHARD),
        "SELECT
            c.cid,
            c.uid,
            c.name,
            c.type as mediaType,
            t.sum as sum,
            t.wallet_sum,
            t.total as total,
            co.email as cmail,
            u.email,
            u.FIO,
            u.login,
            u.phone,
            u.ClientID,
            c.statusMail,
            c.currency,
            t.wallet_sum_debt,
            t.debt,
            t.overdraft_lim,
            t.auto_overdraft_lim,
            t.statusBalanceBanned
        FROM (
            SELECT
                c.cid,
                c.sum as wallet_sum,
                c.sum + ifnull(sum(cuw.sum), 0) as sum,
                (c.sum - c.sum_spent) + ifnull(sum(cuw.sum - cuw.sum_spent), 0) as total,
                sum(IF(cuw.sum - cuw.sum_spent < 0, cuw.sum - cuw.sum_spent, 0)) AS wallet_sum_debt,
                clo.debt,
                clo.overdraft_lim,
                clo.auto_overdraft_lim,
                IFNULL(clo.statusBalanceBanned, 'No') as statusBalanceBanned
            FROM campaigns c
                JOIN clients_options clo ON c.ClientID = clo.ClientID
                LEFT JOIN campaigns cuw ON cuw.uid = c.uid AND cuw.wallet_cid = c.cid
            WHERE c.type = 'wallet'
                AND c.statusMail = 2
                AND clo.overdraft_lim > 0 AND clo.auto_overdraft_lim > 0
                AND (clo.statusBalanceBanned IS NULL OR clo.statusBalanceBanned = 'No')
                $and_cid_in_list
            GROUP BY c.cid
        ) as t
            JOIN campaigns c ON c.cid = t.cid
            LEFT JOIN camp_options co ON co.cid = c.cid
            LEFT JOIN users u ON u.uid = c.uid");

    while (my ($cid, $uid, $name, $mediaType, $sum, $wallet_sum, $total,
        $cmail, $email, $fio, $client_login, $client_phone,
        $ClientID, $statusMail, $currency, $wallet_sum_debt,
        $debt, $overdraft_lim, $auto_overdraft_lim, $statusBalanceBanned) = $sth->fetchrow_array)
    {
        # Вычисляем, сколько денег добавляется с автоовердрафта
        my $auto_overdraft_addition = WalletUtils::get_auto_overdraft_addition({
            type            => 'wallet',
            currency        => $currency,
            sum             => $wallet_sum,
            wallet_sum_debt => $wallet_sum_debt
        }, {
            clientID            => $ClientID,            # идентификатор клиента
            debt                => $debt,                # текущий долг клиента (размер выставленного, но ещё не оплаченного счёта)
            overdraft_lim       => $overdraft_lim,       # 0 или лимит овердрафта
            auto_overdraft_lim  => $auto_overdraft_lim,  # 0 или порог отключения, заданный пользователем
            statusBalanceBanned => $statusBalanceBanned  # "забанен" ли клиент в Балансе ('Yes' или 'No')
        });

        # Если добавки по автоовердрафту фактически не происходит, то этот случай мы просто пропускаем
        next if $auto_overdraft_addition == 0;

        next unless $sum > 0 && $total + $auto_overdraft_addition < $Currencies::EPSILON;

        do_update_table(PPC(shard => $SHARD), 'campaigns', { statusMail => 3 }, where => {
            cid        => $cid,
            statusMail => $statusMail
        });

        my $vars = {
            uid           => $uid,
            client_uid    => $uid,
            fio           => $fio,
            cid           => $cid,
            campaign_id   => $cid,
            camp_name     => $name,
            campaign_name => $name,
            client_login  => $client_login,
            client_email  => $cmail || $email,  # Если есть email, указанный в настройках кампании, используем его
            client_fio    => $fio,
            client_phone  => $client_phone,
            client_id     => $ClientID,
            campaign_type => ($mediaType || 'text'),
        };

        # Посылаем нотификацию о том, что деньги на ОС закончились до такой степени, что достигнут порог отключения
        add_notification($rbac, 'active_orders_money_out_with_auto_overdraft', $vars, {mail_fio_tt_name => 'fio'});
        $log->out("active_orders_money_out_with_auto_overdraft cid: $cid uid: $uid");

        # Сохраняем uid клиента для последующей нотификации менеджеров остановившихся кампаний (если такие будут обнаружены)
        $uids_with_money_out->{$uid} = MONEY_OUT_AND_AUTO_OVERDRAFT_OUT;
    }
    $sth->finish;

    $log->out("Finished process_out_of_money_with_auto_overdrafts");
}

my $uc_clients_cache;
my $cid = '';
extract_script_params( "cid=s" => \$cid);
my $and_cid_in_list = $cid ? 'and c.cid in (' . join(', ', split(/\D+/, $cid)) . ')' : "";

$log->out("start");

my $rbac = RBAC2::Extended->get_singleton(1);

# uid'ы клиентов, у которых закончились деньги на кошельке, и события, которые с ними произошли
my $uids_with_money_out = {};

process_out_of_money_with_auto_overdrafts($rbac, $and_cid_in_list, $uids_with_money_out);

# Turn Off Order
$log->out("turn off");

# выбираем либо простые кампании (если счет не включен), либо кампании "общий счет"
# для кампаний "общий счет" суммируем все суммы на связанных кампаниях

my $sth = exec_sql(PPC(shard=>$SHARD), "SELECT c.cid, c.uid, c.name,
                                                   c.type as mediaType,
                                                   t.sum as sum,
                                                   t.wallet_sum,
                                                   t.total as total,
                                                   co.email as cmail,
                                                   u.email,
                                                   u.FIO,
                                                   u.login,
                                                   u.phone,
                                                   u.ClientID,
                                                   c.statusMail,
                                                   c.currency,
                                                   IFNULL(t.wallet_sum_debt, 0) as wallet_sum_debt,
                                                   IFNULL(clo.debt, 0) as debt,
                                                   IFNULL(clo.overdraft_lim, 0) as overdraft_lim,
                                                   IFNULL(clo.auto_overdraft_lim, 0) as auto_overdraft_lim,
                                                   IFNULL(clo.statusBalanceBanned, 'No') as statusBalanceBanned,
                                                   c.sum_last,
                                                   c.OrderID,
                                                   cuw_order_ids
                                              FROM (
                                                   SELECT c.cid,
                                                          c.sum + ifnull(sum(cuw.sum), 0) as sum,
                                                          c.sum as wallet_sum,
                                                          (c.sum - c.sum_spent) + ifnull(sum(cuw.sum - cuw.sum_spent), 0) as total,
                                                          sum(IF(cuw.sum - cuw.sum_spent < 0, cuw.sum - cuw.sum_spent, 0)) AS wallet_sum_debt,
                                                          GROUP_CONCAT(cuw.OrderID) as cuw_order_ids
                                                     FROM campaigns c
                                                          LEFT JOIN campaigns cuw ON cuw.uid = c.uid AND cuw.wallet_cid = c.cid
                                                    WHERE c.type = 'wallet'
                                                      AND (c.statusMail < 2 OR c.statusMail = 4)
                                                      $and_cid_in_list
                                                    GROUP BY c.cid
                                                    UNION ALL
                                                   SELECT c.cid,
                                                          c.sum as sum,
                                                          0 as wallet_sum,
                                                          (c.sum - c.sum_spent) as total,
                                                          0 AS wallet_sum_debt,
                                                          '' as cuw_order_ids
                                                     FROM campaigns c
                                                    WHERE c.type != 'wallet'
                                                      AND c.wallet_cid = 0
                                                      AND (c.statusMail < 2 OR c.statusMail = 4)
                                                      $and_cid_in_list
                                                   ) as t
                                                   JOIN campaigns c ON c.cid = t.cid
                                                   LEFT JOIN camp_options co ON co.cid = c.cid
                                                   LEFT JOIN users u ON u.uid = c.uid
                                                   LEFT JOIN clients_options clo ON u.ClientID = clo.ClientID");
while (my ($cid, $uid, $name, $mediaType, $sum, $wallet_sum, $total, $cmail, $email, $fio,
    $client_login, $client_phone, $ClientID, $statusMail, $currency, $wallet_sum_debt,
    $debt, $overdraft_lim, $auto_overdraft_lim, $statusBalanceBanned, $sum_last, $order_id, $cuw_order_ids) = $sth->fetchrow_array) {

    next unless $sum > 0 && $total < _get_money_out_limit($currency, $order_id, $cuw_order_ids, $ClientID, $sum_last, $total);

    # Вычисляем, сколько денег добавляется с автоовердрафта
    my $auto_overdraft_addition = 0;
    if ($mediaType eq 'wallet') {
        $auto_overdraft_addition = WalletUtils::get_auto_overdraft_addition({
            type            => 'wallet',
            currency        => $currency,
            sum             => $wallet_sum,
            wallet_sum_debt => $wallet_sum_debt
        }, {
            clientID            => $ClientID,            # идентификатор клиента
            debt                => $debt,                # текущий долг клиента (размер выставленного, но ещё не оплаченного счёта)
            overdraft_lim       => $overdraft_lim,       # 0 или лимит овердрафта
            auto_overdraft_lim  => $auto_overdraft_lim,  # 0 или порог отключения, заданный пользователем
            statusBalanceBanned => $statusBalanceBanned  # "забанен" ли клиент в Балансе ('Yes' или 'No')
        });
    }

    do_sql(PPC(shard => $SHARD), "UPDATE campaigns set statusMail = 2 where cid = ? and statusMail = ?", $cid, $statusMail);

    # Send mails to campaign specific address if present
    $email = $cmail || $email;

    my $vars = {
        uid           => $uid,
        client_uid    => $uid,
        fio           => $fio,
        cid           => $cid,
        campaign_id   => $cid,
        camp_name     => $name,
        campaign_name => $name,
        client_login  => $client_login,
        client_email  => $email,
        client_fio    => $fio,
        client_phone  => $client_phone,
        client_id     => $ClientID,
        campaign_type => ($mediaType || 'text'),
        auto_overdraft_active => ($auto_overdraft_addition > 0) ? 'true' : 'false',
    };

    if ($use_intapi_for_active_orders_money_out->get(300) && ($mediaType eq 'wallet')) { # в Java есть шаблоны только для кошельков
        my $notification_data = {'login' => $client_login, 'cid' => $cid};
        $notification_data->{_campaign_email} = $cmail if $cmail && $cmail gt '';
        JavaIntapi::SendMoneyOutWarning->new(template_params_by_uid => {$uid => $notification_data})->call();
    }
    else {
        add_notification($rbac, 'active_orders_money_out', $vars, {mail_fio_tt_name => 'fio'});
    }
    $log->out("active_orders_money_out cid: $cid uid: $uid");

    if ($mediaType =~ /^(text|wallet)$/) {
        if ($auto_overdraft_addition > 0) {
            $uids_with_money_out->{$uid} = MONEY_OUT_AND_AUTO_OVERDRAFT_ACTIVE;
        } else {
            $uids_with_money_out->{$uid} = MONEY_OUT_AND_NO_AUTO_OVERDRAFT;
        }
    }
}
$sth->finish;

# посылаем предупреждение менеджеру если кончились средства на ВСЁМ клиенте (включая и самоходные кампании)
if (%$uids_with_money_out) {
    my $client_uids_sql = join ',', map {sql_quote($_)} keys $uids_with_money_out;

    do_sql(PPC(shard => $SHARD), "SET SESSION group_concat_max_len = 100000");

    my @currencies = get_currencies_list();
    my %currency2money_out = map { $_ => get_currency_constant($_, 'MONEY_OUT_LIMIT') } @currencies;
    my $total_money_out_case_sql = sql_case('#currency#', \%currency2money_out);
    $total_money_out_case_sql =~ s/`#currency#`/IFNULL(cl.work_currency, 'YND_FIXED')/;

    my $clients = get_all_sql(PPC(shard => $SHARD), "select u.uid
                                          , cl.work_currency
                                          , u.login
                                          , GROUP_CONCAT(distinct c.ManagerUID SEPARATOR ',') as managers_uids
                                     from campaigns c
                                       join users u on c.uid = u.uid
                                       join clients cl on c.ClientID = cl.ClientID
                                     where c.uid in ($client_uids_sql)
                                       and IFNULL(c.currency, 'YND_FIXED') = IFNULL(cl.work_currency, 'YND_FIXED')
                                       and (sum > 0 or sum_spent > 0)
                                     group by u.uid
                                     having SUM(sum - sum_spent) < $total_money_out_case_sql
                                        and SUM(IF(c.ManagerUID > 0, 1, 0)) > 0
                                    ");
    for my $row (@$clients) {
        for my $manager_uid (grep {defined $_ && m/^\d+$/} split /,/, $row->{managers_uids}) {

            my $manager_info = get_user_data($manager_uid, [qw/email FIO/]);
            my $client_id = rbac_get_client_clientid_by_uid($row->{uid});
            my $client_reps = get_all_sql(PPC(shard => $SHARD), [
                'SELECT FIO, login FROM users',
                WHERE => {
                    uid => [
                        grep {!rbac_is_client_chief_rep($_)}
                        @{rbac_get_main_reps_of_client($client_id)}
                    ]
                }
            ]);
            my $mail_vars = {
                manager_email  => $manager_info->{email}
                , manager_uid  => $manager_uid
                , manager_fio  => $manager_info->{FIO}
                , client_login => $row->{login}
                , client_id    => $client_id
                , client_reps  => $client_reps
                , client_reps  => $client_reps
            };
            my $event = $uids_with_money_out->{$row->{uid}};
            if ($event == MONEY_OUT_AND_NO_AUTO_OVERDRAFT) {
                $mail_vars->{auto_overdraft_active} = 'false';
                add_notification($rbac, 'active_orders_money_out_on_client', $mail_vars);
                $log->out("active_orders_money_out_on_client manager_uid: $manager_uid client_login: $row->{login} auto_overdraft: no");
            } elsif ($event == MONEY_OUT_AND_AUTO_OVERDRAFT_ACTIVE) {
                $mail_vars->{auto_overdraft_active} = 'true';
                add_notification($rbac, 'active_orders_money_out_on_client', $mail_vars);
                $log->out("active_orders_money_out_on_client manager_uid: $manager_uid client_login: $row->{login} auto_overdraft: active");
            } elsif ($event == MONEY_OUT_AND_AUTO_OVERDRAFT_OUT) {
                # Не посылаем нотификацию (см. DIRECT-86827)
            }
        }
    }
}

# Order with little funds
$log->out("little funds");
$sth = exec_sql(PPC(shard => $SHARD), "SELECT c.cid, c.uid, c.name,
                                                   IFNULL(co.money_warning_value, $Settings::DEFAULT_MONEY_WARNING_VALUE) AS money_warning_value,
                                                   t.sum as sum,
                                                   t.total as total,
                                                   c.sum_last,
                                                   c.sum_units,
                                                   t.type as mediaType,
                                                   c.currency,
                                                   co.email as cmail,
                                                   u.email,
                                                   u.FIO,
                                                   u.login,
                                                   u.phone,
                                                   u.ClientID,
                                                   c.statusMail
                                              FROM (
                                                   SELECT c.cid,
                                                          c.sum + ifnull(sum(cuw.sum), 0) as sum,
                                                          c.sum_units + ifnull(sum(cuw.sum_units), 0) as sum_units,  
                                                          (c.sum - c.sum_spent) + ifnull(sum(cuw.sum - cuw.sum_spent), 0) as total,
                                                          'wallet' as type
                                                     FROM campaigns c
                                                          LEFT JOIN campaigns cuw ON cuw.uid = c.uid AND cuw.wallet_cid = c.cid
                                                    WHERE c.type = 'wallet'
                                                      AND c.statusMail = 0
                                                      $and_cid_in_list
                                                    GROUP BY c.cid
                                                    UNION ALL
                                                   SELECT c.cid,
                                                          c.sum as sum,
                                                          c.sum_units as sum_units,  
                                                          (c.sum - c.sum_spent) as total,
                                                          c.type
                                                     FROM campaigns c
                                                    WHERE c.type != 'wallet'
                                                      AND c.wallet_cid = 0
                                                      AND c.statusMail = 0
                                                      $and_cid_in_list
                                                   ) as t
                                                   JOIN campaigns c ON c.cid = t.cid
                                                   LEFT JOIN camp_options co ON co.cid = c.cid
                                                   LEFT JOIN users u ON u.uid = c.uid");

while (my ($cid, $uid, $name, $money_warning_value, $sum, $total, $sum_last, $sum_units, $mediaType, $currency, $cmail, $email, $fio, $client_login, $client_phone, $ClientID, $statusMail) = $sth->fetchrow_array) {

    my $money_out_limit = get_currency_constant($currency, 'MONEY_OUT_LIMIT');
    next unless $sum > 0
             && $total > $money_out_limit
             && $sum_last > 0
             && $total / $sum_last < $money_warning_value / 100;

    # Исключаем клиентов у который есть кошелек, установлен уровень остатка средств на кампании по умолнчанию и включена фича 
    if ($mediaType eq 'wallet' && $money_warning_value == $Settings::DEFAULT_MONEY_WARNING_VALUE && is_new_wallet_warnings_client($ClientID)) {
        # таким клиентам письма должна отправлять джоба WalletsWarningsEmailSenderJob
        next;
    }

    do_sql(PPC(shard => $SHARD), "UPDATE campaigns set statusMail=1 where cid = ? and statusMail = ?", $cid, $statusMail);

    # Send mails to campaign specific address if present
    $email = $cmail || $email;

    my $vars = {
        fio           => $fio,
        cid           => $cid,
        uid           => $uid,
        client_uid    => $uid,
        campaign_id   => $cid,
        procent       => $money_warning_value,
        camp_name     => $name,
        campaign_name => $name,
        client_login  => $client_login,
        client_email  => $email,
        client_fio    => $fio,
        client_phone  => $client_phone,
        client_id     => $ClientID,
        campaign_type => ($mediaType || 'text'),
        rest          => $total,
        sum_last      => $sum_last,
        sum_last_units=> 100 * int($sum_last * $sum_units / (100 * $sum)),
        currency      => $currency || 'YND_FIXED',
    };

    add_notification($rbac, 'active_orders_money_warning', $vars, {mail_fio_tt_name => 'fio'});
    $log->out("active_orders_money_warning cid: $cid uid: $uid procent: $money_warning_value rest: $vars->{rest}");
}
$sth->finish;

juggler_ok(service_suffix => (EnvTools::is_sandbox() ? 'sandbox' : undef));

$log->out("finish");


#Определяем пороговое значение, ниже которого считаем, что деньги кончились
sub _get_money_out_limit {
    my ($currency, $order_id, $cuw_order_ids, $client_id, $sum_last, $total) = @_;

    $cuw_order_ids //= '';
    my $limit = get_currency_constant($currency, 'MONEY_OUT_LIMIT');

    #Если остаток уже меньше 3% от последнего платежа, но еще больше дефолтного порога - проверим,
    #не стоит ли его повысить из-за быстрого расхода средств
    if ($total > $limit && $total < 0.03 * $sum_last && Client::ClientFeatures::has_feature($client_id, 'use_dynamic_threshold_for_send_order_warnings')) {
        #Порогом считаем 1/10 среднедневного расхода за последнюю неделю
        my $limit_by_spend = _get_last_week_spend([$order_id, (split /,/, $cuw_order_ids)], $currency)/7/10;    
        $limit = $limit_by_spend if $limit_by_spend > $limit;
    }

    if ($total < $limit) {
        $log->out("order_id: $currency $order_id [$cuw_order_ids] reached limit $limit");
    }

    return $limit;
}

sub _get_last_week_spend {
    my ($order_ids, $currency) = @_;

    my @order_ids = grep {$_ > 0} @$order_ids;
    return 0 unless @order_ids;

    my $date_at_past = now()->add(days => -7)->ymd();
    my $stat = Stat::OrderStatDay::get_orders_sum_spent(
        \@order_ids,
        {sum_spent => {from => $date_at_past, to => now()->ymd()}},
        $currency);

    return $stat->{sum_spent} // 0;
}
