#!/usr/bin/perl

use my_inc "..";


=head1 DESCRIPTION

Скрипт для отправки на почту результатов поиска баннеров в РСЯ (cmd=searchBanners) .
Принимает параметры:
    --shard-id <номер шарда>
    --clientid <clientid клиента>

./protected/ppcSendSearchBannersResults.pl --shard-id 1 --clientid 4528876

=cut

=head1 METADATA

<crontab>
    sharded: 1
    time: */10 * * * *
    <switchman>
        group: scripts-other
    </switchman>
    package: scripts-switchman
</crontab>

<juggler>
    host:   checks_auto.direct.yandex.ru
    sharded:    1
    ttl:        120m
    tag: direct_group_internal_systems
</juggler>

=cut

use Direct::Modern;

use Yandex::DBQueue;
use Yandex::Retry qw/relaxed_guard/;
use Yandex::I18n qw/iget/;
use Yandex::HashUtils qw/hash_cut/;
use Yandex::ReportsXLS qw//;
use POSIX qw/ceil strftime/;
use List::Util qw/min/;
use List::MoreUtils qw/uniq/;
use Yandex::ListUtils qw/xuniq/;
use Settings;
use ScriptHelper sharded => 1, 'Yandex::Log' => 'messages';

use SearchBanners qw/convert_old_style_params search_banners/;
use MailService qw/sendmail_attach/;
use User qw/get_user_data/;
use GeoTools qw/get_geo_names/;
use Campaign qw//;
use WalletUtils qw//;
use URI::Escape qw/uri_escape_utf8/;


my $TRYCOUNT = 3;
my $JOBS_LIMIT = 10;
my $SLEEP_COEF = 1;


extract_script_params(
    'sleep-coef=f' => \$SLEEP_COEF,
    'client-id|clientid=s' => \my $client_id,
);

$log->out("START");

$Yandex::DBQueue::LOG = $log;
my $queue = Yandex::DBQueue->new(PPC(shard => $SHARD), 'bansearch_xls');


ITERATION:
while (1) {
    my $rg = relaxed_guard times => $SLEEP_COEF;
    my @jobs = $queue->grab_jobs(
        ($client_id ? (filters => {ClientID => $client_id}) : ()),
        limit => $JOBS_LIMIT,
    );
    $log->out(sprintf "Got %d jobs", scalar @jobs);
    last ITERATION  if !@jobs;

    JOB:
    for my $job (@jobs) {
        my $FORM = $job->args->{FORM};
        my $PAGE_URL = $job->args->{PAGE_URL};
        my $UID = $job->uid;
        my $user_data = get_user_data($UID, [qw/email lang/]);
        $log->out({UID => $UID, FORM => $FORM, PAGE_URL => $PAGE_URL});

        if ($job->trycount > $TRYCOUNT) {
            $job->mark_failed_permanently({ error => 'Unknown error' });
            next JOB;
        }

        eval {
            my $lang = $user_data->{lang} || 'ru';
            my $lang_guard = Yandex::I18n::init_i18n_guard($lang, check_file => 1);

            my ($criteria, $search_params) = convert_old_style_params($FORM);
            my $max_banners_count_in_file = 25_000;
            if (!$search_params->{limit} || $search_params->{limit} > $max_banners_count_in_file) {
                $search_params->{limit} = $max_banners_count_in_file; 
            }
            $log->out({criteria => $criteria, search_params => $search_params});

            #флаг поиск по фразе в РСЯ 
            #если 1 будем несколько раз вызывать search_banners чтобы найти все баннеры
            my $use_bansearch = ($FORM->{what} eq 'phrase' && $FORM->{use_bansearch}) ? 1 : 0;

            my $MAX_ITERATION_COUNT = 40;
            my $banners_to_send = [];
            my $parts = {first_part => 1, last_part => 1, parts_count => 1};
            for my $iteration (1..$MAX_ITERATION_COUNT) {
                #общее количество баннеров вычисляем только при первом запросе (если выбран поиск по фразе в РСЯ)
                $search_params->{compute_total_banners_count} = ($use_bansearch && $iteration == 1) ? 1 : 0;

                my $banners = get_banners($FORM, $criteria, $search_params);
                $log->out("banners count: " . scalar(@$banners));
                $log->out("use_bansearch_bids count: " . scalar(@{ $search_params->{use_bansearch_bids} }));

                my $total_banners_count = $use_bansearch ? ($FORM->{limit} ? min($FORM->{limit}, $search_params->{total_banners_count})
                                                                           : $search_params->{total_banners_count})
                                                         : @$banners;
                my $iterations_count = ceil($total_banners_count / $search_params->{limit});
                $parts->{parts_count} = $iterations_count;
 
                if (@$banners_to_send && (@$banners_to_send + @$banners > $max_banners_count_in_file)) {

                    send_banners($banners_to_send, $FORM, $PAGE_URL, $parts, $user_data->{email});

                    $banners_to_send = [];
                    $parts->{first_part} = $iteration;
                }
                push @$banners_to_send, @$banners;
                $parts->{last_part} = $iteration;

                last unless ($use_bansearch && $iteration < $iterations_count);
            }
            send_banners($banners_to_send, $FORM, $PAGE_URL, $parts, $user_data->{email});

            $job->mark_finished({});
        };
        if ($@) {
            $log->out("FAILED: $@");
            $job->mark_failed_once();
            next JOB;
        };
    }
}

my $LIFETIME_IN_SECONDS = 3600;
$queue->delete_old_jobs($LIFETIME_IN_SECONDS);

juggler_ok();

$log->out("FINISH");

sub send_banners {
    my ($banners, $FORM, $PAGE_URL, $parts, $email) = @_;
    my $xls_data = @$banners ? get_xls_data($banners, $FORM->{group_camp}, $PAGE_URL) : undef;
    my ($text, $subject) = get_letter_text_and_subject($FORM, $PAGE_URL, $parts);

    send_mail_to_manager($email, $xls_data, $text, $subject, $parts->{first_part}); 
    $log->out("sendmail_attach to $email");
}

sub get_banners {
    my ($FORM, $criteria, $search_params) = @_;

    my $banners_search_result = search_banners($criteria, $search_params);
    my $banners_from_search = $banners_search_result->{banners} // [];

    #from cmd_searchBanners 
    my $sum_debt_all = WalletUtils::get_sum_debt_for_wallets_by_uids([uniq map { $_->{uid} } @{$banners_search_result->{banners}}]);
    my $clients_info;
    if (@$banners_from_search) {
        $clients_info = Client::get_clients_auto_overdraft_info([uniq map { $_->{ClientID} } @$banners_from_search]);
    }
    for my $banner_from_search (@$banners_from_search) {
        WalletUtils::calc_camp_uni_sums_with_wallet($banner_from_search, $sum_debt_all, $clients_info->{$banner_from_search->{ClientID}});
    }

    my $campaigns = [xuniq {$_->{cid}} @$banners_from_search];
    my $spent_today = {map { $_->{OrderID} => $_->{spent_today} } grep { $_->{spent_today} } @$banners_from_search};
    $_->{spent_today} = $spent_today->{ $_->{OrderID} } // 0 for @$campaigns;
    my $camp_statuses = Campaign::CalcCampStatus_mass($campaigns);

    my $banners = [];
    for my $banner_from_search (@$banners_from_search) {
        my $banner = hash_cut($banner_from_search, qw/bid title body cid login FIO href domain geo_text afio alogin mfio mlogin geo camp_name flags/); 
        $banner->{geo_text} = get_geo_names($banner->{geo});
        $banner->{camp_status_text} = $camp_statuses->{ $banner_from_search->{cid} }->{text}; 
        push @$banners, $banner;
    }

    my $sort_field = $FORM->{sort};
    if ($sort_field && grep { $sort_field eq $_ } qw/bid cid login href domain geo_text/) {
        $banners = $sort_field =~ /^(cid|bid)$/ ? [ sort { $a->{$sort_field} <=> $b->{$sort_field} } @$banners ] 
                                                : [ sort { $a->{$sort_field} cmp $b->{$sort_field} } @$banners ];
        $banners = [reverse @$banners] if $FORM->{reverse};
    }
    return $banners;
}


sub get_xls_data {
    my ($banners, $group_camp, $host) = @_;

    my @data;
    my %headers = (title => iget('заголовок'), body => iget('текст объявления'), cid => iget('кампания'), bid => iget('баннер'), 
                   href => iget('ссылка'), geo_text => iget('регион'), login => iget('логин'), camp_status => iget('статус'), 
                   mafio => iget('менеджер/агентство'), camp_name => iget('название'), flags => iget('флаги на баннере'),
    );
    my @fields = $group_camp ? qw/cid camp_name login camp_status mafio/ 
                             : qw/title body cid bid href geo_text login camp_status mafio flags/;
    push @data, [ @headers{ @fields } ];

    foreach my $banner (@$banners) {
        my @row;
        foreach my $field (@fields) {
            my $value;
            if (grep {$field eq $_} qw/title body geo_text camp_name flags/) {
                $value = $banner->{$field};
            } elsif ($field eq 'cid') {
                $value = {data => $banner->{cid},
                          url => "$host?cmd=showCamp&cid=$banner->{cid}&ulogin=$banner->{login}&tab=all"};
            } elsif ($field eq 'bid') {
                $value = {data => $banner->{bid},
                          url => "$host?cmd=showCamp&cid=$banner->{cid}&ulogin=$banner->{login}&search_by=num&search_banner=$banner->{bid}&tab=all"};
            } elsif ($field eq 'href') {
                $value = $banner->{domain} ? {data => $banner->{domain}, url => "http://$banner->{domain}"} : '';
            } elsif ($field eq 'login') {
                $value = "$banner->{FIO} ($banner->{login})";
            } elsif ($field eq 'camp_status') {
                $value = $banner->{camp_status_text};
            } elsif ($field eq 'mafio') {
                $value = ($banner->{alogin} ? iget('Агентство') . ": $banner->{afio} ($banner->{alogin})" : '') .
                         ($banner->{mlogin} ? iget('Менеджер')  . ": $banner->{mfio} ($banner->{mlogin})" : '');
            }
            $value //= '';
            push @row, $value;
        }
        push @data, \@row;
    }

    my %column_width = (title => 40, body => 40, cid => 15, bid => 15, href => 20, geo_text => 40, login => 40, camp_status => 20, 
                        mafio => 50, camp_name => 40, flags => 40);
    my $format = {
        sheetname=> 'banners',
        set_column => [map { { col1 => $_, count => 0, width => $column_width{ $fields[$_] } } } 0..$#fields],
    };
    my $xls = Yandex::ReportsXLS->new(compatibility_mode => 1);
    my $xls_data = $xls->array2excel2scalar([\@data], [$format]);
    return $xls_data;
}


sub get_letter_text_and_subject {
    my ($FORM, $host, $data) = @_;

    my $url = $host . '?' . join('&', map { "$_=" . uri_escape_utf8($FORM->{$_}) } 
                                      grep { $_ ne 'send_xls_report_by_email' } 
                                      sort keys %$FORM); 

    my $text = "$url\n";
    if ($data->{parts_count} > 1 && ($data->{first_part} > 1 || $data->{last_part} < $data->{parts_count})) { 
        $text .= ($data->{first_part} eq $data->{last_part} 
                     ? sprintf(iget('часть %s из %s'), $data->{first_part}, $data->{parts_count})
                     : sprintf(iget('части %s..%s из %s'), @$data{qw/first_part last_part parts_count/}));
    }

    my $subject = iget('Результат поиска баннеров (cmd=searchBanners)');
    return ($text, $subject);
}


sub send_mail_to_manager {
    my ($manager_email, $xls_data, $text, $subject, $part) = @_;

    my $from = sprintf('Yandex.Direct <%s>', $Settings::DEFAULT_FROM_FIELD_DEPENDING_ON_LANG->{'ru'});
    my $attaches = $xls_data ? [{
         Type     => 'application/vnd.ms-excel',
         Filename => join('', 'search_banners_', strftime("%Y%m%d%H%M%S", localtime), "_$part", '.xls'),
         Data     => $xls_data,
    }] : [];
    sendmail_attach($manager_email, $from, $subject, $text, $attaches);
}
