#!/usr/bin/perl

use my_inc '..';

=head1 METADATA

<crontab>
    time: */30 * * * *
    sharded: 1
    ulimit: -v 16000000
    <switchman>
        group: scripts-other
        <leases>
            mem: 16000
        </leases>
    </switchman>
    package: scripts-switchman
</crontab>

<juggler>
    host:   checks_auto.direct.yandex.ru
    sharded: 1
    ttl: 2h
    tag: direct_group_internal_systems
</juggler>

=cut

=head1 NAME

    ppcSendSubclientsConvertedMails.pl

=head1 DESCRIPTION

    Отправляет письмо главному представителю агентства по окончанию конвертации
    всех субклиентов агентства, заказанных через интерфейс заказа массовой
    конвертации [cmd=massMoveToRealMoney].
    Список заказанных клиентов берёт из таблицы subclient_convert_requests.
    После отправки письма запись из таблицы удаляет.

=head1 RUNNING

    Можно запускать по конкретным агентствам:
        --login — логин одного из представителей агентства
        --clientid — ClientID агентства

    ./protected/ppcSendSubclientsConvertedMails.pl --shard-id 4
    ./protected/ppcSendSubclientsConvertedMails.pl --shard-id 4 --login itagency
    ./protected/ppcSendSubclientsConvertedMails.pl --shard-id 4 --clientid 475089

=cut

use Direct::Modern;

use Settings;
use ScriptHelper sharded => 1, 'Yandex::Log' => 'messages';

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::Validate qw(is_valid_id);
use Yandex::ListUtils qw(xflatten chunks xminus);

use Client qw(mass_get_client_currencies);
use Notification qw(add_notification);
use PrimitivesIds qw(get_login2clientid get_uid2login);
use RBAC2::Extended ();
use RBACDirect qw(rbac_get_subclients_clientids);
use RBACElementary qw(rbac_get_chief_rep_of_agency rbac_get_chief_reps_of_clients);
use User qw(get_one_user_field);

my $RBAC_OPERATOR_UID = 1;

my (@agency_ids, @logins);
extract_script_params(
    'login=s@' => \@logins,
    'clientid=s@' => \@agency_ids,
);
@agency_ids = @{ _parse_validate_params(\@logins, \@agency_ids) };


$log->out('START');

if (@agency_ids) {
    $log->out('Working only on agency ClientIDs:', \@agency_ids);
}

$log->out('Fetching unprocessed subclient convert requests');
my @where;
if (@agency_ids) {
    @where = (WHERE => {
        'scr.AgencyID' => \@agency_ids,
    });
}
my $requests = get_all_sql(PPC(shard => $SHARD), ['
    SELECT scr.AgencyID
         , scr.ClientID
         , ccq.state -- если клиент в одном шарде с агентством, то не потребуется второй запрос
    FROM subclient_convert_requests scr
    LEFT JOIN currency_convert_queue ccq ON ccq.ClientID = scr.ClientID
 ', @where, '
    ORDER BY scr.AgencyID, scr.ClientID
']);
$log->out('Got ' . scalar(@$requests) . ' unprocessed subclient convert requests');

$log->out('Collecting clients to fetch state');
my (%agencyid2clientid2state, %agencyid2clientids_to_fetch_state, %known_uncomplete_agencyids);
for my $request (@$requests) {
    my $client_id = $request->{ClientID};
    my $agency_id = $request->{AgencyID};
    my $state = $request->{state};

    next if exists $known_uncomplete_agencyids{ $agency_id };

    $agencyid2clientid2state{ $agency_id }->{ $client_id } = $state;
    if (defined $state) {
        if ($state ne 'DONE') {
            $known_uncomplete_agencyids{ $agency_id } = undef;
            delete $agencyid2clientids_to_fetch_state{ $agency_id };
            $log->out("Skipping AgencyID $agency_id: ClientID $client_id has not been converted yet");
        }
    } else {
        push @{ $agencyid2clientids_to_fetch_state{ $agency_id } }, $client_id;
    }
}
undef $requests;
undef %known_uncomplete_agencyids;

my @clientids_to_fetch_state = xflatten values %agencyid2clientids_to_fetch_state;
undef %agencyid2clientids_to_fetch_state;

$log->out('Fetching states for ' . scalar(@clientids_to_fetch_state) . ' clients');
my $clientid2fetched_state = get_hash_sql(PPC(ClientID => \@clientids_to_fetch_state), ['
    SELECT ccq.ClientID
         , ccq.state
    FROM currency_convert_queue ccq
 ', WHERE => {
        'ccq.ClientID' => SHARD_IDS,
    },
]);
undef @clientids_to_fetch_state;

$log->out('Checking agencies for notifications');
AGENCY: for my $agency_id (keys %agencyid2clientid2state) {
    my @client_ids = keys %{ $agencyid2clientid2state{$agency_id} };
    for my $client_id (@client_ids) {
        my $state = $agencyid2clientid2state{$agency_id}->{$client_id} // $clientid2fetched_state->{$client_id};
        if (defined $state) {
            if ($state ne 'DONE') {
                $log->out("ClientID $client_id is not yet converted, skipping AgencyID $agency_id");
                next AGENCY;
            }
        } else {
            $log->out("WARNING: Got no state for ClientID $client_id and AgencyID $agency_id. This is strange, as it should be queued for converting. Skipping agency.");
            next AGENCY;
        }
    }

    $log->out("Sending notification for AgencyID $agency_id after converting ClientIDs:", \@client_ids);
    my $notify_success = eval {
        _notify_agency($agency_id, \@client_ids);
        1;
    };
    if ($notify_success) {
        $log->out("Sent notification for AgencyID $agency_id");
        do_delete_from_table(PPC(ClientID => $agency_id), 'subclient_convert_requests', where => {
            AgencyID => $agency_id,
            ClientID => \@client_ids,
        });
        $log->out("Deleted records for AgencyID $agency_id");
    } else {
        $log->out("Error notifying AgencyID $agency_id, skipping: $@");
    }
}

juggler_ok();

$log->out('FINISH');



=head3 _parse_validate_params

    @agency_ids = @{ _parse_validate_params(\@logins, \@agency_ids) };

=cut

sub _parse_validate_params {
    my ($logins, $agency_ids) = @_;

    if (@$agency_ids) {
        for my $agency_id (@$agency_ids) {
            if (!is_valid_id($agency_id)) {
                die "invalid ClientID $agency_id given in command line";
            }
        }
    }
    if (@$logins) {
        my $login2clientid = get_login2clientid(login => $logins);
        for my $login (@$logins) {
            if (exists $login2clientid->{$login}) {
                push @$agency_ids, $login2clientid->{$login};
            } else {
                die "no ClientID found for login $login in command line";
            }
        }
    }
    return $agency_ids;
}

=head3 _notify_agency

    _notify_agency($agency_id, \@converted_client_ids);

=cut

sub _notify_agency {
    my ($agency_id, $converted_client_ids) = @_;

    my $rbac = RBAC2::Extended->get_singleton($RBAC_OPERATOR_UID);
    my $agency_chief_uid = rbac_get_chief_rep_of_agency($agency_id);

    my $all_clids = rbac_get_subclients_clientids($rbac, $agency_chief_uid);
    my $unconverted_client_ids = xminus $all_clids, $converted_client_ids;
    undef $all_clids;

    my @unconverted_clients;
    if ($unconverted_client_ids) {
        my $unconverted_clid2currencies = mass_get_client_currencies($unconverted_client_ids);
        $unconverted_client_ids = [ grep { $unconverted_clid2currencies->{$_}->{work_currency} eq 'YND_FIXED' } @$unconverted_client_ids ];
        undef $unconverted_clid2currencies;

        my $unconverted_clid2chief_uid = (@$unconverted_client_ids) ? rbac_get_chief_reps_of_clients($unconverted_client_ids) : {};
        my $unconverted_chief_uid2login = get_uid2login(uid => [values %$unconverted_clid2chief_uid]);
        for my $client_id (@$unconverted_client_ids) {
            my $chief_uid = $unconverted_clid2chief_uid->{$client_id};
            my $chief_login = $unconverted_chief_uid2login->{$chief_uid};
            push @unconverted_clients, {
                ClientID => $client_id,
                chief_login => $chief_login,
            };
        }
    }
    undef $unconverted_client_ids;

    my $agency_chief_fio = get_one_user_field($agency_chief_uid, 'fio');

    my $vars = {
        agency_id => $agency_id,
        agency_chief => {
            uid => $agency_chief_uid,
            FIO => $agency_chief_fio,
        },
        converted_clients_cnt => scalar(@$converted_client_ids),
        unconverted_clients_cnt => scalar(@unconverted_clients),
        unconverted_clients => \@unconverted_clients,
    };

    add_notification($rbac, 'subclients_convert_finished', $vars);
}
