#!/usr/bin/perl

use my_inc "..";

=head1 DESCRIPTION

Скрипт отправляет уведомления о понижении позиции

=head2 CAVEATS

    перед тем, как искать почему не было отослано уведомление о понижении позиции в этом скрипте следует убедиться что:
    1. У клиента не включён временной таргетинг (тем более расширенный) 
    2. У клиента верные настройки по поводу доставки уведомлений (как на кампании так и в параметрах пользователя)
    3. Клиент не пользуется API для редактирования кампании
    4. Обратить внимание, если у клиента не стратегия по умолчанию

=head1 METADATA

<crontab>
    params: --run-type=15
    time: 9,24,39,54 * * * *
    sharded: 1
    <switchman>
        group: scripts-other
        <leases>
            mem: 280
        </leases>
    </switchman>
    package: scripts-switchman
</crontab>
<crontab>
    params: --run-type=30
    time: 6,36 * * * *
    sharded: 1
    <switchman>
        group: scripts-other
        <leases>
            mem: 230
        </leases>
    </switchman>
    package: scripts-switchman
</crontab>
<crontab>
    params: --par-id=0 --run-type=60
    time: 3 * * * *
    sharded: 1
    <switchman>
        group: scripts-other
        <leases>
            mem: 380
        </leases>
    </switchman>
    package: scripts-switchman
</crontab>
<crontab>
    params: --par-id=1 --run-type=60
    time: 18 * * * *
    sharded: 1
    <switchman>
        group: scripts-other
        <leases>
            mem: 380
        </leases>
    </switchman>
    package: scripts-switchman
</crontab>
<crontab>
    params: --par-id=2 --run-type=60
    time: 33 * * * *
    sharded: 1
    <switchman>
        group: scripts-other
        <leases>
            mem: 380
        </leases>
    </switchman>
    package: scripts-switchman
</crontab>
<crontab>
    params: --par-id=3 --run-type=60
    time: 48 * * * *
    sharded: 1
    <switchman>
        group: scripts-other
        <leases>
            mem: 380
        </leases>
    </switchman>
    package: scripts-switchman
</crontab>
<juggler>
    host:   checks_auto.direct.yandex.ru
    raw_events:     scripts.ppcSendWarnPlace.working.$queue.shard_$shard
    sharded:        1
    vars:           queue<ttl=50m>=15_0
    vars:           queue<ttl=1h40m>=30_0
    vars:           queue<ttl=2h30m>=60_0,60_1,60_2,60_3
    tag: direct_group_internal_systems
</juggler>

<crontab>
    params: --run-type=15
    time: 9,24,39,54 * * * *
    package: scripts-sandbox
    only_shards: 1
    sharded: 1
</crontab>
<juggler>
    host:   checks_auto.direct.yandex.ru
    name:           scripts.ppcSendWarnPlace.working.sandbox
    raw_host:       CGROUP%direct_sandbox
    raw_events:     scripts.ppcSendWarnPlace.working.sandbox.$queue.shard_$shard
    vars:           shard=1
    vars:           queue=15_0
    ttl:            50m
    tag: direct_group_internal_systems
</juggler>

=head1 NAME

$Id$

=cut

use warnings;
use strict;

use EV;
use POSIX qw(strftime);

use Yandex::DBShards;
use Yandex::DBTools;
use Yandex::HashUtils;

use Settings;
use ScriptHelper sharded => 1, get_file_lock => undef, 'Yandex::Log' => 'messages';

use BS::TrafaretAuction;
use EnvTools;
use PlacePrice;
use LockTools;
use TextTools;
use Primitives;
use TimeTarget;
use Notification;
use RBAC2::Extended;
use AdGroupTools;
use Models::AdGroup ();
use Campaign;

use Property;

### NB! при изменении числа потоков - не забудь поменять crontab и juggler-проверки! ###
# для каждой периодичности запуска - число работающих чайлдов
my %MAXCHILD = (15 => 1, 30 => 1, 60 => 4 );

use constant SLEEP         => 5;
use constant DEBUG         => 0;
use constant SQL_CHUNK_SIZE => 10_000;

# при использовании групп, по какому проценту баннеров фраза должна выпасть из позиции для отправления предупреждения (DIRECT-36089)
use constant THRESHOLD_BANNERS_IN_ADGROUP => 70;

# ходим в торги по такому количеству фраз (возможно больше, делим дополнительно по группам объявлений)
use constant PROCESS_MAX_PHRASES => 5_000;

$0 =~ m!([^/]+)$!;
my $name = $1;
my $debug_cids;

# обрабатываем параметры

my ($PAR_ID, $RUN_TYPE);
extract_script_params(
    'par-id=i' => \$PAR_ID,
    'run-type=i' => \$RUN_TYPE,
    'cid=s' => \$debug_cids,
);
my $cid_shard = get_shard_multi(cid => [split /\D/, $debug_cids || '']);

if (!$RUN_TYPE || !exists $MAXCHILD{$RUN_TYPE}) {
    die sprintf("Usage: %s [--par-id=NNN] --run-type=%s", $name, join('|', keys %MAXCHILD));
}

if (!defined $PAR_ID && $MAXCHILD{$RUN_TYPE} > 1) {
    die "par-id is not defined";
}
$PAR_ID = 0 if !defined $PAR_ID && $MAXCHILD{$RUN_TYPE} == 1;
if ($PAR_ID !~ /^\d+$/ || $PAR_ID >= $MAXCHILD{$RUN_TYPE}) {
    die "Incorrect par-id";
}
my $WARNINGS_TABLE = "warnings_$RUN_TYPE".($MAXCHILD{$RUN_TYPE} > 1 ? "_$PAR_ID" : '');

# лочимся, создаём лог-объект
get_file_lock(0, "${name}_${RUN_TYPE}_${PAR_ID}_${SHARD}");
my $trace = Yandex::Trace->new(service => 'direct.script', method => $name, tags => "run_type_$RUN_TYPE,par_id_$PAR_ID,shard_$SHARD");
$log->msg_prefix("[shard_$SHARD,run_type_$RUN_TYPE,par_id_$PAR_ID]");
$log->out('start');

# Новая заливка пишет данные пошардово, если этих данных нет - используем пропертю от старой заливки
my $time_to_sync = Property->new('bsStatTime_shard_'.$SHARD)->get();
if (!defined $time_to_sync) {
    $log->out("Get time_to_sync from old property");
    $time_to_sync = Property->new('bsStatTime_0')->get();
}
$log->out("time_to_sync: $time_to_sync");
$log->die("Incorrect time_to_sync: '$time_to_sync'") if $time_to_sync !~ /^\d{14}$/;
my $laststat_prop = Property->new("ppcSendWarn_last_shard$SHARD" . '_' . $RUN_TYPE . '_' . $PAR_ID);
my $laststat = $laststat_prop->get() || Property->new("ppcSendWarn_last_shard$SHARD" . '_'  . $RUN_TYPE)->get() || strftime('%Y%m%d%H%M%S', localtime);

collect_warnings();

$laststat_prop->set($time_to_sync);

send_notification();

my $sandbox_suffix = is_sandbox() ? 'sandbox.' : '';
juggler_ok(service_suffix => "$sandbox_suffix${RUN_TYPE}_${PAR_ID}");

$log->out('finish');

################################################################################
# Обработка всех фраз одного банера -
#   заносим предупреждения в warnings и warnplace
sub process_bids($) {
    my $arr = shift;
    my $result = [];

    return [] unless @$arr;

    my $cid;
    my %banners;
    for my $ph (@$arr) {
        $banners{$ph->{bid}} ||= hash_cut
                                   $ph,
                                   qw/title body OrderID geo filter_domain phone currency pid
                                      cid adgroup_type statusShowsForecast no_extended_geotargeting
                                      banner_type is_bs_rarely_loaded/;
        push @{$banners{$ph->{bid}}->{phrases}}, $ph;
        $cid = $ph->{cid} unless defined $cid; # все фразы принадлежат одному баннеру, как следствие - одной кампании
    }

    my $banners = [values %banners];
    Models::AdGroup::update_phrases_shows_forecast($banners);
    trafaret_auction($banners);

    for my $p ( @$arr ) {
        next if $p->{nobsdata};
        my $new_place = calcPlace( $p->{price}, $p->{guarantee}, $p->{premium} );
        $new_place = PlacePrice::get_warnplace_value($new_place);

        # если включена стратегия "показ справа" (DIRECT-12930)
        if ($p->{strategy} eq 'strategy_no_premium') {
            my $strategy_place = $p->{strategy_decoded}->{place} || '';
            if (
                $strategy_place eq 'highest_place'
                && $p->{place} != $PlacePrice::PLACES{GUARANTEE1}
                && $p->{place} != PlacePrice::get_guarantee_entry_place()
            ) {
                $log->out("gap: cid=$p->{cid} ph_id=$p->{id} strategy_no_premium place isn't guarantee");
                next;
            }
        }

        my $result_row = hash_cut $p, qw/uid cid bid id Clicks Shows ManagerUID AgencyUID place pid group_name/;
        $result_row->{new_place} = $new_place;
        push @$result, $result_row;
    }

    return $result;
}

sub collect_warnings {

    $log->out('start query');

    my $predefined_cids_sql = '';
    if ($debug_cids && $debug_cids =~ /\d/) {
        $predefined_cids_sql = join ",", map {sql_quote($_)} split /\D/, $debug_cids;
        $predefined_cids_sql = "and c.cid in ($predefined_cids_sql)";
        $laststat = '20050101000000';
    }
    
    # за счёт FORCE KEY достигаем сортировки по cid, bid
    my $sth = exec_sql(PPC(shard => $SHARD), "select STRAIGHT_JOIN 
                                                bi.id, b.bid, bi.norm_phrase, p.geo, c.cid, p.pid,
                                                bi.place, bi.phrase, bi.price, bph.phraseIdHistory,
                                                auct.Rank, auct.Shows, auct.Clicks,
                                                c.uid, c.ManagerUID, c.AgencyUID,
                                                c.OrderID, p.PriorityID, b.BannerID, bi.PhraseID,
                                                b.domain, vc.phone, b.title, b.body, b.banner_type,
                                                ifnull(fd.filter_domain, b.domain) filter_domain,
                                                c.autobudget,
                                                c.platform,
                                                IFNULL(s.type, '') as strategy_name, s.strategy_data,
                                                c.timeTarget, c.timezone_id,
                                                IFNULL(c.currency, 'YND_FIXED') AS currency,
                                                p.group_name, p.adgroup_type, p.is_bs_rarely_loaded
                                              , p.statusShowsForecast
                                              , bi.showsForecast
                                              , FIND_IN_SET('no_extended_geotargeting', c.opts)>0 as no_extended_geotargeting
                                             from
                                                campaigns c FORCE INDEX (PRIMARY)
                                                left join campaigns wc on c.wallet_cid = wc.cid
                                                left join camp_options co on co.cid=c.cid
                                                left join strategies s on c.strategy_id = s.strategy_id
                                                join phrases p on p.cid = c.cid
                                                join bs_auction_stat auct on auct.pid = p.pid
                                                join bids bi on bi.pid = p.pid and bi.PhraseID = auct.PhraseID
                                                join banners b on b.pid = p.pid
                                                left join filter_domain fd on fd.domain = b.domain
                                                left join vcards vc on vc.vcard_id = b.vcard_id
                                                left join bids_phraseid_history bph on bph.cid = bi.cid and bph.id = bi.id
                                             where
                                                (c.cid % " . $MAXCHILD{$RUN_TYPE} . " = " . $PAR_ID . ")
                                                and c.type='text'
                                                and auct.stattime > " . $laststat . "
                                                and b.banner_type = 'text'
                                                and b.statusActive = 'Yes'
                                                and c.statusShow='Yes'
                                                and c.autobudget='No'
                                                and c.sum - c.sum_spent + IF(c.wallet_cid, wc.sum - wc.sum_spent, 0) >= 0.01
                                                and c.start_time < now()
                                                and ifnull(auct.rank, 1) > 0
                                                and bi.warn='Yes'
                                                and bi.place > 0
                                                and bi.is_suspended = 0
                                                and ifnull(co.warnPlaceInterval, 60) = " . $RUN_TYPE . " 
                                                $predefined_cids_sql
                                             ");

    my @warnings = ();
    while (my $warn_place_data = $sth->fetchall_arrayref({}, SQL_CHUNK_SIZE)) {
        $log->out("got chunk of ".scalar(@$warn_place_data)." items");
        $log->out('start bs queries');
        my @to_process = ();

        # Обрабатываем по одному банеру
        for my $r (@{$warn_place_data}) {
            next if !defined $r->{price} || !defined $r->{place};
            if (TimeTarget::is_extended_timetarget($r->{timeTarget})) {
                $log->out("gap: cid=$r->{cid} has extended timetarget");
                next;
            }
             
            # пропускаем кампании которые сейчас отключены по временному таргетингу
            if (TimeTarget::timetarget_current_coef($r->{timeTarget}, $r->{timezone_id}) == 0) {
                $log->out("gap: cid=$r->{cid} stoped by timetarget");
                next;    
            }

            Campaign::_deserialize_camp_fields($r);

            $r->{place} = PlacePrice::set_new_place_style($r->{place});
            $r->{place} = PlacePrice::get_warnplace_value($r->{place});

            $r->{strategy} = detect_strategy($r);

            if (@to_process >= PROCESS_MAX_PHRASES && $to_process[-1]->{pid} != $r->{pid}) {
                # обстукиваем торги кусками примерно по N фраз
                push @warnings, @{process_bids(\@to_process)};
                @to_process = ();
            }
            push @to_process, $r;
        }
        push @warnings, @{process_bids(\@to_process)};
        
        last if @$warn_place_data < SQL_CHUNK_SIZE;
    }

    if (@warnings) {
        # для баннеров в группах, высылаем уведомление, если более 70% объявлений не могут быть показаны на выбранной позиции.
        my %skip_adgroup_pids;
        my %group_by_adgroup;
        for my $row (@warnings) {
            push @{$group_by_adgroup{$row->{pid}}}, $row;
        }
        for my $pid (keys %group_by_adgroup) {
            my $count_all_banners = scalar @{$group_by_adgroup{$pid}};
            my $count_warnings_banners = scalar grep {$_->{new_place} > $_->{place} || $_->{new_place} == 0} @{$group_by_adgroup{$pid}};
            $skip_adgroup_pids{$pid} = 1 if $count_warnings_banners / $count_all_banners * 100 <= THRESHOLD_BANNERS_IN_ADGROUP; # процент маленький, такие группы пропускаем
        }

        # посылаем уведомление, если используются группы (и позиция упала у более 70% объявлений),
        # если группы не используются, то должна измениться позиция (новая > старой)
        @warnings = grep {! $skip_adgroup_pids{$_->{pid}} && $_->{new_place} > $_->{place} || $_->{new_place} == 0} @warnings;        
        $log->out('gap: pids skipped by 70% threshold ' . join ',', keys %skip_adgroup_pids) if keys %skip_adgroup_pids;

        # inserts data
        my @tbl_warnings = sort {$a->[0] <=> $b->[0]}
                           map {[$_->{id}, $_->{place}]}
                           @warnings;
        do_mass_insert_sql(PPC(shard => $SHARD), "INSERT /* no_readonly */ IGNORE into $WARNINGS_TABLE (id, old_place) VALUES %s", \@tbl_warnings, {sleep => 1});
        while (@tbl_warnings) {
            do_sql(PPC(shard => $SHARD), sprintf("UPDATE bids set warn = 'No', modtime = modtime where id in (%s)", join(",", map {$_->[0]} splice(@tbl_warnings, 0, 5000))));
            sleep 1;
        }

        my @tbl_warnplace = map {[@{$_}{qw/uid cid pid bid id Clicks Shows ManagerUID AgencyUID new_place place/}]} @warnings;
        do_mass_insert_sql(PPC(shard => $SHARD), "insert into warnplace (uid, cid, pid, bid, id, clicks, shows, manageruid, agencyuid, statusPlace, old_place)
                                  values %s
                                  on duplicate key update
                                  uid = values(uid)
                                  , cid = values(cid)
                                  , pid = values(pid)
                                  , bid = values(bid)
                                  , clicks = values(clicks)
                                  , shows = values(shows)
                                  , manageruid = values(manageruid)
                                  , agencyuid = values(agencyuid)
                                  , statusPlace = values(statusPlace)
                                  , old_place = values(old_place)
                                  , done = 'No'",
                                  \@tbl_warnplace, {sleep => 1});
    }
    return 1;
}

sub send_notification {
    $log->out('start warnplace query');
    my $sth = exec_sql(PPC(shard => $SHARD),
                             "SELECT c.cid
                                     , c.name
                                     , b.bid
                                     , b.title
                                     , w.old_place
                                     , bi.id AS bids_id
                                     , bi.phrase
                                     , w.id
                                     , u.fio
                                     , u.uid
                                     , u.sendWarn
                                     , u.ClientID
                                     , p.pid
                                     , p.group_name
                              FROM $WARNINGS_TABLE w
                                JOIN bids bi ON bi.id = w.id
                                JOIN campaigns c ON c.cid = bi.cid
                                LEFT JOIN campaigns wc on c.wallet_cid = wc.cid
                                JOIN banners b ON b.pid = bi.pid
                                JOIN phrases p ON p.pid = b.pid
                                JOIN users u ON u.uid = c.uid
                              WHERE c.statusShow = 'Yes'
                                and b.banner_type = 'text'
                                and c.sum - c.sum_spent + IF(c.wallet_cid, wc.sum - wc.sum_spent, 0) >= 0.01
                              "
                       );
    
    my (%CAMP, @IDs);
    while (my $row = $sth->fetchrow_hashref()) {
        push(@IDs, $row->{id});
        my $cid = $row->{cid};

        unless (defined $CAMP{$cid}) {
            $CAMP{$cid} = {
                name       => $row->{name}
                , uid      => $row->{uid}
                , valid    => $row->{valid}
                , fio      => $row->{fio} || ''
                , ClientID => $row->{ClientID}
            };
        }
        unless (defined $CAMP{$cid}->{bids}->{$row->{bid}}) {
            $CAMP{$cid}->{bids}->{$row->{bid}} = {
                name => html2string($row->{title})
                , pid => $row->{pid}
                , group_name => $row->{group_name}
            };
        }
        push(@{$CAMP{$cid}->{bids}->{$row->{bid}}->{PlacePrice::set_new_place_style($row->{old_place})}}, {
            phrase => $row->{phrase}
            , id => $row->{bids_id}
        });
    }
    $sth->finish;
    truncate_warnings(PPC(shard => $SHARD), $WARNINGS_TABLE);

    my $cids = [ keys %CAMP ];
    my $universal_campaign_cids = mass_is_universal_campaign($cids);

    my $rbac = RBAC2::Extended->get_singleton(1);

    $log->out('start send warnplace mails');
    foreach my $cid (keys %CAMP) {

        # игнорируем нотификации для универсальных кампаний
         if ($universal_campaign_cids->{$cid}) {
            $log->out("skip warn_place notification about universal campaign $cid to UID $CAMP{$cid}->{uid}");
            next;
         }

        my ($gar, $spec) = (0, 0);
        my $mailvars = {
            cid => $cid,
            client_uid => $CAMP{$cid}->{uid},
            client_fio => $CAMP{$cid}->{fio},
            fio => $CAMP{$cid}->{fio},
            camp_name => $CAMP{$cid}->{name},
            ClientID => $CAMP{$cid}->{ClientID},
        };
    
        foreach my $bid (keys %{$CAMP{$cid}->{bids}}) {
            my $banner = $CAMP{$cid}->{bids}->{$bid};
            push @{$mailvars->{banners}}, {
                bid => $bid
                , banner_name => $banner->{name}
                , pid => $banner->{pid}
                , group_name => $banner->{group_name}
            };
            for (@{$banner->{PlacePrice::get_premium_entry_place()}}) {
                push @{ $mailvars->{banners}[-1]{ph_spec} }, $_;
                $spec++;
            }
            for (@{$banner->{PlacePrice::get_guarantee_entry_place()}}) {
                push @{ $mailvars->{banners}[-1]{ph_gar} }, $_;
                $gar++;
            }
        }
    
        $mailvars->{spec} = $spec;
        $mailvars->{gar} = $gar;

        AdGroupTools::separate_adgroups_from_banners_struct($mailvars, keep_one_banner => 1);
        my $res = eval {
            add_notification($rbac, 'warn_place', $mailvars);
            return 1;
        };
        if (!$res || $@) {
            $log->out("warn_place about $cid to UID $mailvars->{client_uid} FAILED: $@");
        } else {
            $log->out("warn_place about $cid to UID $mailvars->{client_uid}");
        }
    }
    return 1;
}

=head2 truncate_warnings

    Не используем truncate, чтобы в БД было поменьше DDL
    https://st.yandex-team.ru/DIRECT-126384

=cut
sub truncate_warnings {
    my ($db, $tbl) = @_;
    my $chunk_size = 10;
    while(my @ids = @{get_one_column_sql($db, "SELECT id FROM $tbl LIMIT $chunk_size")}) {
        do_delete_from_table($db, $tbl, where => {id => \@ids});
        last if @ids < $chunk_size;
    }
}
