#!/usr/bin/perl

use my_inc "..";



=head1 METADATA

<crontab>
    time: */2 * * * *
    package: monitoring
</crontab>

=cut

=head1 DESCRIPTION

    Собирает из логов данные по 500 ошибкам и записывает в файлы для мониторинга (по одному файлу на каждый хост + суммарный) количество уникальных uid'ов, у которых была такая ошибка
    На созданные файлики смотрит запускаемый из monrun'а скрипт protected/maintenance/internal_errors.sh, см. etc/monitoring/monrun/conf.d/internal_errors.conf

    По API собираем ошибки только для публично доступных версий. Берем список из Settings::AVAILABLE_API_VERSIONS

    TODO: отказаться от файлов, скрипта и кастомного конфига monrun и переделать на POD-секцию monrun + juggler-проверку на последний успешный запуск

=cut

use strict;
use warnings;
use utf8;

use ScriptHelper;

use Settings;
use Tools;

use Yandex::TimeCommon;
use Yandex::Advmon;

use File::Slurp;

my $PERIOD_MINUTES = 20; # minutes
my $LOOK_BACK_SECONDS = $PERIOD_MINUTES * 60;

my $NUMFILE = "$Settings::MONITOR_ROOT/server_errors.num";

my $clh = get_clickhouse_handler('cloud');

$log->out('START');

# собираем статистику глобально и по каждому серверу
my %stat;
my %monitor_stat;

$log->out('Collecting stat about frontend errors');
fill_cmd_stat_from_clickhouse(\%stat);

# собираем данные по ошибкам для API
$log->out('Collecting stat about API errors');
fill_api_stat_from_clickhouse(\%stat);

$log->out('Statistics collected:', \%stat);

# выводим информацию в файлы
$log->out('Writing stat to files');
for my $host ('', "API", direct_frontends()) {
    my $host_stat = $stat{$host} || {};
    my ($errnum, @msg) = (0);
    ## здесь у Freenode::DollarAB ложное срабатывание
    ## no critic (Freenode::DollarAB)
    for my $cmd (sort {scalar(keys %{$host_stat->{$b}}) <=> scalar(keys %{$host_stat->{$a}})} keys %$host_stat) {
        my $errors = keys %{ $host_stat->{$cmd} };
        $errnum += $errors;
        push @msg, "$cmd: $errors";
    }

    $log->out("Error count for host '$host' is $errnum. Error types:", \@msg);

    # записываем в файл
    write_file($NUMFILE.($host ? ".$host" : ''), {atomic => 1}, "$errnum\n".join(', ', @msg));

}

# и в графит
$log->out('Writing stat to graphite');
local $Yandex::Advmon::GRAPHITE_PREFIX = sub {[qw/direct_one_min db_configurations/, $Settings::CONFIGURATION]};
monitor_values({
    flow => { ServerErrors => \%monitor_stat },
});

$log->out('FINISH');

sub _host2graphite {
    my ($host) = @_;

    $host =~ s/\./_/g;
    return $host;
}

sub fill_cmd_stat_from_clickhouse {
    my $stat = shift;

    _fill_some_stat_from_clickhouse($stat,
        "SELECT distinct cmd, host, uid
             FROM ppclog_cmd
             WHERE cmd not in ('die', 'ajaxSpellCheck')
               AND http_status=500
               AND log_time >= (now() - $LOOK_BACK_SECONDS)
            FORMAT JSON",
        sub {
            my ($stat, $cmd, $host, $uid) = @_;
            $stat{$host}{$cmd}{$uid} = $stat{''}{$cmd}{$uid} = undef;
        });

    return;
}

sub fill_api_stat_from_clickhouse {
    my $stat = shift;

    _fill_some_stat_from_clickhouse(
        $stat,
        "SELECT DISTINCT cmd, host, uid
                FROM ppclog_api
                WHERE log_date >= yesterday()
                    AND api_version in (104,5)
                    AND http_status in (500, 358)
                    AND log_time >= (now() - $LOOK_BACK_SECONDS)
                FORMAT JSON",
        sub {
            my ($stat, $cmd, $host, $uid) = @_;
            $stat{ "" }->{ "api_" . ($cmd||'') }->{ $uid } =
                $stat{ $host }->{ "api_" . ($cmd||'') }->{ $uid } =
                $stat{ "API" }->{ "api_" . ($cmd||'') }->{ $uid } = undef;
        }
    );

    return;
}


sub _fill_some_stat_from_clickhouse {
    my ($stat, $sql, $fill_sub) = @_;

    my $look_back_seconds = $PERIOD_MINUTES * 60;

    my $result = $clh->query($sql);
    for my $row ( @{ $result->json->{data} } ) {
        my ($cmd, $host, $uid) = @$row{qw/cmd host uid/};
        my $host_graphite = _host2graphite($host);
        $monitor_stat{errors}{$cmd || 'UNKNOWN'}{$host_graphite}++;
        if (!exists $stat{$host}{ "api_" . ($cmd||'') }{$uid}) {
            $monitor_stat{uids_with_error}{$cmd || 'UNKNOWN'}{$host_graphite}++;
        }

        $fill_sub->($stat, $cmd, $host, $uid);
    }
    return;
}


