#!/usr/bin/perl

use my_inc "..";

=head1 METADATA

<crontab>
    sharded: 1
    time: */5 * * * *
    <switchman>
        group: scripts-other
    </switchman>
    package: scripts-switchman
</crontab>

<juggler>
    host:   checks_auto.direct.yandex.ru
    sharded:    1
    ttl:        30m
    tag: direct_group_internal_systems
</juggler>


=cut

=head1 DESCRIPTION

Скрипт для простановки и снятия авто-видео для кампаний.
Принимает параметры:
    --shard-id <номер шарда>
    --clientid <clientid клиента>
    --cid <номер кампании> - создает задание в очереди и запускает обработку для нужного ClientID
    --action <set|reset> - действие для указанного cid. по-умолчанию 'set'
    --creative_id - при создании задания (если указан --cid) указать creative_id

./protected/ppcSetAutoResources.pl —shard-id 1 —clientid 4528876

=cut

use Direct::Modern;

use Yandex::DBQueue;
use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::Retry qw/relaxed_guard/;
use Yandex::ListUtils qw/chunks nsort/;
use Yandex::I18n;
use Direct::AdGroups2;
use Direct::VideoAdditions;
use Direct::Validation::BannersPerformance qw/validate_video_additions_layout_id/;
use List::MoreUtils qw/uniq/;
use JSON;

use Direct::Banners;

use Campaign::Types qw/get_camp_type/;
use Primitives qw/filter_archived_campaigns/;
use PrimitivesIds qw/get_clientid get_uid/;

use Settings;
use ScriptHelper sharded => 1, 'Yandex::Log' => 'messages';


my $TRYCOUNT = 3;
my $JOB_GRAB_FOR = 2 * 60; # в случае ошибки оставляем задачу в Grabbed, таким образом $JOB_GRAB_FOR это пауза при ошибке
my $JOBS_LIMIT = 10;
my $SLEEP_COEF = 1;
my ($CID, $ACTION, $CREATIVE_ID);

extract_script_params(
    'sleep-coef=f' => \$SLEEP_COEF,
    'client-id|clientid=s' => \my $client_id,
    'cid=i' => \$CID,
    'action=s' => \$ACTION,
    'creative_id=i' => \$CREATIVE_ID,
);

$log->out("START");

$Yandex::DBQueue::LOG = $log;
my $queue = Yandex::DBQueue->new(PPC(shard => $SHARD), 'set_auto_resources');

if ($CID) {
    $ACTION //= 'set';
    $client_id = get_clientid(cid => $CID);
    my $job = $queue->insert_job({
        job_id => get_new_id('job_id'),
        ClientID => $client_id,
        uid => get_uid(cid => $CID),
        args => { cid => $CID, action => $ACTION, ($CREATIVE_ID ? (creative_id => $CREATIVE_ID) : ()) },
    });
    $log->out("created job @{[$job->job_id]} for cid=$CID / ClientID=$client_id");
}

ITERATION:
while (1) {
    my $rg = relaxed_guard times => $SLEEP_COEF;
    my @jobs = $queue->grab_jobs(
        ($client_id ? (filters => {ClientID => $client_id}) : ()),
        limit => $JOBS_LIMIT,
        grab_for => $JOB_GRAB_FOR,
    );

    $log->out(sprintf "Got %d jobs", scalar @jobs);
    last ITERATION  if !@jobs;

    my %archived_cids = map { $_ => undef } @{ filter_archived_campaigns([map { $_->args->{cid} } @jobs]) };
    JOB:
    for my $job (@jobs) {

        my $cid = $job->args->{cid};
        my $action = $job->args->{action};
        my $bids = $job->args->{bid};
        my $creative_id = $job->args->{creative_id};

        $log->out("Going to process cid=$cid ($action)");

        if ($job->trycount > $TRYCOUNT) {
            $job->mark_failed_permanently({ error => 'Unknown error' });
            next JOB;
        }

        if (exists $archived_cids{$cid}) {
            $job->mark_failed_permanently({ error => 'Campaign has been archived' });            
            $log->out("Campaign $cid has been archived (skip)");
            next JOB;
        }

        my $client_id = get_clientid(cid => $cid);

        my $creative = undef;
        
        if ($creative_id) {
            # номер креатива задан при создании задания
            $creative = Direct::VideoAdditions->get_by(creative_id => $creative_id, get_uid(cid => $cid))->items->[0];
            unless ($creative) {
                my $bs_creative = Direct::VideoAdditions->recieve_video_additions($client_id, [$creative_id])->{recieved};
                unless ($bs_creative && @$bs_creative) {
                    $log->out("Creative # $creative_id not found");
                    $job->mark_failed_permanently({ error => "Creative # $creative_id not found" });
                    next JOB;
                }
                Direct::VideoAdditions->save_video_additions(get_uid(cid => $cid), $client_id, $bs_creative);
                $creative = Direct::VideoAdditions->get_video_additions_from_bs($client_id, $bs_creative)->[0];
                unless ($creative) {
                    $log->out("Creative # $creative_id not found");
                    $job->mark_failed_permanently({ error => "Creative # $creative_id not found" });
                    next JOB;
                }
            }

            # тут нужно проверять banner_type
            # но для text|mobile_content они совпадают с типом кампании, а других тут пока не бывает
            my $camp_type = get_camp_type(cid => $cid);
            my $vr_layout = validate_video_additions_layout_id($camp_type => $creative);
            if ($vr_layout) {
                $log->out("invalid layout id " . $creative->layout_id . " for creative " . $creative->id);
                $job->mark_failed_permanently({ error => "invalid layout id " . $creative->layout_id . " for creative " . $creative->id });
                next JOB;
            }
        }

        eval {
            my $banners = Direct::Banners->get_by(
                campaign_id => $cid,
                banner_type => [qw/ text mobile_content /], #при добавлении нового типа проверить допустимые пресеты (validate_video_additions_layout_id)
                with_vcard => 1,
                with_sitelinks => 1,
                with_image => 1,
                with_roles => [qw/Direct::Model::Role::OnlyBannersResources/],
                with_video_resources => 1,
                filter => {
                    statusArch => 'No',
                    ($bids && @$bids ? ( 'b.bid' => $bids ) : ()),
                }
            )->items;
            croak "No banners found"  if !@$banners;
            $log->out(sprintf "%d banners to process", scalar @$banners);

            my $adgroups = Direct::AdGroups2->get(
                [uniq map { $_->adgroup_id } @$banners],
                adgroup_type => [qw/ base mobile_content /],
                extended => 1,
            )->items_by('id');
            for my $banner (@$banners) {
                $banner->old($banner->clone);
                $banner->adgroup($adgroups->{$banner->adgroup_id});
            }

            my $catalogia_rubrics = get_hash_sql(PPC(shard => $SHARD),
                ["SELECT b.bid, cbr.categories_bs from banners b left join catalogia_banners_rubrics cbr on cbr.bid = b.bid",
                where => {'b.bid' => [map { $_->id } @$banners]}]);

            if ($action eq 'set') {
                CHUNK: for my $banners_chunk (chunks $banners, 1000) {
                    if ($creative) {
                        for my $banner (@$banners_chunk) {
                            $log->out("set creative # $creative_id for banner # @{[$banner->id]}");
                            if (!$banner->has_creative) {
                                my $banner_creative = Direct::Model::BannerCreative->new(id => 0, creative => $creative);
                                $banner->creative($banner_creative);
                            }
                            else {
                                next if $banner->creative->id == $creative_id;
                                $banner->creative->creative($creative);
                            }
                        }
                        next CHUNK;
                    }

                    # группируем по типу + категориям + языку баннера
                    my %banner_by_params;
                    for my $banner (@$banners_chunk) {
                        my $key = _get_banner_group_key($banner, $catalogia_rubrics->{$banner->id} // '');
                        push @{$banner_by_params{$key}}, $banner;
                    }

                    for my $banners_group (values %banner_by_params) {
                        my $base_banner = $banners_group->[0];
                        my $categories = [split ',', $catalogia_rubrics->{$base_banner->id} // ''];
                        my $lang = $base_banner->detect_lang;
                        my $locale = Yandex::I18n::get_locale($lang);
                        # страшновато заводить целую новую локаль в Директе, поэтому прописываем белорусскую локаль здесь явно
                        if (!defined($locale) && $lang eq 'be') {
                            $locale = 'be_BY';
                        }

                        for my $generate_chunk (chunks $banners_group, 30) {
                            my $count = scalar @$generate_chunk;
                            my $conditions = [ {
                                banner_type => $base_banner->banner_type,
                                category_ids => $categories,
                                count => $count,
                                locale => $locale,
                            } ];
                            $log->out(['generate conditions:', $conditions]);
                            
                            my $gen_video_additions = Direct::VideoAdditions->generate_video_additions($client_id, $conditions);
                            $log->out("generated @{[scalar uniq map { $_->{creative_id} } @$gen_video_additions]} additions");

                            my $operator_uid = $job->{uid};
                            Direct::VideoAdditions->save_video_additions($operator_uid, $client_id, $gen_video_additions);
                            my $video_additions = Direct::VideoAdditions->get_video_additions_from_bs($client_id, $gen_video_additions);

                            for my $addition (@$video_additions) {
                                my $banner = shift @$generate_chunk;
                                $log->out(sprintf "banner # %s : set generated creative %s", $banner->id, $addition->id);
                                unless ($banner) {
                                    die "got too many banners for categories '".join '', @{$addition->yacontext_categories}."'";
                                }
                                my $vr_layout = validate_video_additions_layout_id($base_banner->banner_type => $addition);
                                die "invalid layout id " . $addition->layout_id . " for creative " . $addition->id if $vr_layout;
                                if ($banner->has_creative) {
                                    $banner->creative->creative($addition);
                                }
                                else {
                                    my $creative = Direct::Model::BannerCreative->new(id => 0, creative => $addition);
                                    $banner->creative($creative);
                                }
                            }
                        }
                    }
                }
            }
            elsif ($action eq 'reset') {
                for my $banner (@$banners) {
                    $banner->clear_creative();
                }
            }
            else {
                croak "Unsupported action <$action>";
            }

            my $type = $banners->[0]->banner_type;
            my $class = Direct::Banners::get_business_logic_class($type);
            my $logic = $class->new(items => $banners);
            $logic->update(1);

            $job->mark_finished({});
            juggler_ok();
            1;
        }
        or do {
            $log->out("FAILED cid=$cid: $@");
            # оставляем job в состоянии Grabbed, чтобы не подхватить его сразу же на следующей итерации
            # $job->mark_failed_once();
            next JOB;
        };
    }
}

$queue->delete_old_jobs(3600);

juggler_ok();

$log->out("FINISH");


sub _get_banner_group_key {
    my ($banner, $categories) = @_;

    state $jsoner = JSON->new()->utf8(0)->canonical;
    my $key_struct = [
        $banner->banner_type,
        $categories,
        $banner->detect_lang,
    ];
    return $jsoner->encode($key_struct);
}
