#!/usr/bin/perl

=head1 METADATA

<crontab>
    time: 17 */6 * * *
    <switchman>
        group: scripts-other
        <leases>
            mem: 300
        </leases>
    </switchman>
    package: scripts-switchman
</crontab>
<juggler>
    host:   checks_auto.direct.yandex.ru
    ttl: 19h
    tag: direct_group_internal_systems
    <notification>
        template: on_status_change
        status: OK
        status: CRIT
        method: telegram
        login: DISMonitoring
    </notification>
</juggler>

=cut

=head1 DESCRIPTION

    Скрипт получает данные по изменениям статистики от БК, начиная с момента последнего получения этих изменений,
    и рассылает соответствующие уведомления клиентам

=cut

use Direct::Modern;

use List::MoreUtils qw(uniq);

use Yandex::DateTime qw(now datetime);
use Yandex::DBShards qw/SHARD_IDS sharded_chunks/;
use Yandex::DBTools;
use Yandex::HashUtils;

use my_inc "..";

use Campaign::Types qw/get_camp_kind_types/;
use Currency::Format;
use Currency::Rate qw/convert_currency/;
use Client;
use Notification;
use Property;
use ScriptHelper 'Yandex::Log' => 'messages';
use Settings;
use Stat::RollbackNotify;
use TextTools;
use JSON qw/to_json/;

# отправляем новое письмо об откате статистики, только если сумма возврата превышает указанный порог в рублях
# другие валюты приводятся к рублям по текущему курсу
my $MAIL_SENDING_THRESHOLD_SUM_RUB = 100;

extract_script_params();

$log->out("start");

my $last_done_time_prop = new Property('stat_rollback_last_done_time');
my $start_time = $last_done_time_prop->get();

if ($start_time) {
    $start_time = datetime($start_time)->add( hours => 1 );
} else {
    $start_time = now();
}
$start_time = join '', $start_time->ymd(''), sprintf('%02d%02d%02d', $start_time->hour(), 0, 0); # округляем до часов

my ($orders_data, $last_done_time) = Stat::RollbackNotify::get_stat_rollback_data($log, $start_time);

Stat::RollbackNotify::update_direct_stat_rollbacks($orders_data);

unless (keys %$orders_data && $last_done_time) {
    $log->out('no data recieved');

    # считаем что за предыдущий час данные уже должны были прийти ранее
    $last_done_time = datetime($start_time)->add( hours => -1 );
    $last_done_time = join '', $last_done_time->ymd(''), sprintf('%02d%02d%02d', $last_done_time->hour(), 0, 0);
    $last_done_time_prop->set($last_done_time);

    juggler_ok(description => 'no data recieved');

    exit;
}

my $camps = get_hashes_hash_sql(PPC(OrderID => [ keys %$orders_data ]), [
   "SELECT c.OrderID
         , c.cid
         , c.uid
         , IFNULL(c.currency, 'YND_FIXED') AS currency
         , c.ManagerUID
         , c.name AS campaign_name
         , c.type AS campaign_type
         , c.wallet_cid
         , u.ClientID
      FROM campaigns c
      JOIN users u ON u.uid = c.uid",
     WHERE => { 'c.OrderID' => SHARD_IDS,
                type => get_camp_kind_types('web_edit') },
]);

my $client_ids = [ uniq map { $_->{ClientID} } values %$camps ];
my $new_stat_rollbacks_clients = mass_is_new_stat_rollbacks_enabled_for_client($client_ids);
my %users = ();
my %clients_nds = ();
my $new_stat_rollbacks_data = {};
foreach my $OrderID (keys %$orders_data) {
    my $camp = $camps->{$OrderID};
    unless ($OrderID && $camp) {
        $log->out("Incorrect OrderID = $OrderID");
        next;
    }

    my @periods = ();
    my $sum_cur = 0;
    my $shows = 0;
    my $client_nds;
    $client_nds = $clients_nds{$camp->{ClientID}} //= get_client_NDS($camp->{ClientID}) unless $camp->{currency} eq 'YND_FIXED';
    foreach my $item (@{$orders_data->{$OrderID}}) {
        $sum_cur += ($camp->{currency} eq 'YND_FIXED' ? $item->{sum} : $item->{sum_cur}/(1+$client_nds/100));
        $shows += $item->{shows};
        my $update_date = substr($item->{update_time}, 0, 8); # берем только дату ГГГГММДД
        push @periods, $update_date;

        if($new_stat_rollbacks_clients->{$camp->{ClientID}} && defined $item->{drop_type}) {
            $new_stat_rollbacks_data->{$camp->{uid}}->{$item->{drop_type}}->{sum_cur} += ($camp->{currency} eq 'YND_FIXED' ? $item->{sum} : $item->{sum_cur}/(1+$client_nds/100));
            $new_stat_rollbacks_data->{$camp->{uid}}->{$item->{drop_type}}->{currency} = $camp->{currency};
            push @{$new_stat_rollbacks_data->{$camp->{uid}}->{$item->{drop_type}}->{periods}}, $update_date;
            push @{$new_stat_rollbacks_data->{$camp->{uid}}->{$item->{drop_type}}->{cids}}, $camp->{cid};
        }
    }
    next if $sum_cur < 0.01 * 1_000_000;

    @periods = sort { $a cmp $b } map { join '-', /^(\d{4})(\d\d)(\d\d)$/ } uniq @periods;
    my $period_groups = Stat::RollbackNotify::group_periods(\@periods);

    my $camp_info = hash_cut($camp, qw/cid wallet_cid campaign_name campaign_type ManagerUID/);
    $camp_info->{currency} = format_currency($camp->{currency});
    $camp_info->{periods} = $period_groups;
    $camp_info->{sum} = round2s($sum_cur / 1_000_000);
    $camp_info->{shows} = $shows;

    $users{$camp->{uid}} ||= {};
    push @{$users{$camp->{uid}}->{$camp->{campaign_type}}}, $camp_info;
}

if(%$new_stat_rollbacks_data) {
    foreach my $chunk (sharded_chunks(uid => [keys %$new_stat_rollbacks_data], 1_000)) {
        my @rows_to_insert;
        for my $uid (@{$chunk->{uid}}) {
            for my $drop_type (keys %{$new_stat_rollbacks_data->{$uid}}) {
                my $sum = $new_stat_rollbacks_data->{$uid}->{$drop_type}->{sum_cur};
                my $currency = $new_stat_rollbacks_data->{$uid}->{$drop_type}->{currency};
                # приводим сумму возврата к рублям
                my $sum_rub = $currency eq 'RUB' ? $sum : convert_currency($sum, $currency, 'RUB');
                # чтобы не спамить клиентов, пропускаем письма со слишком маленькими возвратами
                next if $sum_rub < $MAIL_SENDING_THRESHOLD_SUM_RUB * 1_000_000;
                my $data = {
                    drop_type => $drop_type,
                    cids => $new_stat_rollbacks_data->{$uid}->{$drop_type}->{cids},
                    periods => $new_stat_rollbacks_data->{$uid}->{$drop_type}->{periods},
                    sum_cur => round2s($sum / 1_000_000),
                };
                push @rows_to_insert, [$uid, to_json($data)];
            }
        }
        do_mass_insert_sql(PPC(shard => $chunk->{shard}), "INSERT INTO stat_rollback_data (uid, data) VALUES %s", \@rows_to_insert);
    }
}

# получаем информацию о пользователях - для уведомлений менеджерам
my $users_info = get_hashes_hash_sql(PPC(uid => [keys %users]), [
    "SELECT uid
          , FIO AS client_fio
          , login AS client_login
          , phone AS client_phone
          , email AS client_email
          , ClientID AS client_id
       FROM users",
    WHERE => {uid => SHARD_IDS}   
]);

foreach my $uid (keys %users) {

    foreach my $type (keys %{$users{$uid}}) {
        my $vars = $users_info->{$uid};
        $vars->{cids} = $users{$uid}->{$type};
        $vars->{type} = $type;
        # если для клиента включены новые письма об откатах статистики, то отправляем письмо только менеджеру
        $vars->{for_manager_only} = $new_stat_rollbacks_clients->{$users_info->{$uid}->{client_id}};

        my $log_data = { data => { %$vars } }; # hash copy for log

        eval {             # $rbac
            add_notification(undef, 'stat_rollback', $vars);
        };
    
        $log_data->{status} = $@ ? 'error' : 'sent';
        $log->out($log_data);
    }
}

$last_done_time_prop->set($last_done_time);

juggler_ok();

$log->out('finish');
