#!/usr/bin/perl
use my_inc "..";
use Direct::Modern;

use List::MoreUtils qw/any part/;
use Scalar::Util qw/blessed/;

use Yandex::Advmon;
use Yandex::DBTools;

use Direct::Creatives qw/sync_creative_screenshots sync_creatives/;
use ScriptHelper sharded => 1, 'Yandex::Log' => 'messages';
use Settings;
use ShardingTools;

=head1 DESCRIPTION

Пока нам не приходят нотификации о появлении у креатива скриншота забираем их сами

Параметры:
    --shard-id=X            - номер шарда, клиенты которого будут обработаны
    --client_id=YYYYY       - обработать только указанных клиентов (из шарда X) (можно указывать несколько раз)
    --creative_id=YYY       - обработать только указанные креативы (можно указывать несколько раз)
    --bstorage_api_url=http://... - ходить на указанный адрес API (для тестирования)

=head1 METADATA

<crontab>
    time: */2 0-23 * * *
    package: scripts-switchman
    sharded: 1
    <switchman>
        group:  scripts-other
    </switchman>
</crontab>

<juggler>
    host:   checks_auto.direct.yandex.ru
    ttl: 1h
    sharded: 1
    tag: direct_group_internal_systems
</juggler>

=cut

# Обрабатывать не больше указанного количества заданий за итерацию.
our $MAX_TASKS = 200;
our $DELETE_PROCESSED_TASK_AFTER_HOURS = 1;
our $SYNC_AFTER_MINUTES = 0;
our $MAX_ATTEMPTS = 10;

sub check_failed_tasks {
# Выбираем задания с превышенным числом попыток, пишем id проблемных креативов в лог ставим им статус Fail
    my $failed = get_one_column_sql(PPC(shard => $SHARD), [
            qq/SELECT creative_id FROM creative_banner_storage_sync/,
            where => {
                sync_status  => 'New',
                attempts__ge => $MAX_ATTEMPTS,
            },
        ],
    );

    $log->out(sprintf(qq/Synchronization failed for creative %s - max attempts reached: %s/, $_, $MAX_ATTEMPTS)) foreach @$failed;
    do_sql(
        PPC(shard => $SHARD),
        [qq/UPDATE creative_banner_storage_sync SET sync_status = 'Failed'/, where => {creative_id => $failed}]
    ) if @$failed;
    return;
}

sub main {
    $log->out("START");

    my (@debug_client_id, @debug_creative_id, $bstorage_api_url, $max_tasks, $max_attempts);

    extract_script_params(
        'client_id=i@'   => \@debug_client_id,
        'creative_id=i@' => \@debug_creative_id,
        'bstorage_api_url=s' => \$bstorage_api_url,
        'max_tasks=i'    => \$max_tasks,
        'max_attempts'   => \$max_attempts,
    );

    $BannerStorage::BS_REST_API_URL = $bstorage_api_url if $bstorage_api_url;
    $max_tasks //= $MAX_TASKS;
    $max_attempts //= $MAX_ATTEMPTS;

    unless ($SHARD && any { $_ eq $SHARD } ppc_shards()) {
        $log->die('shard-id is not specified or has invalid value');
    }

# Удаляем все выполненные задания старше часа
    do_sql(PPC(shard => $SHARD), [qq/DELETE FROM creative_banner_storage_sync/,
                where => {
                    sync_status=>[qw/Failed Processed/],
                    _TEXT => qq/TIMESTAMPDIFF(HOUR, modifed, NOW() ) > ?/
                }], $DELETE_PROCESSED_TASK_AFTER_HOURS);
# Обрабатываем задания с превышенным числом попыток
    check_failed_tasks();
                
    my $creatives = get_all_sql(PPC(shard => $SHARD), [
            qq/SELECT creative_id, full_sync FROM creative_banner_storage_sync/,
            where => {
                sync_status => 'New',
                _TEXT => qq/modifed <= DATE_SUB(NOW(), INTERVAL ? MINUTE)/,
                scalar @debug_creative_id ? (creative_id => \@debug_creative_id) : (),
            },
           'ORDER BY modifed LIMIT ?',
        ],
        $SYNC_AFTER_MINUTES, $max_tasks);

    my ($full_sync, $screenshots_only) = part {$_->{full_sync} ? 0 : 1} @$creatives;
    my $counts = {};
    foreach my $creatives_for_sync ($full_sync, $screenshots_only) {
        my ($processed, $skipped);
        if ($creatives_for_sync && @$creatives_for_sync) {  
            # Инкрементируем всем выбранным заданиям число попыток
            do_sql( PPC(shard => $SHARD), [qq/
                        UPDATE creative_banner_storage_sync
                        SET attempts = attempts + 1/,
                        where => {creative_id => [map {$_->{creative_id}} @$creatives_for_sync]},
            ], );
           
            Direct::Creatives::fill_in_clients($creatives_for_sync);
            ($processed, $skipped) = $creatives_for_sync eq ($screenshots_only // 'none') ?
                sync_creative_screenshots($creatives_for_sync) : sync_creatives($creatives_for_sync);
            do_sql( PPC(shard => $SHARD), [
                    qq/UPDATE creative_banner_storage_sync SET sync_status='Processed'/,
                    where => {creative_id => [map {blessed($_) ? $_->id : $_} @$processed]},
            ]) if @$processed;
        }
        $counts->{processed} += @{$processed // []};
        $counts->{skipped} += @{$skipped // []};
    }
    juggler_ok() unless (@debug_client_id || @debug_creative_id);

    my $monitor_prefix = 'flow.ppcSyncModifedCreatives.';
    monitor_values({
        $monitor_prefix.'tasks_given.shard_'.$SHARD     => scalar(@$creatives),
        $monitor_prefix.'tasks_processed.shard_'.$SHARD => $counts->{processed},
        $monitor_prefix.'tasks_shifted.shard_'.$SHARD   => $counts->{skipped},
        $monitor_prefix.'tasks_total.shard_'.$SHARD     => get_one_field_sql(PPC(shard => $SHARD), qq/SELECT COUNT(*) FROM creative_banner_storage_sync/),
    });

    $log->out('Finish');
}

main();
