#!/usr/bin/perl
use strict;
use warnings;

=head1 METADATA

<crontab>
    time: */5 * * * *
    sharded: 1
    <switchman>
        group: scripts-other
        <leases>
            mem: 230
        </leases>
    </switchman>
    package: scripts-switchman
</crontab>
<juggler>
    host:   checks_auto.direct.yandex.ru
    sharded: 1
    ttl: 30m
    tag: direct_group_internal_systems
</juggler>

=head1 NAME

ppcXLSImageUploadFailureNotification.pl

=head1 SYNOPSIS

    ppcXLSImageUploadFailureNotification.pl --shard-id 1

=head1 DESCRIPTION

Отправить уведомления об ошибках загрузки изображений через XLS.

Как это работает: в таблице xls_history ищутся недавние записи со значением statusEmailed = "No".
Для каждой такой записи по данным в banner_images_process_queue_bid + DBQueue выясняется, были ли в файле изображения
и в каком состоянии сейчас находится их обработка. Случаи такие:

а: изображений нет - в строчке ставится statusEmailed = "Yes", чтобы больше она не обрабатывалась
б: изображения есть, часть из них ещё не обработана - запись пропускается, когда изображения будут обработаны,
   скрипт обработает эту запись в одной из следующих итераций
в: изображения есть, все обработаны, ошибок нет - в строчке ставится statusEmailed = "Yes"
г: изображения есть, все обработаны, есть ошибки - отправляется уведомление

Уведомление отправляется на адрес оператора, который загрузил Excel-файл, за исключением случая,
когда его загрузил вешальщик. Если его загрузил вешальщик, уведомление отправляется менеджеру
кампании или менеджеру агентства.

=cut

use Try::Tiny;
use List::UtilsBy qw/count_by/;

use Yandex::DBTools;
use Yandex::HashUtils;

use my_inc '..';
use Campaign;
use Notification;
use RBAC2::Extended;
use RBACDirect;
use RBACElementary;
use ScriptHelper sharded => 1, 'Yandex::Log' => 'messages';
use Settings;
use User;

use Yandex::DBQueue;

=head1 SUBROUTINES/METHODS

=head2 $BATCH_SIZE

Размер пачки записей в таблице xls_history, обрабатываемых за один запуск.

=cut

my $BATCH_SIZE = 10000;

extract_script_params();

my $rbac = RBAC2::Extended->get_singleton(1);

my $status = run();
exit $status;

=head1 SUBROUTINES/METHODS

=head2 mark_emailed

=cut

sub mark_emailed {
    my ($xls_id) = @_;
    do_sql( PPC( shard => $SHARD ), [ 'update xls_history set statusEmailed = "Yes"', where => { id => $xls_id } ] )
}

=head2 run

=cut

sub run {
    $log->out('start');

    my $rows = get_all_sql( PPC( shard => $SHARD ), qq{
        select id, cid from xls_history
        where type="import" and statusEmailed = "No" and logdate > now() - interval 1 day
        order by logdate
        limit $BATCH_SIZE
    } );

    my $queue = Yandex::DBQueue->new(PPC(shard => $SHARD), 'banner_images');

    foreach my $row (@$rows) {
        my $cid    = $row->{cid};
        my $xls_id = $row->{id};

        my $job_ids = get_one_column_sql( PPC( shard => $SHARD ), [
            'select job_id from banner_images_process_queue_bid',
            where => { xls_id => $row->{id} },
        ] );
        
        unless (@$job_ids) {
            mark_emailed($xls_id);
            next;
        }

        my $jobs = $queue->find_jobs(job_id => $job_ids);
        my %status_statistics = count_by { $_->status } @$jobs;

        # если картинки всё ещё обрабатываются, обработку этой строчки xls_history нужно
        # отложить до следующего запуска
        next if $status_statistics{New} || $status_statistics{Grabbed};

        # обработка этого задания завершена - нужно ли отправлять письмо?
        unless ( $status_statistics{Failed} ) {
            # нет, нет, никогда, картинок не было или все картинки обработались успешно
            mark_emailed($xls_id);
            next;
        }

        # да, есть ошибки при загрузке
        try {
            send_email_and_mark_emailed( $cid, $xls_id, $jobs );
        } catch {
            $log->out("error sending email: $_");
        };
    }

    juggler_ok();

    $log->out('finish');
    return 0;
}

=head2 send_email_and_mark_emailed

=cut

sub send_email_and_mark_emailed {
    my ( $cid, $xls_id, $jobs ) = @_;

    my $ClientID = $jobs->[0]->ClientID;
    my $uid = $jobs->[0]->uid;

    my $role = rbac_who_is( $rbac, $uid );

    # если это был вешальщик, письмо нужно отправить менеджеру, только искать этого менеджера нужно
    # по-разному в зависимости от типа обслуживания
    if ( $role eq 'placer' ) {
        my $uids_row = get_one_line_sql( PPC( cid => $cid ),
            [ 'select ManagerUID, AgencyUID from campaigns', where => { cid => $cid } ] );

        if ( my $manager_uid = $uids_row->{ManagerUID} ) {
            # вешальщик работал с менеджерской кампанией - отправляем менеджеру кампании
            $uid = $manager_uid;
        } elsif ( my $agency_uid = $uids_row->{AgencyUID} ) {
            # вешальщик работал с агентской кампанией - отправляем менеджеру агентства кампании
            $uid = rbac_get_manager_of_agency( $rbac, $agency_uid, 'text' )
        } else {
            # вешальщик что-то делал с индивидуальным клиентом - такого быть не должно,
            # но если случилось, письмо не отсылаем
            mark_emailed($xls_id);
            return;
        }

        $role = rbac_who_is( $rbac, $uid );
    }

    $log->out("sending a notification: xls_id=$xls_id, cid=$cid, ClientID=$ClientID, uid=$uid");

    my $vars = { xls_id => $xls_id, cid => $cid, uid => $uid, role => $role };

    my $camp_info = get_camp_info($cid);

    my $user_data_fields = [ qw( login phone fio email ) ];
    my $user_info   = get_user_data( $uid,              $user_data_fields );
    my $client_info = get_user_data( $camp_info->{uid}, $user_data_fields );

    hash_merge $vars, $user_info, {
        camp_name    => $camp_info->{name},
        client_login => $client_info->{login},
        client_phone => $client_info->{phone},
        client_fio   => $client_info->{fio},
    };

    add_notification( $rbac, 'xls_image_upload_failure', $vars );

    mark_emailed($xls_id);

    return;
}
