#!/usr/bin/perl

use my_inc "../..";



=head1 DESCRIPTION

Генерируем js-саджесты стран и городов
(формат тот же, что раньше забирали с export.yandex.ru)

=cut

use Direct::Modern;

use geobase;

use JSON;
use Encode;
use File::Slurp;
use IO::Scalar;
use List::Util qw/first/;
use List::MoreUtils qw/all first_index/;

use ScriptHelper;

use Yandex::ExecuteJS;
use Yandex::HashUtils qw/hash_cut/;

use Settings;


my @rules = (
    { 
#      url => 'http://export.yandex.ru/autocomplete/countries.xml?format=jsonp&code=yes&callback=countries.js&localize=yes',
      file => "$Settings::ROOT/data/countries.js",
      build => sub { _build_suggest('countries.js', type => 3) },
      check => sub {check_js_array('countries.js', [100, 1000], @_)},
    },
    { 
#      url => 'http://export.yandex.ru/autocomplete/cities.xml?format=jsonp&code=yes&callback=cities.js&localize=yes',
      file => "$Settings::ROOT/data/cities.js",
      build => sub { _build_suggest('cities.js', type => [6], need_country => 1) },
      # more settlements
      # build => sub { _build_suggest('cities.js', type => [6, 7], need_country => 1) },
      check => sub {check_js_array('cities.js', [1000, 60000], @_)},
    }
);

$log->out('START');

for my $rule (@rules) {
    $log->out("Preparing $rule->{file}");
    my $cont = $rule->{build}->();
    $rule->{check}->($cont)  if $rule->{check};
    write_file($rule->{file}, {atomic => 1}, Encode::encode_utf8($cont));
}

$log->out('FINISH');

exit;



# сборка сажджеста
sub _build_suggest {
    my ($name, %O) = @_;
    my $data = _get_suggest_data(%O);
    my $js_opts = {
        canonical => 1,
#        pretty => 1,
    };
    my $js = qq#window["$name"](# . to_json($data, $js_opts) . qq#);\n#;
    return $js;
}


# подготовка данных для саджестов
sub _get_suggest_data {
    my %O = @_;

    state $name_map = {
        name => 'name',
        en_name => 'enname',
        ua_name => 'ukname',
        tr_name => 'trname',
    };

    state $suggest_fields = [ qw/id code parentId/, values %$name_map ];

    my $type = $O{type};
    my %our_type = map {($_ => 1)} (ref $type ? @$type : ($type));

    my %reg;
    my %by_name;
    for my $geo_id (keys %geobase::Region) {
        my $geo = $geobase::Region{$geo_id};
        next if !$our_type{$geo->{type}};

        # {id:"29386",name:"Абхазия",ukname:"Абхазія",trname:"Abhazya",enname:"Abkhasia",code:""}
        # {id:"123188",parentId:"10021",name:" Найсне",ukname:"Кнісна",enname:"Knysna",code:""}
        my $reg = {
            id => $geo_id,
        };

        my $paths = _get_path_alternatives($geo_id);
        for my $path (@$paths) {
            $reg->{_path} = $path;
            $reg->{code} = _get_phone_code($geo_id, $path),
            my $country = _get_country($path);
            $reg->{parentId} = $country  if $O{need_country};

            for my $nkey (keys %$name_map) {
                my $name = $geo->{$nkey};
                next if !$name;
                $reg->{$name_map->{$nkey}} = $name;
                push @{$by_name{$nkey}->{$country}->{$name}}, $geo_id;
            }

            $reg{$country}->{$geo_id} = {%$reg};
        }
    }

    for my $nkey (keys %$name_map) {
        for my $country (keys %{$by_name{$nkey}}) {
            my $ids_by_name = $by_name{$nkey}->{$country};
            for my $name (keys %$ids_by_name) {
                my $reg_ids = $ids_by_name->{$name};
                next if @$reg_ids <= 1;

                my @regs = map {$reg{$country}->{$_}} @$reg_ids;
                _resolve_uniq_names(\@regs, $nkey, $name_map->{$nkey});
            }
        }
    }

    my @suggest =
        sort {$a->{name} cmp $b->{name} || $a->{id} cmp $b->{id}}
        map {hash_cut $_, @$suggest_fields}
        map {values %$_} values %reg;

    return \@suggest;
}


# получить телефонный код региона;
# если в самом регионе не задан - берём у предков
sub _get_phone_code {
    my ($geo_id, $path) = @_;

    for my $id ($geo_id, reverse @$path) {
        my $geo = $geobase::Region{$id};
        return $geo->{phone_code} if $geo->{phone_code};
    }

    return "";
}


# получить варианты транслокального пути до корня геодерева
sub _get_path_alternatives {
    my ($id) = @_;

    state $disputed = {
        977 => [187, 115092], # крым: КрФО или Украина
    };

    my $orig_path = $geobase::Region{$id}->{path};
    my @paths;
    for my $did (keys %$disputed) {
        my $i = first_index {$_ == $did} @$orig_path;
        next if $i<0;

        for my $parent (@{$disputed->{$did}}) {
            my @path = @$orig_path;
            my $parent_path = $geobase::Region{$parent}->{path};
            splice @path, 0, $i, @$parent_path, $parent;
            push @paths, \@path;
        }

        last;
    }

    push @paths, $orig_path  if !@paths;
    return \@paths;
}


# прописать в названия регионов уточнения для уникализации
sub _resolve_uniq_names {
    my ($regs, $gkey, $skey) = @_;

    my $name = $regs->[0]->{$skey};

    my %path;
    for my $reg (@$regs) {
        my $id = $reg->{id};
        my $thin_path = _thin_out_path($reg->{_path});
        $path{$id} = [map {$geobase::Region{$_}->{$gkey}} @$thin_path];
    }

    _filter_paths([values %path], 0);

    for my $reg (@$regs) {
        my $id = $reg->{id};
        next if !@{$path{$id}};

        state $reg_rename = {
            'Москва и Московская область' => 'Московская область',
            'Moscow and Moscow Oblast' => 'Moscow Oblast',
            'Москва та Московська область' => 'Московська область',
            'Moskova ve Moskovskaya oblastı' => 'Moskovskaya oblastı',
            'Санкт-Петербург и Ленинградская область' => 'Ленинградская область',
            'Saint-Petersburg and Leningrad Oblast' => 'Leningrad Oblast',
            'Санкт-Петербург і Ленінградська область' => 'Ленінградська область',
            'Saint-Petersburg ve Leningradskaya oblastı' => 'Leningradskaya oblastı',
        };

        $reg->{$skey} .= ' (' . join(', ', map {$reg_rename->{$_} || $_} @{$path{$id}}) . ')';
    }

    $log->out([$name => [map {$_->{$skey}} @$regs]]);

    return;
}


# убрать из пути "незначимые" регионы
sub _thin_out_path {
    my ($orig_path) = @_;

    my @path = @$orig_path;
    my @thin_path;
    while (my $id = shift @path) {
        next if _is_useless_geo($id, \@path);
        push @thin_path, $id;
    }

    return \@thin_path;
}


# проверка на "незначимость" региона
# незначимыми считаем группировки (типа федеральных округов)
# и объединения (типа Москвы+МО, если в адресе есть Москва)
sub _is_useless_geo {
    my ($id, $rest) = @_;

    state $useless_type = +{ map {($_ => 1)} (2, 4) };
    state $useless_id = +{
        1 => 213,   # Москва+МО для Москвы
        10174 => 2, # СПб+ЛО для СПб
    };

    return 0  if !@$rest;
    return 1  if $useless_type->{$geobase::Region{$id}->{type}};
    return 1  if $useless_id->{$id} && $rest->[0] == $useless_id->{$id};
    return 0;
}


# оставить в путях только регионы-предки, необходимые для уникализации
sub _filter_paths {
    my ($paths, $pos) = @_;

    my ($any_path) = $paths->[0];
    while (all {($_->[$pos] || 1) eq ($any_path->[$pos] || 2)} @$paths) {
        splice @$_, $pos, 1  for @$paths;
    }

    my %first_part;
    for my $path (@$paths) {
        next if @$path <= $pos;
        push @{$first_part{$path->[$pos]}}, $path;
    }

    for my $part (keys %first_part) {
        my $subpaths = $first_part{$part};
        if (@$subpaths > 1) {
            _filter_paths($subpaths, $pos + 1);
        }
        else {
            splice @{$subpaths->[0]}, $pos+1;
        }
    }

    return;
}


# получить код страны для региона
sub _get_country {
    my ($path) = @_;

    my $country = first {$geobase::Region{$_}->{type} == 3} @$path;
    return $country || 0;
}



# проверка на то, что нормальный js файл и элементов в массиве нормальное количество
sub check_js_array {
    my ($callback_name, $bounds, $js_data) = @_;

    my $arr = call_js(
        IO::Scalar->new(\ "window={};window[\"$callback_name\"]=function(a) { window.x = a}; $js_data; function ret() {return window.x}"),
        'ret',
        [],
        # 1МВ по умолчанию мало
        js_maxbytes => 7 * 1024 * 1024
        );

    if (ref($arr) ne 'ARRAY') {
        die "check for $callback_name: returned not array";
    }

    # есть все нужные поля
    for my $row (@$arr) {
        for my $field (qw/name code id/) {
            die "check for $callback_name: not defined field $field" unless defined $row->{$field};
        }
    }

    my $len = @$arr;
    if ($bounds && $len < $bounds->[0] || $len > $bounds->[1]) {
        die "check for $callback_name: incorrect array size: $len (must be $bounds->[0] .. $bounds->[1])";
    }
}

