#!/usr/bin/perl


=head1 METADATA

<crontab>
    params: /var/www/ppc.yandex.ru/data/t/prebuild/networks /var/www/ppc.yandex.ru/protected/data/networks
    time: 11 * * * *
    package: internal-networks
</crontab>

<crontab>
    params: --update-zookeeper /var/www/ppc.yandex.ru/data/t/prebuild/networks /var/www/ppc.yandex.ru/protected/data/networks
    time: */10 * * * *
    package: scripts-ppcback
</crontab>

=cut

=head1 NAME

    update_networks_file.pl - раскрыть макросы в файле с сетями

=head1 DESCRIPTION

    update_networks_file.pl --help
    update_networks_file.pl template_dir result_dir
    update_networks_file.pl --update-zookeeper template_dir result_dir

    Два обязательных аргумента: шаблон файла с сетями, результирующий файл
    Или директория с шаблонами и директория для результатов

    Возможные параметры командной строки:
    --help - вывести справку
    --init - перегененировать файл только если его нет
    --update-zookeeper - сохранить json с сетями в Zppkeeper, для java-проектов

    --skip-errors
    --no-skip-errors 
    что делать, если раскрытие одного макроса падает: 
    skip -- игнорировать и продолжать работать (итоговый файл может получиться неполным); 
    no-skip -- падать (файл может вообще не получиться, но не может получиться неполным)
    
=cut

use strict;
use warnings;

use Path::Tiny;
use Socket qw/ AF_INET AF_INET6 :addrinfo /;
use Net::INET6Glue::INET_is_INET6;
use Regexp::IPv6 qw($IPv6_re);
use LWP::UserAgent;
use JSON;
use POSIX qw/strftime/;

use my_inc '../..';

use Yandex::Retry;
use Yandex::Retry qw/retry/;
use Yandex::Shell qw/yash_qx yash_system/;

use EnvTools;
use ScriptHelper 'Yandex::Log' => 'messages';

use utf8;

my $RACKTABLES_URL_TEMPLATE = 'https://racktables.yandex.net/export/networklist.php?report=%s';

# реальный максимальный размер ноды - 1Mb, ставим поменьше, чтобы было время на реакцию
my $ZK_NODE_BORDER_SIZE = int(1024*1024 * 0.9);
# конфиг для zk-delivery
my $ZK_DELIVERY_CONFIG = "/etc/zk-delivery/ppc.cfg";
# нода в zk
my $ZK_NETWORK_CONFIG_PATH = "/direct/network-config.json";

my $INIT;
my $UPDATE_ZOOKEEPER;
my $SKIP_ERRORS = is_beta() ? 1 : 0;

Getopt::Long::GetOptions(
    "help" => \&usage,
    "init" => \$INIT,
    "update-zookeeper" => \$UPDATE_ZOOKEEPER,
    "skip-errors!" => \$SKIP_ERRORS,
    );

if (@ARGV != 2 || !-d $ARGV[0] || !-d $ARGV[1]) {
    usage();
}
my ($tmpl, $out) = map {path($_)} @ARGV;

$log->out("start");

my %nets;
for my $tmpl_file ($tmpl->children(qr/\.tmpl$/)) {
    my $out_file = $out->child( $tmpl_file->basename =~ s/\.tmpl$//r );
    next if $INIT && $out_file->exists && !$UPDATE_ZOOKEEPER;

    my $name = $tmpl_file->basename =~ s/\..*//r;

    $nets{$name} = get_networks_from_tmpl($tmpl_file);
    $out_file->spew_utf8(@{$nets{$name}});
}
update_zookeeper(\%nets) if $UPDATE_ZOOKEEPER;

$log->out("finish");


sub get_networks_from_tmpl {
    my ($tmpl_file) = @_;

    my @res = ("# generated automatically by $0 from template $tmpl_file at ".strftime("%Y-%m-%d %H:%M:%S", localtime)."\n");
    for my $line ($tmpl_file->lines_utf8) {
        eval {
            my @nets;
	    # Проверяем есть ли ограничения использования
            if ($line =~ /^\# \s* FOR_DEV_TEST_ONLY \s+ (.*) $/x) {
                if (! is_production()) {
                    $line = "$1\n";
                } else {
                    next;
                }
            }
	    if ($line =~ /^(_[A-Z0-9_]+_)(\s.*|#.*|$)/s) {
                # в строке - макрос
                my ($macro, $comment) = ($1, $2);
                for my $net (_expand_macro($macro)) {
                    push @res, "$net # from $macro $comment";
                }                
            } elsif ($line =~ /^\s*RACKTABLES:(\w+)/) {
                # таблица адресов из racktables
                push @res, "\n", "# $line\n";
                for my $net (_expand_racktables($1)) {
                    push @res, $net;
                }
                push @res, "\n";
            } else {
                # простая строка
                push @res, $line;
            }
        };
        die $@ if $@ && ! $SKIP_ERRORS;
    }
    
    $log->out(\@res);
    return \@res;
}


sub _expand_macro {
    my ($macro) = @_;
    
    my $items = retry tries => 3, pauses => [3], sub { 
        _hbf_macro($macro);
    };
    die "Error in expand macro $macro: no networks" if !$items || !@$items;
    
    # этот хеш нужен, чтобы пропускать повторно встречающиеся адреса
    # (пример: beta.geocontext.yandex.ru or beta-precise.geocontext.yandex.ru)
    my %added_ips;
    
    for my $item (@$items) {
        if ( $item =~ /^\d+\.\d+\.\d+\.\d+(\/\d+)?$/ ) {
            # ipv4: добавляем
            $added_ips{$item} = 1;
        } elsif ( $item =~ m!^[\dabcdef:/]+$! ) {
            my @ipnet = split(/\//, $item);
            # ipv6: тоже добавляем
            if ( $ipnet[0] =~ /^$IPv6_re$/ ) {
                # проверяем, что адрес валидный
                $added_ips{$item} = 1;
            } else {
                $log->out("Bad ipv6 address $item. Ignore it.")
            }
        } elsif ( my ($proj, $ip, $mask) = $item =~ m!^([0-9a-f]+)\@([0-9a-f:]+)/(\d+)$! ) {
            # host based firewall
            if ( $ip =~ /^$IPv6_re$/ ) {
                # проверяем, что адрес валидный
                $added_ips{$item} = 1;
            } else {
                $log->out("Bad hbf network $item. Ignore it.")
            }
        } else {
            # если мы до сюда добрались, наверное, $item - это hostname
            my ($err, @dns_results) = getaddrinfo( $item, '' );
            
            if ($err) {
                $log->out("Cannot resolve $item from macro $macro: $err");
                next;
            }
            
            for my $ip_info (@dns_results) {
                next if $ip_info->{family} != AF_INET && $ip_info->{family} != AF_INET6; 

                my $ip = [ getnameinfo( $ip_info->{addr}, NI_NUMERICHOST, NIx_NOSERV ) ]->[1];
                $added_ips{$ip} = 1;
            }
        }
    }

    return sort keys %added_ips;
}

sub _hbf_macro {
    my ($macro) = @_;
    my $ua = LWP::UserAgent->new(timeout => 30, ssl_opts => { SSL_ca_path => '/etc/ssl/certs' });
    my $url = "http://hbf.yandex.net/macros/$macro?format=json";
    my $resp = $ua->get($url);
    die "Can't get $url: ".$resp->status_line if !$resp->is_success;
    return from_json($resp->decoded_content);
}
    

sub _expand_racktables {
    my ($report) = @_;
        
    my $url = sprintf $RACKTABLES_URL_TEMPLATE, $report;
    my $ua = LWP::UserAgent->new(timeout => 30, ssl_opts => { SSL_ca_path => '/etc/ssl/certs' });

    return retry tries => 3, pauses => [3], sub {
        my $resp = $ua->get($url);
        die "Can't get $url: ".$resp->status_line if !$resp->is_success;
        my @ret;
        for my $line (split /\n/, $resp->decoded_content) {
            if ($line =~ /^(\d+\.\d+\.\d+\.\d+\/\d+)(?:\s+(.*))?/) {
                # ipv4- префикс + необязательный комментарий
                push @ret, "$1".(defined $2 ? "\t# $2" : '')."\n";
            } elsif ($line =~ m!^([0-9abcdef:/]+)(?:\s+(.*))?$!) {
                # ipv6- префикс + необязательный комментарий
                my ($ipstr, $comment) = ($1, $2);
                my @ipnet = split(/\//, $line);
                if ( $ipnet[0] =~ /^$IPv6_re$/ ) {
                    # проверяем, что адрес валидный
                    push @ret, "$ipstr".(defined $comment ? "\t# $comment" : '')."\n";
                } else {
                    $log->out("Bad ipv6 address $line. Ignore it.");
                }
            } elsif ($line !~ /^\s*(#.*)?$/) {
                die "Incorrect line: $line";
            }
        }
        if (@ret < 20) {
            die "Content of $url too short: only ".scalar(@ret)." lines, we want at least 20";
        }
        return sort @ret;
    };
}

# пишем сети в нужном формате в zookeeper
sub update_zookeeper {
    my ($nets) = @_;

    # парсим
    my @json;
    while(my ($name, $lines) = each %$nets) {
        my @nets;
        for my $line (@$lines) {
            if ($line =~ /^\s*([\@0-9\.\:a-f]+(?:\/\d+)?)/i) {
                push @nets, $1;
            }
        }
        push @json, {name => $name, masks => [map {+{mask => $_}} sort @nets]};
    }

    # генерим json
    my $data = JSON->new->canonical->encode([sort {$a->{name} cmp $b->{name}} @json]);
    $data =~ s/   / /g;
    if (length($data) >= $ZK_NODE_BORDER_SIZE) {
        die sprintf("Size of json data for zookeeper more than limit: %d > %d", length($data), $ZK_NODE_BORDER_SIZE);
    }

    # проверяем на изменение
    my $old_data = yash_qx("/usr/local/bin/zk-delivery-get", -c => $ZK_DELIVERY_CONFIG, $ZK_NETWORK_CONFIG_PATH);
    if ($old_data ne $data) {
        # пишем
        $log->out("save data to zookeeper, ".length($data)." bytes");
        my $stderr = "";
        yash_system("/usr/local/bin/zk-delivery-set", -c => $ZK_DELIVERY_CONFIG, '--log' => \$stderr, '--log-level' => 'error', $ZK_NETWORK_CONFIG_PATH, \$data);
        print STDERR $stderr if $stderr;
    } else {
        $log->out("zookeeper contains the same data");
    }
}

