#!/usr/bin/env python
# -*- coding: utf-8 -*-

""" METADATA

<crontab>
    time: */5 * * * *
    <switchman>
        lockname: export_features_history.py
        group: scripts-other
        delay: 3
    </switchman>
    package: scripts-switchman
</crontab>
<crontab>
    time: */5 * * * *
    <switchman>
        lockname: export_features_history.py
        group: scripts-test
        delay: 3
    </switchman>
    package: conf-test-scripts
</crontab>
<juggler>
    host:       direct.prod_javaperl
    name:       scripts.update_features_history.working
    raw_events: scripts.update_features_history.working
    ttl:        8m
    tag:        directadmin_TV
</juggler>

"""

"""
Собираем историю изменения фич
"""

import sys
import os.path
import logging
import requests
import time
from datetime import datetime, timedelta

sys.path[0:0] = [os.path.realpath(os.path.join(os.path.dirname(__file__), ".."))]

from yandex.juggler import juggler_queue_event
from direct.tools import set_logging
from direct.db import db_engine, get_db_config
import settings

SERVICE_NAME = 'scripts.update_features_history.working'
PROPERTY_NAME = 'update_features_history.last_event_datetime'
CLICKHOUSE_RETRIES = 4


def query_to_clh(query, cfg):
    for i in xrange(CLICKHOUSE_RETRIES):
        req = requests.post(
            "%s://%s:%s/?database=%s&max_block_size=1024" % (
                'https' if cfg.get('ssl') else 'http',
                cfg['host'],
                cfg['port'],
                cfg['db'],
            ),
            data=query,
            headers={"X-ClickHouse-User": cfg['user'], "X-ClickHouse-Key": cfg['pass']},
            verify=False,
        )

        try:
            return req.json()
        except:
            logging.critical("can't parse response: status_code: %s, content: %s" % (req.status_code, req.content))

        if i + 1 == CLICKHOUSE_RETRIES:
            raise
        else:
            time.sleep(1)


def get_new_settings(names, values, datetime, reqid):
    try:
        return values[names.index('settings')]
    except:
        logging.error("settings are not changed (datetime: %s, reqid: %s)" % (datetime, reqid))
        return "{}"


def main():
    ppcdict_engine = db_engine('ppcdict')
    last_dt_result = ppcdict_engine.execute(
        "SELECT value FROM ppc_properties WHERE name='%s'" % PROPERTY_NAME).fetchall()

    binlog_cfg = get_db_config('ppchouse:cloud_writer')

    if not last_dt_result or not last_dt_result[0] or not last_dt_result[0][0]:
        # при первом запуске собираем данные с начала 2019 года
        last_added_datetime = '2019-01-01 00:00:00'
    else:
        last_added_datetime = last_dt_result[0][0]

    last_added_value = datetime.strptime(last_added_datetime, '%Y-%m-%d %H:%M:%S')
    last_added_date_binlogs = last_added_value.strftime('%Y-%m-%d')

    # has(row.name, 'settings')=1 OR operation="DELETE":
    # не вставляем записи с new_settings="{}" если не было удаления фичи.
    # Пустые настройки возможны, например, если изменений в settings не было, но были изменения в ticket_id
    binlogs = query_to_clh(
        """
        SELECT reqid, datetime, operation, row.name, row.value, primary_key
        FROM binlog_rows_v2
        WHERE date>='%s'
          AND db='ppcdict'
          AND table='features'
          AND (has(row.name, 'settings')=1 OR operation='DELETE')
          AND datetime>'%s' %s
        ORDER BY datetime
        FORMAT JSON
        """ % (
            last_added_date_binlogs,
            last_added_datetime,
            "AND source='testing:ppcdict'" if settings.CONFIGURATION == 'test' else ""
        ),
        binlog_cfg
    )

    if not binlogs['data']:
        return

    # разный формат времени в binlog_rows_v2 и ppclog_cmd
    # с запасом убираем целый день, чтобы ничего не пропустить
    last_added_date_cmd = (last_added_value - timedelta(days=1)).strftime('%Y-%m-%d')

    reqids = list(set(log['reqid'] for log in binlogs['data'] if log['reqid'] != "0"))
    req2uid = {}
    if reqids:
        logcmds = query_to_clh(
            """
            SELECT reqid, uid
            FROM ppclog_cmd
            WHERE log_date>='%s' AND reqid IN (%s)
            FORMAT JSON
            """ % (
                last_added_date_cmd,
                ", ".join(reqids)
            ),
            binlog_cfg
        )
        req2uid = {logcmd['reqid']: logcmd['uid'] for logcmd in logcmds['data']}

    features_id2name = {
        str(id): name for id, name in ppcdict_engine.execute(
            "SELECT feature_id, feature_text_id FROM features WHERE feature_id IN (%s)" % ",".join(
                set(binlog['primary_key'] for binlog in binlogs['data'])
            )
        )
    }

    def get_deleted_feature_text_id(feature_id):
        record = ppcdict_engine.execute(
            """
            SELECT feature_text_id
            FROM features_history
            WHERE feature_id=%s AND feature_text_id <> ''
            ORDER BY event_time DESC
            LIMIT 1
            """, feature_id).fetchone()
        return '' if not record else record[0]

    values = [
        [
            binlog['reqid'],
            binlog['datetime'],
            binlog['operation'],
            binlog['primary_key'],
            features_id2name.get(binlog['primary_key'], '') or get_deleted_feature_text_id(binlog['primary_key']),
            get_new_settings(binlog['row.name'], binlog['row.value'], binlog['datetime'], binlog['reqid']),
            req2uid.get(binlog['reqid'], '0'),
        ] for binlog in binlogs['data']
    ]

    ppcdict_engine.execute(
        """
        INSERT INTO features_history
            (reqid, event_time, event_type, feature_id, feature_text_id, new_settings, operator_uid)
        VALUES
            %s
        """ % (",".join("(%s)" % ",".join("'%s'" % field for field in value) for value in values))
    )

    if not last_dt_result:
        ppcdict_engine.execute(
            "INSERT INTO ppc_properties (name, value) VALUES ('%s', '%s')" % (
                PROPERTY_NAME, binlogs['data'][-1]['datetime']
            )
        )
    else:
        ppcdict_engine.execute(
            "UPDATE ppc_properties SET value='%s' WHERE name='%s'" % (
                binlogs['data'][-1]['datetime'], PROPERTY_NAME
            )
        )


if __name__ == '__main__':
    try:
        set_logging(loglevel=logging.WARNING)
        logging.warn("start")

        main()

        juggler_queue_event(SERVICE_NAME, 'OK', '')
        logging.warn('success')
    except Exception as e:
        logging.exception('unexpected exception')
        juggler_queue_event(SERVICE_NAME, 'CRIT', 'unexpected exception: %s %s' % (type(e), e))
