# -*- encoding: utf-8 -*-

import os
import fcntl

class FileLock(object):
    """ Блокировка файлов с поддержкой context manager
    """

    def __init__(self, filename):
        """ Создание объекта для работы с блокировками
        """
        self.filename = filename
        self._fd = os.open(self.filename, os.O_CREAT|os.O_RDWR)
        self.is_locked = False

    def acquire(self):
        """ Получить блокировку файла.
        """
        fcntl.lockf(self._fd, fcntl.LOCK_EX|fcntl.LOCK_NB)
        self.is_locked = True

    def release(self):
        """ Освободить блокировку файла.
        """
        if self.is_locked:
            fcntl.lockf(self._fd, fcntl.LOCK_UN)

    def __enter__(self):
        """ Вызывается при использовании в выражении with.
            Получает блокировку файла.
        """
        if not self.is_locked:
            self.acquire()
        return self


    def __exit__(self, type, value, traceback):
        """ Activated at the end of the with statement.
            It automatically releases the lock if it isn't locked.
        """
        if self.is_locked:
            self.release()
