# -*- encoding: utf-8 -*-
"""
Сборник часто используемых функций
"""

from collections import defaultdict
from contextlib import contextmanager
from itertools import groupby
import pwd, os, sys
import re

__all__ = ['flatten', 'uniq', 'elogin', 'host_dc', 'save_pydoc', 'dict_merge',
           'ichunks', 'chunks', 'tmp_attr',
           ]

def flatten(arr):
    """
    Из массива массивов получить один плоский массив склеиванием
    """
    ret = []
    for item in arr:
        ret.extend(item)
    return ret

def uniq(arr):
    """
    Из массива поучить массив уникальных элементов
    """
    was = set()
    ret = []
    for item in arr:
        if not item in was:
            was.add(item)
            ret.append(item)
    return ret

def elogin():
    """
    Получить логин эффективного пользователя
    """
    return pwd.getpwuid(os.getuid())[0]

def host_dc(host_name):
    """
    Пытаемся получить из имени сервера букву, обозначающую датацентр
    Если не смогли - возвращаем None
    """
    if host_name is None:
        return None
    m = re.match(r'^.*[0-9]([a-z])\.yandex\.[a-z]+$', host_name)
    if m:
        return m.group(1)
    else:
        return None

def save_pydoc(*decorators):
    """
    Pydoc не срабатывает для функций, обёрнутых в декоратор.
    Этот декоратор позволяет сохранить __doc__ сквозь все декораторы
    """
    def spd(f):
        newf = f
        for d in decorators:
            newf = d(newf)
        newf.__doc__ = f.__doc__
        newf.__module__ = f.__module__
        return newf
    return spd


def dict_merge(*args):
    """
    Объединить несколько словарей в один, перезаписывая значения, если надо
    Аналон Yandex::HashUtils::hash_merge, за ис
    """
    ret = {}
    for d in args:
        ret.update(d)
    return ret


def full_groupby(l, key=lambda x: x):
    """
    Аналог itertools.groupby но без необходимости сортировать входной массив
    и неленивый
    """
    d = defaultdict(list)
    for item in l:
        d[key(item)].append(item)
    return d.items()


def ichunks(seq, chunk_size):
    """
    Разбиваем последовательность на куски длины навной chunk_size
    (последний кусок может быть меньше)
    Функция - генератор, возвращющий генераторы
    """
    for _, grp in groupby(enumerate(seq), lambda x: x[0] // chunk_size):
        yield (v for _, v in grp)


def chunks(seq, chunk_size):
    """
    Аналог ichunks, но возвращает массив массивов
    """
    chunk_size = int(chunk_size)
    seq = list(seq)
    if chunk_size <= 0:
        raise Exception("Incorrect chunk_size")
    start = 0
    ret = []
    while start < len(seq):
        ret.append(seq[start : start + min(chunk_size, len(seq) - start)])
        start += chunk_size
    return ret


@contextmanager
def tmp_attr(obj, attr, value):
    """
    ContextManager для временной установки значения аттрибута у объекта
    """
    old_value = getattr(obj, attr)
    setattr(obj, attr, value)
    try:
        yield
    finally:
        setattr(obj, attr, old_value)


if __name__ == '__main__':
    #print direct_hosts()
    print list(uniq([123,345,34,23,23,23,123,4355]))
