#!/usr/bin/perl

use strict;
use warnings;

use Test::More;

use Currencies qw/round_price_to_currency_step/;

use Yandex::Test::UTF8Builder;
use utf8;

# переопределяем параметры шагов валют, чтобы не зависеть от их изменения у реальных валют
local %Currencies::_CURRENCY_DESCRIPTION = (
    C001 => {
        AUCTION_STEP => 0.01,
        MAX_PRICE => 20,
    },
    C010 => {
        AUCTION_STEP => 0.1,
        MAX_PRICE => 20,
    },
    C123 => {
        AUCTION_STEP => 1.23,
        MAX_PRICE => 20,
    },
);

my @tests = (
    # [$price, $currency, \%O, $expected_result]

    # округление нулевых ставок
    # 0 кратен любому шагу, но запрос на округление вверх рассматриваем как запрос
    # "какую ставку надо поставить, чтобы получить позицию не хуже, чем за указанную цены"
    # поэтому возвращаем минимальный шаг торгов (иначе же за ноль торговаться не будет)
    [0, 'C001', {down => 1}, 0],
    [0, 'C001', {up => 1}, 0.01],
    [0, 'C123', {down => 1}, 0],
    [0, 'C123', {up => 1}, 1.23],

    # округление одноцентовых ("маленьких") ставок
    [0.01, 'C001', {down => 1}, 0.01],
    [0.01, 'C001', {up => 1}, 0.01],
    [0.01, 'C123', {down => 1}, 0],
    [0.01, 'C123', {up => 1}, 1.23],

    # округления вниз/вверх "больших" ставок
    [6.121, 'C001', {down => 1}, 6.12],
    [6.121, 'C001', {up => 1}, 6.13],
    [6.121, 'C123', {down => 1}, 4.92],
    [6.121, 'C123', {up => 1}, 6.15],

    # если ставка уже кратна шагу торгов, то не меняем её
    [6.15, 'C001', {down => 1}, 6.15],
    [6.15, 'C001', {up => 1}, 6.15],
    [6.15, 'C123', {down => 1}, 6.15],
    [6.15, 'C123', {up => 1}, 6.15],

    # округления вниз/вверх ставок, близких к максимальной
    [19, 'C001', {down => 1}, 19],
    [19, 'C001', {up => 1}, 19],
    [19, 'C123', {down => 1}, 18.45],
    [19, 'C123', {up => 1}, 19.68],

    # округления вниз/вверх максимальных ставок
    [20, 'C001', {down => 1}, 20],
    [20, 'C001', {up => 1}, 20],
    [20, 'C123', {down => 1}, 19.68],
    [20, 'C123', {up => 1}, 20],

    # округления вниз/вверх для ставок выше максимальных
    [21, 'C001', {down => 1}, 20],
    [21, 'C001', {up => 1}, 20],
    [21, 'C123', {down => 1}, 20],
    [21, 'C123', {up => 1}, 20],

    # проверяем случаи с неточным представлением чисел
    [1.2, 'C010', {down => 1}, 1.2],
    [1.2, 'C010', {up => 1}, 1.2],
    [1.4, 'C010', {down => 1}, 1.4],
    [1.4, 'C010', {up => 1}, 1.4],
    [1.93, 'C001', {up => 1}, 1.93],

    # для случая вычисления в AutoBroker.pm
    [201/100, 'C001', {down => 1}, 2.01],
    [201/100, 'C001', {up => 1}, 2.01],
    [200/100, 'C001', {down => 1}, 2.00]
);

for my $test(@tests) {
    _run_test($test, '==');
}

# вдобавок стреляем по площадям
# идём с шагом 0.05 по ставкам от 0 до 20 для валюты с шагом торгов 0.1
# и проверяем, что при округлении вниз получившееся значение не превышает исходного, 
# а при округлении вверх не меньше исходного
my %round_types = (up => '>=', down => '<=');
for (my $price = 0; $price <= 20; $price += 0.5) {
    while (my ($round_type, $op) = each %round_types) {
        my $test = [$price, 'C010', {$round_type => 1}, $price];
        _run_test($test, $op);
    }
}

done_testing;

sub _run_test {
    my ($test, $op) = @_;

    my ($price, $currency, $opts, $expected_result) = @$test;
    my $up_or_down_str = ($opts->{up}) ? 'up' : 'down';
    my $test_name = "rounding $price for currency $currency $up_or_down_str";
    my $result = round_price_to_currency_step($price, $currency, %$opts);
    cmp_ok("$result", $op, "$expected_result", $test_name);
}
