package ru.yandex.autotests.direct.utils;

import org.hamcrest.Matcher;
import org.junit.rules.TestRule;
import org.junit.runner.Description;
import org.junit.runners.model.Statement;

import static org.hamcrest.CoreMatchers.either;
import static org.hamcrest.core.IsInstanceOf.instanceOf;

/**
 * Usage:
 *
 * @Rule public RetryRule retry = RetryRule.retry()
 * <p/>
 * .ifException(WebdriverAssertionError.class)
 * <p/>
 * .every(20, TimeUnit.SECONDS)
 * <p/>
 * .times(2);
 */
public class RetryRule implements TestRule {

    private Matcher<Object> matcher;
    private int attempts = 3;

    private RetryRule() {
    }

    public static RetryRule retry() {
        return new RetryRule();
    }

    public RetryRule times(int attempts) {
        this.attempts = attempts;
        return this;
    }

    @SuppressWarnings("unchecked")
    public RetryRule ifException(Matcher<?> newMatcher) {
        if (matcher == null) {
            matcher = (Matcher<Object>) newMatcher;
        } else {
            matcher = either(matcher).or((Matcher<Object>) newMatcher);
        }
        return this;
    }


    /**
     * Adds to the list of requirements for any thrown exception that it
     * <p/>
     * should be an instance of {@code type}
     */
    public RetryRule ifException(Class<? extends Throwable> type) {
        return ifException(instanceOf(type));
    }

    @Override
    public Statement apply(final Statement base, final Description description) {
        return new Statement() {
            @Override
            public void evaluate() throws Throwable {
                Throwable e = null;
                for (int i = 0; i <= attempts; i++) {
                    try {
                        base.evaluate();
                        return;
                    } catch (Throwable t) {
                        e = t;
                        if (matcher == null || !matcher.matches(e)) {
                            throw e;
                        }
                    }
                }
                throw e;
            }
        };
    }

}
