package ru.yandex.autotests.direct.utils;

import org.yaml.snakeyaml.Yaml;
import org.yaml.snakeyaml.constructor.ConstructorException;
import ru.yandex.autotests.direct.data.ApiMode;
import ru.yandex.autotests.direct.data.Cmd;
import ru.yandex.autotests.direct.objects.MonthlyHist;
import ru.yandex.autotests.direct.objects.WeeklyHist;
import ru.yandex.autotests.direct.objects.WordStat;
import ru.yandex.autotests.direct.objects.advqstrings.Word;

import java.util.Map;
import java.util.TreeMap;

import static ru.yandex.autotests.direct.data.AdvqConstants.MAX_TAINTED_COUNT;
import static ru.yandex.autotests.direct.utils.ApiHelper.getApiResponse;

/**
 * User: xy6er
 * Date: 17.06.13
 * Time: 21:19
 */

/**
 * YAML конвертер в java bean.
 */
public class YamlHelper {

    /**
     * Конвертирует YAML в бин WordStat. Выполняет запрос к апи,
     * пока не получит бин, у которого tainted=false,
     * или пока количество попыток не превысит MAX_TAINTED_COUNT,
     * иначе вернет последний полученный бин.
     * @param apiMode Тестируемая среда
     * @param word Фраза, по которой нужен срез
     * @return WordStat bean
     */
    public static WordStat getWordStat(ApiMode apiMode, Word word, int phPageSize) {
        Map<String, String> params = new TreeMap<>();
        params.put("ph_page_size", String.valueOf(phPageSize));
        Yaml yaml = new Yaml();
        WordStat wordStat = yaml.loadAs(getApiResponse(apiMode, Cmd.SEARCH, word, params), WordStat.class);
        int taintedCount = 0;
        while (wordStat.isTainted() && taintedCount < MAX_TAINTED_COUNT) {
            taintedCount++;
            wordStat = yaml.loadAs(getApiResponse(apiMode, Cmd.SEARCH, word), WordStat.class);
        }

        return wordStat;
    }

    public static WordStat getWordStat(ApiMode apiMode, Word word) {
        return getWordStat(apiMode, word, 0);
    }

    /**
     * Конвертирует YAML в бин WeeklyHist. Выполняет запрос к апи,
     * пока не получит бин, у которого tainted=false,
     * или пока количество попыток не превысит MAX_TAINTED_COUNT,
     * иначе вернет последний полученный бин.
     * @param apiMode Тестируемая среда
     * @param word Фраза, по которой нужен срез
     * @return WeeklyHist bean
     */
    public static WeeklyHist getWeeklyHist(ApiMode apiMode, Word word) {
        WeeklyHist weeklyHist = convertYamlToBean(getApiResponse(apiMode, Cmd.WEEKLY_HIST, word), WeeklyHist.class);
        int taintedCount = 0;
        while (weeklyHist.isTainted() && taintedCount < MAX_TAINTED_COUNT) {
            taintedCount++;
            weeklyHist = convertYamlToBean(getApiResponse(apiMode, Cmd.WEEKLY_HIST, word), WeeklyHist.class);
        }

        return weeklyHist;
    }

    /**
     * Конвертирует YAML в бин MonthlyHist. Выполняет запрос к апи,
     * пока не получит бин, у которого tainted=false,
     * или пока количество попыток не превысит MAX_TAINTED_COUNT,
     * иначе вернет последний полученный бин.
     * @param apiMode Тестируемая среда
     * @param word Фраза, по которой нужен срез
     * @return MonthlyHist bean
     */
    public static MonthlyHist getMonthlyHist(ApiMode apiMode, Word word) {
        Yaml yaml = new Yaml();
        MonthlyHist monthlyHist = yaml.loadAs(getApiResponse(apiMode, Cmd.MONTHLY_HIST, word), MonthlyHist.class);
        int attempts = 0;
        while (monthlyHist.isTainted() && attempts < MAX_TAINTED_COUNT) {
            attempts++;
            monthlyHist = yaml.loadAs(getApiResponse(apiMode, Cmd.MONTHLY_HIST, word), MonthlyHist.class);
        }

        return monthlyHist;
    }

    /**
     * Обертка для конвертации YAML в JavaBean
     * Нужен для понятного сообщения в случаи если конвертация не удалась.
     */
    public static <YamlBean> YamlBean convertYamlToBean(String yaml, Class<YamlBean> type) {
        Yaml yamlConverter = new Yaml();
        try {
            return yamlConverter.loadAs(yaml, type);
        } catch (ConstructorException e) {
            throw new AdvqError("Не удалось сконвертировать YAML в JavaBean " + type.getSimpleName(), e);
        }
    }

}
