package ru.yandex.autotests.allure;

import java.io.StringWriter;
import java.io.Writer;
import java.util.HashMap;
import java.util.Map;

import freemarker.cache.ClassTemplateLoader;
import freemarker.cache.TemplateLoader;
import freemarker.template.Configuration;
import freemarker.template.DefaultObjectWrapper;
import freemarker.template.Template;
import org.junit.runner.Description;
import org.junit.runner.notification.RunListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.qatools.allure.Allure;
import ru.yandex.qatools.allure.events.MakeAttachmentEvent;

/**
 * Listener, который в каждый аллюрный отчет по тесту добавляет страничку
 * с дополнительной информацией и действиями
 *
 * @author upcfrost
 */
@RunListener.ThreadSafe
public class AllureSandboxReportActionsAttachListener extends RunListener {
    private static final String SANDBOX_LAUNCH_PROPERTY = "Sandbox.Launch";
    private static final Logger logger = LoggerFactory.getLogger(AllureSandboxReportActionsAttachListener.class);

    public AllureSandboxReportActionsAttachListener() {
        logger.error("AllureSandboxReportActionsAttachListener attached");
    }

    @Override
    public void testFinished(Description description) {
        //Если нет проперти с определением среды аэро, считаем что это локальный запуск и ниего не делаем
        if (System.getProperty(SANDBOX_LAUNCH_PROPERTY) == null) {
            return;
        }

        String testClassName = description.getClassName();
        String testMethodName = description.getMethodName();
        //Смотрим на какой акве (прод/тестинг) запущен тест

        Map<String, String> root = new HashMap<>();
        root.put("testClassName", testClassName);
        root.put("testMethodName", testMethodName);

        String attachmentHTML;

        try {
            TemplateLoader file = new ClassTemplateLoader(this.getClass(), "/");
            Configuration cfg = new Configuration();
            cfg.setTemplateLoader(file);
            cfg.setObjectWrapper(new DefaultObjectWrapper());
            cfg.setDefaultEncoding("UTF-8");
            //Имя файла в котором лежим темплейт для аттачмента
            Template template = cfg.getTemplate("sandboxTestNameAttachment.ftl");
            Writer fileWriter = new StringWriter();
            try {
                template.process(root, fileWriter);
            } finally {
                fileWriter.close();
            }
            attachmentHTML = fileWriter.toString();
        } catch (Exception e) {
            throw new RuntimeException("Ошибка", e);
        }

        Allure.LIFECYCLE.fire(new MakeAttachmentEvent(attachmentHTML.getBytes(),
                "Дополнительная информация", "text/html"));
    }
}
