package ru.yandex.autotests.direct.api.adextensions.delete;

import org.apache.commons.lang.RandomStringUtils;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.adextensions.AdExtensionsFeatures;
import ru.yandex.autotests.direct.api.adextensions.AdExtensionsLogins;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.model.api5.adextensions.DeleteRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.IdsCriteriaMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.apache.commons.lang3.StringUtils.capitalize;
import static org.hamcrest.Matchers.equalTo;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by pavryabov on 30.03.16.
 * https://st.yandex-team.ru/TESTIRT-8935
 */
@Aqua.Test
@Features(AdExtensionsFeatures.DELETE)
@Description("Проверка правильности списания баллов при удалении уточнений")
@Issue("https://st.yandex-team.ru/DIRECT-50725")
public class AdExtensionsDeleteUnitsTest {
    //DIRECT-52726

    private static final String CLIENT = AdExtensionsLogins.UNITS_DELETE_LOGIN;

    private static final int CALL_COST = 10;
    private static final int AD_EXTENSION_COST = 1;
    private static final int DEFAULT_ERROR_COST = 20;
    private static final int COMMON_REQUEST_ERROR = 50;

    @ClassRule
    public static final ApiSteps api = new ApiSteps().version(104).as(CLIENT);

    @ClassRule
    public static final SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private Long id1;

    @Before
    @Step("Подготовка данных для теста")
    public void initTest() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(CLIENT);
        id1 = api.userSteps.adExtensionsSteps().addCalloutWithText(RandomStringUtils.randomAlphabetic(8));
    }

    @Test
    @Description("Удалить одно уточнение")
    @ru.yandex.qatools.allure.annotations.TestCaseId("2175")
    public void deleteOneAdExtension() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(CLIENT);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(CLIENT);
        api.userSteps.adExtensionsSteps().adExtensionsDelete(id1);
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(CLIENT);
        assertThat("списалось правильное количество баллов",
                unitsBefore - unitsAfter, equalTo(CALL_COST + AD_EXTENSION_COST));
    }

    @Test
    @Description("Удалить два уточнения")
    @ru.yandex.qatools.allure.annotations.TestCaseId("2176")
    public void deleteTwoAdExtensions() {
        Long id2 = api.userSteps.adExtensionsSteps().addCalloutWithText(RandomStringUtils.randomAlphabetic(8));
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(CLIENT);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(CLIENT);
        api.userSteps.adExtensionsSteps().adExtensionsDelete(id1, id2);
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(CLIENT);
        assertThat("списалось правильное количество баллов",
                unitsBefore - unitsAfter, equalTo(CALL_COST + 2 * AD_EXTENSION_COST));
    }

    @Test
    @Description("Запрос на удаление уточнения с ошибкой")
    @ru.yandex.qatools.allure.annotations.TestCaseId("2177")
    public void deleteOneAdExtensionWithError() {
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(CLIENT);
        api.userSteps.adExtensionsSteps().adExtensionsDelete(-id1);
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(CLIENT);
        assertThat("списалось правильное количество баллов",
                unitsBefore - unitsAfter, equalTo(CALL_COST + DEFAULT_ERROR_COST));
    }

    @Test
    @Description("Запрос на удаление двух уточнений с одной ошибкой")
    @ru.yandex.qatools.allure.annotations.TestCaseId("2178")
    public void deleteTwoAdExtensionsWithOneError() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(CLIENT);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(CLIENT);
        api.userSteps.adExtensionsSteps().adExtensionsDelete(id1, -id1);
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(CLIENT);
        assertThat("списалось правильное количество баллов",
                unitsBefore - unitsAfter, equalTo(CALL_COST + DEFAULT_ERROR_COST + AD_EXTENSION_COST));
    }

    @Test
    @Description("Запрос на удаление уточнений с двумя ошибками")
    @ru.yandex.qatools.allure.annotations.TestCaseId("2179")
    public void deleteTwoAdExtensionsWithTwoErrors() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(CLIENT);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(CLIENT);
        api.userSteps.adExtensionsSteps().adExtensionsDelete(-id1, -id1);
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(CLIENT);
        assertThat("списалось правильное количество баллов",
                unitsBefore - unitsAfter, equalTo(CALL_COST + 2 * DEFAULT_ERROR_COST));
    }

    @Test
    @Description("Невалидный запрос на удаление уточнения")
    @ru.yandex.qatools.allure.annotations.TestCaseId("2180")
    public void deleteWithInvalidRequest() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(CLIENT);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(CLIENT);
        api.userSteps.adExtensionsSteps().expectErrorOnAdExtensionsDelete(
                new DeleteRequestMap().withSelectionCriteria(new IdsCriteriaMap()),
                new Api5Error(8000, Api5ErrorDetailsJava.MISSING_PARAMETER_VALUE,
                        capitalize(IdsCriteriaMap.IDS)));
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(CLIENT);
        assertThat("списалось правильное количество баллов",
                unitsBefore - unitsAfter, equalTo(COMMON_REQUEST_ERROR));
    }

    @Test
    @Description("Запрос на удаление уточнения, когда у клиента отсутствуют баллы")
    @ru.yandex.qatools.allure.annotations.TestCaseId("2181")
    public void callAdExtensionsDeleteWithZeroUnits() {
        int units = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(CLIENT);
        api.userSteps.clientFakeSteps().fakeWithdrawClientUnits(CLIENT, units);
        units = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(CLIENT);
        assumeThat("у клиента не осталось баллов", units, equalTo(0));
        api.userSteps.adExtensionsSteps().expectErrorOnAdExtensionsDelete(new DeleteRequestMap()
                        .withSelectionCriteria(new IdsCriteriaMap().withIds(id1)),
                new Api5Error(152, Api5ErrorDetails.NOT_ENOUGH_UNITS_FOR_OPERATION)
        );
    }
}
