package ru.yandex.autotests.direct.api.adextensions.delete;

import org.apache.commons.lang.RandomStringUtils;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.adextensions.AdExtensionsFeatures;
import ru.yandex.autotests.direct.api.adextensions.AdExtensionsLogins;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorMessage;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.IdsCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

/**
 * Created by pavryabov on 31.03.16.
 * https://st.yandex-team.ru/TESTIRT-8935
 */
@Aqua.Test
@Features(AdExtensionsFeatures.DELETE)
@Description("Id, которые должны вызывать ошибку в запросе на удаление уточнения")
@Issue("https://st.yandex-team.ru/DIRECT-50725")
public class IdsNegativeTest {

    private static final String LOGIN = AdExtensionsLogins.AD_EXTENSIONS_CLIENT;
    private static final String ANOTHER_LOGIN = AdExtensionsLogins.AD_EXTENSIONS_CLIENT2;

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).as(LOGIN);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private Long id;

    @BeforeClass
    public static void initTest() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(LOGIN);
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(ANOTHER_LOGIN);
    }

    @Before
    @Step("Подготовка данных для теста")
    public void createAd() {
        api.as(LOGIN);
        id = api.userSteps.adExtensionsSteps().addCalloutWithText(RandomStringUtils.randomAlphabetic(8));
    }

    @Test
    @Description("Нулевой идентификатор уточнения в запросе")
    @ru.yandex.qatools.allure.annotations.TestCaseId("2197")
    public void deleteZeroId() {
        api.userSteps.adExtensionsSteps()
                .shouldGetResultOnDelete(new IdsCriteriaMap().withIds(0l),
                        ExpectedResult.errors(new Notification(5005,
                                Api5ErrorDetailsJava.ID_MUST_BE_POSITIVE_INTEGER)));
    }

    @Test
    @Description("Чужой идентификатор уточнения в запросе")
    @ru.yandex.qatools.allure.annotations.TestCaseId("2199")
    public void deleteNotMyId() {
        api.as(ANOTHER_LOGIN);
        api.userSteps.adExtensionsSteps()
                .shouldGetResultOnDelete(new IdsCriteriaMap().withIds(id),
                        ExpectedResult
                                .errors(new Notification(8800, Api5ErrorDetailsJava.AD_EXTENSION_NOT_FOUND_NO_ID)));
    }

    @Test
    @Description("Удалённый идентификатор уточнения в запросе")
    @ru.yandex.qatools.allure.annotations.TestCaseId("2200")
    public void deleteDeletedId() {
        api.userSteps.adExtensionsSteps().adExtensionsDelete(id);
        api.userSteps.adExtensionsSteps()
                .shouldGetResultOnDelete(new IdsCriteriaMap().withIds(id),
                        ExpectedResult.warnings(id,
                                new Notification(10025, Api5ErrorDetailsJava.AD_EXTENSION_IS_MARKED_AS_DELETED)
                                        .withMessageFromResource(Api5ErrorMessage.OBJECT_IS_MARKED_AS_DELETED_JAVA)
                        ));
    }

    @Test
    @Description("Несуществующий идентификатор уточнения в запросе")
    @ru.yandex.qatools.allure.annotations.TestCaseId("2201")
    public void deleteNonExistentId() {
        api.userSteps.adExtensionsSteps()
                .shouldGetResultOnDelete(new IdsCriteriaMap().withIds(id + 500_000),
                        ExpectedResult
                                .errors(new Notification(8800, Api5ErrorDetailsJava.AD_EXTENSION_NOT_FOUND_NO_ID)));
    }

    @Test
    @Description("Отрицательный идентификатор уточнения в запросе")
    @ru.yandex.qatools.allure.annotations.TestCaseId("2202")
    public void deleteNegativeId() {
        api.userSteps.adExtensionsSteps()
                .shouldGetResultOnDelete(new IdsCriteriaMap().withIds(-id),
                        ExpectedResult.errors(new Notification(5005,
                                Api5ErrorDetailsJava.ID_MUST_BE_POSITIVE_INTEGER)));
    }

    @Test
    @Description("Дублирование идентификатора уточнения в запросе")
    @ru.yandex.qatools.allure.annotations.TestCaseId("2198")
    public void deleteOneIdTwoTimesInRequest() {
        api.userSteps.adExtensionsSteps()
                .shouldGetResultOnDelete(new IdsCriteriaMap().withIds(id, id),
                        ExpectedResult.errors(new Notification(9800,
                                Api5ErrorDetailsJava.AD_EXTENSION_ID_PRESENT_MORE_THAN_ONCE_IN_REQUEST)),
                        ExpectedResult.errors(new Notification(9800,
                                Api5ErrorDetailsJava.AD_EXTENSION_ID_PRESENT_MORE_THAN_ONCE_IN_REQUEST)));
    }
}
