package ru.yandex.autotests.direct.api.adgroups.add;

import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.adgroups.AdGroupsFeatures;
import ru.yandex.autotests.direct.api.adgroups.AdGroupsLogins;
import ru.yandex.autotests.directapi.ApiStories;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.model.api5.adgroups.AdGroupAddItemMap;
import ru.yandex.autotests.directapi.model.api5.adgroups.AddRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Stories;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.apache.commons.lang3.StringUtils.capitalize;
import static org.hamcrest.Matchers.equalTo;
import static ru.yandex.autotests.directapi.rules.ApiSteps.USE_OPERATOR_UNITS_AUTO;
import static ru.yandex.autotests.directapi.rules.ApiSteps.USE_OPERATOR_UNITS_FALSE;
import static ru.yandex.autotests.directapi.rules.ApiSteps.USE_OPERATOR_UNITS_TRUE;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

/**
 * Created by pavryabov on 10.01.15.
 */
@Aqua.Test
@Features(AdGroupsFeatures.ADD)
@Stories(ApiStories.UNITS)
public class AdGroupsAddAgencyUnitsTest {
    private static final String agency = AdGroupsLogins.AGENCY_UNITS;
    private static final String subclient = AdGroupsLogins.SUBCLIENT_UNITS;

    @ClassRule
    public static final ApiSteps api = new ApiSteps().version(104);

    @ClassRule
    public static final SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trashman = new Trashman(api);

    static Long campaignId;

    private static final int ADD_COST = 20;
    private static final int ADD_COST_GROUP = 20;
    private static final int DEFAULT_ERROR_COST = 20;
    private static final int GENERAL_ERROR_COST = 50;

    @BeforeClass
    public static void initTest() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(agency);
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(subclient);
        campaignId = api.as(agency).userSteps.campaignSteps().addDefaultTextCampaign(subclient);
    }

    @Before
    public void clearSpentUnits() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(agency);
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(subclient);
    }

    @Test
    public void agencyAddOneGroup() {
        //DIRECT-38060
        api.as(agency);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(agency);
        api.userSteps.adGroupsSteps().adGroupsAdd(
                new AddRequestMap().withAdGroups(
                        new AdGroupAddItemMap().defaultAdGroupAddItem(campaignId)),
                subclient);
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(agency);
        assertThat("списалось правильное количество баллов",
                unitsAfter, equalTo(unitsBefore));
    }

    @Test
    public void agencyAddNoSubclientLogin() {
        api.as(agency);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(agency);
        api.userSteps.adGroupsSteps().expectErrorOnAdGroupsAdd(
                new AddRequestMap().withAdGroups(
                        new AdGroupAddItemMap().defaultAdGroupAddItem(campaignId)),
                new Api5Error(8000, Api5ErrorDetailsJava.NOT_CLIENT_IN_CLIENT_LOGIN)
        );
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(agency);
        assertThat("списалось правильное количество баллов", unitsBefore - unitsAfter, equalTo(GENERAL_ERROR_COST));
    }

    @Test
    public void subclientAddOneGroup() {
        api.as(subclient);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(subclient);
        api.userSteps.clientFakeSteps().enableAllowEditCampaigns(subclient);
        api.as(subclient).userSteps.adGroupsSteps().addDefaultGroup(campaignId);
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(subclient);
        assertThat("списалось правильное количество баллов",
                unitsBefore - unitsAfter, equalTo(ADD_COST + ADD_COST_GROUP));
    }

    @Test
    public void agencyAddOneGroupCheckSubclientUnits() {
        //DIRECT-38060
        api.as(agency);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(subclient);
        api.userSteps.adGroupsSteps().adGroupsAdd(
                new AddRequestMap().withAdGroups(
                        new AdGroupAddItemMap().defaultAdGroupAddItem(campaignId)),
                subclient);
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(subclient);
        assertThat("баллы субклиента не изменились", unitsBefore - unitsAfter, equalTo(ADD_COST + ADD_COST_GROUP));
    }

    @Test
    public void agencyAddOneGroupUseOperatorUnitsTrueAgencyHasEnoughUnits() {
        //DIRECT-53823
        api.as(agency);
        int operatorUnitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(agency);
        int subclientUnitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(subclient);
        api.useOperatorUnits(USE_OPERATOR_UNITS_TRUE).userSteps.adGroupsSteps().adGroupsAdd(
                new AddRequestMap().withAdGroups(
                        new AdGroupAddItemMap().defaultAdGroupAddItem(campaignId)),
                subclient);
        int operatorUnitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(agency);
        int subclientUnitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(subclient);
        assertThat("списались баллы с оператора (агентства)", operatorUnitsBefore - operatorUnitsAfter,
                equalTo(ADD_COST + ADD_COST_GROUP));
        assertThat("баллы субклиента не изменились", subclientUnitsAfter, equalTo(subclientUnitsBefore));
    }

    @Test(expected = Exception.class)
    public void agencyAddOneGroupUseOperatorUnitsTrueAgencyHasNotEnoughUnits() {
        api.userSteps.clientFakeSteps()
                .fakeWithdrawClientUnits(agency, api.userSteps.clientFakeSteps().fakeClientUnitsBalance(agency));
        api.as(agency);
        api.useOperatorUnits(USE_OPERATOR_UNITS_TRUE).userSteps.adGroupsSteps().adGroupsAdd(
                new AddRequestMap().withAdGroups(
                        new AdGroupAddItemMap().defaultAdGroupAddItem(campaignId)),
                subclient);
    }

    @Test
    public void agencyAddOneGroupUseOperatorUnitsAutoClientHasEnoughUnits() {
        api.as(agency);
        int operatorUnitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(agency);
        int subclientUnitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(subclient);
        api.useOperatorUnits(USE_OPERATOR_UNITS_AUTO).userSteps.adGroupsSteps().adGroupsAdd(
                new AddRequestMap().withAdGroups(
                        new AdGroupAddItemMap().defaultAdGroupAddItem(campaignId)),
                subclient);
        int operatorUnitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(agency);
        int subclientUnitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(subclient);
        assertThat("списались баллы субклиента", subclientUnitsBefore - subclientUnitsAfter,
                equalTo(ADD_COST + ADD_COST_GROUP));
        assertThat("баллы оператора (агентства) не изменились", operatorUnitsBefore,
                equalTo(operatorUnitsAfter));
    }

    @Test
    public void agencyAddOneGroupUseOperatorUnitsAutoClientHasNotEnoughUnits() {
        api.userSteps.clientFakeSteps()
                .fakeWithdrawClientUnits(subclient, api.userSteps.clientFakeSteps().fakeClientUnitsBalance(subclient));
        api.as(agency);
        int operatorUnitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(agency);
        int subclientUnitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(subclient);
        api.useOperatorUnits(USE_OPERATOR_UNITS_AUTO).userSteps.adGroupsSteps().adGroupsAdd(
                new AddRequestMap().withAdGroups(
                        new AdGroupAddItemMap().defaultAdGroupAddItem(campaignId)),
                subclient);
        int operatorUnitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(agency);
        int subclientUnitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(subclient);
        assertThat("баллы субклиента не изменились", subclientUnitsAfter, equalTo(subclientUnitsBefore));
        assertThat("списались баллы с оператора (агентства)", operatorUnitsBefore - operatorUnitsAfter,
                equalTo(ADD_COST + ADD_COST_GROUP));
    }

    @Test(expected = Exception.class)
    public void agencyAddOneGroupUseOperatorUnitsAutoBothHaveNotEnoughUnits() {
        api.userSteps.clientFakeSteps()
                .fakeWithdrawClientUnits(subclient, api.userSteps.clientFakeSteps().fakeClientUnitsBalance(subclient));
        api.userSteps.clientFakeSteps()
                .fakeWithdrawClientUnits(agency, api.userSteps.clientFakeSteps().fakeClientUnitsBalance(agency));
        api.as(agency);
        api.useOperatorUnits(USE_OPERATOR_UNITS_AUTO).userSteps.adGroupsSteps().adGroupsAdd(
                new AddRequestMap().withAdGroups(
                        new AdGroupAddItemMap().defaultAdGroupAddItem(campaignId)),
                subclient);
    }

    @Test
    public void agencyAddOneGroupUseOperatorUnitsFalseClientHasEnoughUnits() {
        api.as(agency);
        int operatorUnitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(agency);
        int subclientUnitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(subclient);
        api.useOperatorUnits(USE_OPERATOR_UNITS_FALSE).userSteps.adGroupsSteps().adGroupsAdd(
                new AddRequestMap().withAdGroups(
                        new AdGroupAddItemMap().defaultAdGroupAddItem(campaignId)),
                subclient);
        int operatorUnitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(agency);
        int subclientUnitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(subclient);
        assertThat("списались баллы субклиента", subclientUnitsBefore - subclientUnitsAfter,
                equalTo(ADD_COST + ADD_COST_GROUP));
        assertThat("баллы оператора (агентства) не изменились", operatorUnitsBefore,
                equalTo(operatorUnitsAfter));
    }

    @Test(expected = Exception.class)
    public void agencyAddOneGroupUseOperatorUnitsFalseClientHasNotEnoughUnits() {
        api.userSteps.clientFakeSteps()
                .fakeWithdrawClientUnits(subclient, api.userSteps.clientFakeSteps().fakeClientUnitsBalance(subclient));
        api.as(agency);
        api.useOperatorUnits(USE_OPERATOR_UNITS_FALSE).userSteps.adGroupsSteps().adGroupsAdd(
                new AddRequestMap().withAdGroups(
                        new AdGroupAddItemMap().defaultAdGroupAddItem(campaignId)),
                subclient);
    }

    @Test
    public void subclientAddOneGroupCheckAgencyUnits() {
        api.as(subclient);
        int unitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(agency);
        api.userSteps.adGroupsSteps().adGroupsAdd(
                new AddRequestMap().withAdGroups(
                        new AdGroupAddItemMap().defaultAdGroupAddItem(campaignId)
                )
        );
        int unitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(agency);
        assertThat("баллы агентства не изменились", unitsAfter, equalTo(unitsBefore));
    }

    @Test
    public void subclientAddOneGroupCheckErrorForUseOperatorUnits() {
        //DIRECT-53823
        api.as(subclient);
        api.useOperatorUnits(USE_OPERATOR_UNITS_TRUE).userSteps.adGroupsSteps().expectErrorOnAdGroupsAdd(
                new AddRequestMap().withAdGroups(
                        new AdGroupAddItemMap().defaultAdGroupAddItem(campaignId)
                ),
                new Api5Error(8000, Api5ErrorDetails.HEADER_USE_OPERATOR_UNITS_INACCESSIBLE));
    }

    @Test
    public void agencyAddOneGroupWithObjectErrorCheckUnits() {
        //DIRECT-56906
        api.as(agency);
        int operatorUnitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(agency);
        int subclientUnitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(subclient);
        api.userSteps.adGroupsSteps().shouldGetResultOnAdd(
                new AddRequestMap().withAdGroups(
                        new AdGroupAddItemMap().defaultAdGroupAddItem(0L)),
                subclient,
                ExpectedResult.errors(new Notification(5005,
                        Api5ErrorDetailsJava.FIELD_MUST_BE_POSITIVE_INTEGER,
                        capitalize(AdGroupAddItemMap.CAMPAIGN_ID))));
        int operatorUnitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(agency);
        int subclientUnitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(subclient);
        assertThat("списались баллы с субклиента", subclientUnitsBefore - subclientUnitsAfter,
                equalTo(ADD_COST + DEFAULT_ERROR_COST));
        assertThat("баллы агентства не изменились", operatorUnitsAfter, equalTo(operatorUnitsBefore));
    }

    @Test
    public void agencyAddOneGroupWithRequestErrorCheckUnits() {
        //DIRECT-56906
        api.as(agency);
        int operatorUnitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(agency);
        int subclientUnitsBefore = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(subclient);
        api.userSteps.adGroupsSteps().expectErrorOnAdGroupsAdd(
                new AddRequestMap().withAdGroups(
                        new AdGroupAddItemMap()
                                .withCampaignId(campaignId)
                                .withName(AdGroupAddItemMap.DEFAULT_NAME)
                ),
                subclient,
                new Api5Error(8000, Api5ErrorDetailsJava.IN_ELEMENT_OF_ARRAY_MISSING_REQUIRED_FIELD,
                        capitalize(AddRequestMap.AD_GROUPS),
                        capitalize(AdGroupAddItemMap.REGION_IDS)));

        int operatorUnitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(agency);
        int subclientUnitsAfter = api.userSteps.clientFakeSteps().fakeClientUnitsBalance(subclient);
        assertThat("списались баллы с агентства", operatorUnitsBefore - operatorUnitsAfter,
                equalTo(GENERAL_ERROR_COST));
        assertThat("баллы субклиента не изменились", subclientUnitsAfter, equalTo(subclientUnitsBefore));
    }
}
