package ru.yandex.autotests.direct.api.adgroups.add;

import java.util.HashMap;

import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.adgroups.AdGroupsFeatures;
import ru.yandex.autotests.direct.api.adgroups.AdGroupsLogins;
import ru.yandex.autotests.directapi.ApiStories;
import ru.yandex.autotests.directapi.apiclient.config.ProtocolType;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.apiclient.version5.ServiceNames;
import ru.yandex.autotests.directapi.model.api5.Action;
import ru.yandex.autotests.directapi.model.api5.adgroups.AdGroupAddItemMap;
import ru.yandex.autotests.directapi.model.api5.adgroups.AddRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Stories;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.apache.commons.lang3.StringUtils.capitalize;
import static ru.yandex.autotests.directapi.apiclient.errors.Path.path;

/**
 * Created by pavryabov on 09.12.14.
 * https://st.yandex-team.ru/TESTIRT-3706
 */
@Aqua.Test
@Features(AdGroupsFeatures.ADD)
@Stories(ApiStories.JSON5)
public class AddJSONTest {

    private static final String CLIENT = AdGroupsLogins.CLIENT;

    @ClassRule
    public static final ApiSteps api = new ApiSteps().version(104).protocol(ProtocolType.JSON).as(CLIENT);

    @ClassRule
    public static final SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trashman = new Trashman(api);

    static Long campaignId;

    private static class JSONObject extends HashMap<String, Object> {
    }

    @BeforeClass
    public static void initTest() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(CLIENT);
        campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign(CLIENT);
    }

    @Before
    public void clearSpentUnits() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(AdGroupsLogins.CLIENT);
    }

    @Test
    public void addWithValidParams() {
        api.userSteps.adGroupsSteps().shouldGetResultOnAdd(
                new AddRequestMap().withAdGroups(
                        new AdGroupAddItemMap().defaultAdGroupAddItem(campaignId)
                ),
                ExpectedResult.success()
        );
    }

    @Test
    public void addWithInvalidAndValidGroups() {
        api.userSteps.adGroupsSteps().shouldGetResultOnAdd(
                new AddRequestMap().withAdGroups(
                        new AdGroupAddItemMap().defaultAdGroupAddItem(campaignId + 10000),
                        new AdGroupAddItemMap().defaultAdGroupAddItem(campaignId)
                ),
                ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.CAMPAIGN_NOT_FOUND)),
                ExpectedResult.success()
        );
    }

    @Test
    public void addWithoutCampaignId() {
        JSONObject request = new JSONObject();
        JSONObject adGroups = new JSONObject();
        adGroups.put(capitalize(AdGroupAddItemMap.NAME), AdGroupAddItemMap.DEFAULT_NAME);
        adGroups.put(capitalize(AdGroupAddItemMap.REGION_IDS), new Long[]{AdGroupAddItemMap.DEFAULT_REGION_ID});
        request.put(capitalize(AddRequestMap.AD_GROUPS), new Object[]{adGroups});
        api.userSteps.adGroupsSteps().expectErrorOnAdGroupsAdd(
                request,
                new Api5Error(8000, Api5ErrorDetailsJava.IN_ELEMENT_OF_ARRAY_MISSING_REQUIRED_FIELD,
                        capitalize(AddRequestMap.AD_GROUPS),
                        capitalize(AdGroupAddItemMap.CAMPAIGN_ID)));
    }

    @Test
    public void addWithStringInCampaignId() {
        JSONObject request = new JSONObject();
        JSONObject adGroups = new JSONObject();
        adGroups.put(capitalize(AdGroupAddItemMap.NAME), AdGroupAddItemMap.DEFAULT_NAME);
        adGroups.put(capitalize(AdGroupAddItemMap.CAMPAIGN_ID), "string");
        adGroups.put(capitalize(AdGroupAddItemMap.REGION_IDS), new Long[]{AdGroupAddItemMap.DEFAULT_REGION_ID});
        request.put(capitalize(AddRequestMap.AD_GROUPS), new Object[]{adGroups});
        api.userSteps.adGroupsSteps().expectErrorOnAdGroupsAdd(
                request,
                new Api5Error(8000, Api5ErrorDetailsJava.FIELD_MUST_CONTAIN_INTEGER_VALUE,
                        path(AddRequestMap.AD_GROUPS, AdGroupAddItemMap.CAMPAIGN_ID)));
    }


    @Test
    @Issue("https://st.yandex-team.ru/DIRECT-64794")
    public void addAuthErrorOnNoClientLoginHeaderFromAgencyDespiteInvalidRequest() {
        JSONObject request = new JSONObject();
        JSONObject adGroups = new JSONObject();
        adGroups.put(capitalize(AdGroupAddItemMap.CAMPAIGN_ID), "string");
        request.put(capitalize(AddRequestMap.AD_GROUPS), new Object[]{adGroups});
        api.as(AdGroupsLogins.AGENCY).userSteps.adGroupsSteps().expectErrorOnAdGroupsAdd(
                request,
                new Api5Error(8000, Api5ErrorDetailsJava.NOT_CLIENT_IN_CLIENT_LOGIN)
        );
    }

    @Test
    public void addWithArrayInCampaignId() {
        JSONObject request = new JSONObject();
        JSONObject adGroups = new JSONObject();
        adGroups.put(capitalize(AdGroupAddItemMap.NAME), AdGroupAddItemMap.DEFAULT_NAME);
        adGroups.put(capitalize(AdGroupAddItemMap.CAMPAIGN_ID), new Long[]{campaignId});
        adGroups.put(capitalize(AdGroupAddItemMap.REGION_IDS), new Long[]{AdGroupAddItemMap.DEFAULT_REGION_ID});
        request.put(capitalize(AddRequestMap.AD_GROUPS), new Object[]{adGroups});

        api.userSteps.adGroupsSteps().expectErrorOnAdGroupsAdd(
                request,
                new Api5Error(8000, Api5ErrorDetailsJava.FIELD_MUST_NOT_CONTAIN_ARRAY,
                        path(AddRequestMap.AD_GROUPS, AdGroupAddItemMap.CAMPAIGN_ID)));
    }

    @Test
    public void addWithStringInRegionIds() {
        JSONObject request = new JSONObject();
        JSONObject adGroups = new JSONObject();
        adGroups.put(capitalize(AdGroupAddItemMap.NAME), AdGroupAddItemMap.DEFAULT_NAME);
        adGroups.put(capitalize(AdGroupAddItemMap.CAMPAIGN_ID), campaignId);
        adGroups.put(capitalize(AdGroupAddItemMap.REGION_IDS), new String[]{"string"});
        request.put(capitalize(AddRequestMap.AD_GROUPS), new Object[]{adGroups});

        api.userSteps.adGroupsSteps().expectErrorOnAdGroupsAdd(
                request,
                new Api5Error(8000, Api5ErrorDetailsJava.ELEMENT_OF_ARRAY_MUST_CONTAIN_INTEGER_VALUE,
                        path(AddRequestMap.AD_GROUPS, AdGroupAddItemMap.REGION_IDS)));
    }

    @Test
    public void addWithNotArrayInRegionIds() {
        //DIRECT-37558
        JSONObject request = new JSONObject();
        JSONObject adGroups = new JSONObject();
        adGroups.put(capitalize(AdGroupAddItemMap.NAME), AdGroupAddItemMap.DEFAULT_NAME);
        adGroups.put(capitalize(AdGroupAddItemMap.CAMPAIGN_ID), campaignId);
        adGroups.put(capitalize(AdGroupAddItemMap.REGION_IDS), AdGroupAddItemMap.DEFAULT_REGION_ID);
        request.put(capitalize(AddRequestMap.AD_GROUPS), new Object[]{adGroups});

        api.userSteps.adGroupsSteps().expectErrorOnAdGroupsAdd(
                request,
                new Api5Error(8000, Api5ErrorDetailsJava.FIELD_MUST_CONTAIN_ARRAY,
                        path(AddRequestMap.AD_GROUPS, AdGroupAddItemMap.REGION_IDS)));
    }

    @Test
    public void addWithInvalidField() {
        //DIRECT-37561
        JSONObject request = new JSONObject();
        JSONObject adGroups = new JSONObject();
        adGroups.put(capitalize(AdGroupAddItemMap.NAME), AdGroupAddItemMap.DEFAULT_NAME);
        adGroups.put(capitalize(AdGroupAddItemMap.CAMPAIGN_ID), campaignId);
        adGroups.put(AdGroupAddItemMap.CAMPAIGN_ID, campaignId);
        adGroups.put(capitalize(AdGroupAddItemMap.REGION_IDS), new Long[]{AdGroupAddItemMap.DEFAULT_REGION_ID});
        request.put(capitalize(AddRequestMap.AD_GROUPS), new Object[]{adGroups});

        api.userSteps.adGroupsSteps().expectErrorOnAdGroupsAdd(
                request,
                new Api5Error(8000, Api5ErrorDetailsJava.ELEMENT_OF_ARRAY_CONTAIN_UNKNOWN_FIELD,
                        path(AddRequestMap.AD_GROUPS), AdGroupAddItemMap.CAMPAIGN_ID));
    }

    @Test
    public void addWithNoItemsInNegativeKeywords() {
        JSONObject request = new JSONObject();
        JSONObject adGroups = new JSONObject();
        adGroups.put(capitalize(AdGroupAddItemMap.NAME), AdGroupAddItemMap.DEFAULT_NAME);
        adGroups.put(capitalize(AdGroupAddItemMap.CAMPAIGN_ID), campaignId);
        adGroups.put(capitalize(AdGroupAddItemMap.REGION_IDS), new Long[]{AdGroupAddItemMap.DEFAULT_REGION_ID});
        adGroups.put(capitalize(AdGroupAddItemMap.NEGATIVE_KEYWORDS),
                new String[]{AdGroupAddItemMap.DEFAULT_NEGATIVE_KEYWORD});
        request.put(capitalize(AddRequestMap.AD_GROUPS), new Object[]{adGroups});

        api.userSteps.adGroupsSteps().expectErrorOnAdGroupsAdd(
                request,
                new Api5Error(8000, Api5ErrorDetailsJava.FIELD_MUST_NOT_CONTAIN_ARRAY,
                        path(AddRequestMap.AD_GROUPS, AdGroupAddItemMap.NEGATIVE_KEYWORDS)));
    }

    @Test
    public void addWithNotArrayInAdGroups() {
        //DIRECT-37558
        JSONObject request = new JSONObject();
        JSONObject adGroups = new JSONObject();
        adGroups.put(capitalize(AdGroupAddItemMap.NAME), AdGroupAddItemMap.DEFAULT_NAME);
        adGroups.put(capitalize(AdGroupAddItemMap.CAMPAIGN_ID), campaignId);
        adGroups.put(capitalize(AdGroupAddItemMap.REGION_IDS), new Long[]{AdGroupAddItemMap.DEFAULT_REGION_ID});
        request.put(capitalize(AddRequestMap.AD_GROUPS), adGroups);
        api.userSteps.adGroupsSteps().expectErrorOnAdGroupsAdd(
                request,
                new Api5Error(8000, Api5ErrorDetails.FIELD_MUST_CONTAIN_ARRAY,
                        capitalize(AddRequestMap.AD_GROUPS))
        );
    }

    @Test
    public void callAdGroupsWithInvalidAction() {
        JSONObject request = new JSONObject();
        JSONObject adGroups = new JSONObject();
        adGroups.put(capitalize(AdGroupAddItemMap.NAME), AdGroupAddItemMap.DEFAULT_NAME);
        adGroups.put(capitalize(AdGroupAddItemMap.REGION_IDS), new Long[]{AdGroupAddItemMap.DEFAULT_REGION_ID});
        request.put(capitalize(AddRequestMap.AD_GROUPS), new Object[]{adGroups});

        api.userSteps.shouldGetJSONErrorOn(ServiceNames.AD_GROUPS, null, Action.ACTION_INVALID,
                request,
                new Api5Error(55, Api5ErrorDetailsJava.OPERATION_IS_NOT_SET));
    }

    @Test
    public void callAdGroupsWithEmptyAction() {
        JSONObject request = new JSONObject();
        JSONObject adGroups = new JSONObject();
        adGroups.put(capitalize(AdGroupAddItemMap.NAME), AdGroupAddItemMap.DEFAULT_NAME);
        adGroups.put(capitalize(AdGroupAddItemMap.REGION_IDS), new Long[]{AdGroupAddItemMap.DEFAULT_REGION_ID});
        request.put(capitalize(AddRequestMap.AD_GROUPS), new Object[]{adGroups});
        api.userSteps.shouldGetJSONErrorOn(ServiceNames.AD_GROUPS, null, Action.ACTION_EMPTY,
                request,
                new Api5Error(55, Api5ErrorDetailsJava.OPERATION_IS_NOT_SET));
    }

    @Test
    public void addWithNullInNegativeKeywords() {
        JSONObject request = new JSONObject();
        JSONObject adGroups = new JSONObject();
        adGroups.put(capitalize(AdGroupAddItemMap.NAME), AdGroupAddItemMap.DEFAULT_NAME);
        adGroups.put(capitalize(AdGroupAddItemMap.CAMPAIGN_ID), campaignId);
        adGroups.put(capitalize(AdGroupAddItemMap.REGION_IDS), new Long[]{AdGroupAddItemMap.DEFAULT_REGION_ID});
        adGroups.put(capitalize(AdGroupAddItemMap.NEGATIVE_KEYWORDS), "null");//будет восприниматься как null
        request.put(capitalize(AddRequestMap.AD_GROUPS), new Object[]{adGroups});

        api.userSteps.adGroupsSteps().expectErrorOnAdGroupsAdd(
                request,
                new Api5Error(8000, Api5ErrorDetails.CANNOT_HAVE_NULL_VALUE,
                        capitalize(AdGroupAddItemMap.NEGATIVE_KEYWORDS)));
    }
}
