package ru.yandex.autotests.direct.api.adgroups.add;

import java.util.Collection;
import java.util.List;
import java.util.function.Function;

import com.yandex.direct.api.v5.adgroups.AddResponse;
import com.yandex.direct.api.v5.general.ActionResult;
import org.hamcrest.Matcher;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.adgroups.AdGroupsFeatures;
import ru.yandex.autotests.direct.api.adgroups.AdGroupsLogins;
import ru.yandex.autotests.direct.utils.config.DirectTestRunProperties;
import ru.yandex.autotests.direct.utils.model.RegionIDValues;
import ru.yandex.autotests.direct.utils.model.RegionNameValues;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.apiclient.version5.ServiceNames;
import ru.yandex.autotests.directapi.model.api5.Action;
import ru.yandex.autotests.directapi.model.api5.adgroups.AdGroupAddItemMap;
import ru.yandex.autotests.directapi.model.api5.adgroups.AddRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Arrays.asList;
import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toList;
import static org.hamcrest.CoreMatchers.notNullValue;
import static org.hamcrest.Matchers.contains;
import static org.hamcrest.Matchers.either;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Author pavryabov
 * Date 15.08.14
 * https://st.yandex-team.ru/TESTIRT-2022
 */
@Aqua.Test
@Features(AdGroupsFeatures.ADD)
@RunWith(Parameterized.class)
public class AddRegionIdsNegativeTest {

    @ClassRule
    public static final ApiSteps api = new ApiSteps().as(AdGroupsLogins.CLIENT);

    @ClassRule
    public static final SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static final Long MOSCOW = RegionIDValues.MOSCOW.getId().longValue();
    private static final Long CHERNOGOLOVKA = RegionIDValues.CHERNOGOLOVKA.getId().longValue();
    private static final Long MOSCOW_REGION = RegionIDValues.MOSCOW_AND_MOSCOW_REGION.getId().longValue();
    private static final Long SPB = RegionIDValues.SPB.getId().longValue();
    private static final Long RUSSIA = RegionIDValues.RUSSIA_COUNTRY.getId().longValue();
    private static final Long CRIMEA = RegionIDValues.CRIMEA.getId().longValue();
    private static final Long UKRAINE = RegionIDValues.UKRAINE.getId().longValue();
    private static final Long EUROPE = RegionIDValues.EUROPE.getId().longValue();
    private static final Long SNG = RegionIDValues.SNG.getId().longValue();

    // TODO DIRECT-75345 perlError/javaError надо объединить в error, когда adgroups перейдут на java
    private static class TestCase {
        private String description;
        private List<Long> regionIds;
        private Notification perlError;
        private boolean hasJavaError;
        private Notification javaError;

        private TestCase(String description, List<Long> regionIds,
                Notification perlError, Notification javaError)
        {
            this.description = description;
            this.regionIds = regionIds;
            this.perlError = perlError;
            this.hasJavaError = true;
            this.javaError = checkNotNull(javaError);
        }

        @Override
        public String toString() {
            return description;
        }

        private Matcher<ActionResult> getMatcher() {
            if (hasJavaError) {
                return either(ExpectedResult.errors(perlError).getActionResultMatcher())
                        .or(ExpectedResult.errors(javaError).getActionResultMatcher());
            }

            return ExpectedResult.errors(perlError).getActionResultMatcher();
        }
    }

    @Parameterized.Parameter
    public TestCase testCase;

    @Parameterized.Parameters(name = "{0}")
    public static Collection testData() {
        String locale = DirectTestRunProperties.getInstance().getDirectApiLocale();
        Function<RegionNameValues, String> localRegionNameGetter = locale.equals("ru")
                ? RegionNameValues::getRussianName
                : RegionNameValues::getEnglishName;

        List<TestCase> testCases = asList(
                new TestCase("Duplicate region", asList(MOSCOW, CHERNOGOLOVKA, MOSCOW),
                        new Notification(5120, Api5ErrorDetails.REGION_IS_DULICATED,
                                localRegionNameGetter.apply(RegionNameValues.MOSCOW)),
                        new Notification(5120, Api5ErrorDetailsJava.REGION_IS_DUPLICATED,
                                localRegionNameGetter.apply(RegionNameValues.MOSCOW))),
                new TestCase("Only negative", singletonList(-MOSCOW),
                        new Notification(5120, Api5ErrorDetails.ALL_REGIONS_CANNOT_BE_EXCLUDED),
                        new Notification(5120, Api5ErrorDetailsJava.ALL_REGIONS_CANNOT_BE_EXCLUDED)),
                new TestCase("Only negatives", asList(-MOSCOW, -CHERNOGOLOVKA),
                        new Notification(5120, Api5ErrorDetails.ALL_REGIONS_CANNOT_BE_EXCLUDED),
                        new Notification(5120, Api5ErrorDetailsJava.ALL_REGIONS_CANNOT_BE_EXCLUDED)),
                new TestCase("Invalid region", singletonList(123456L),
                        new Notification(5120, Api5ErrorDetails.INVALID_OR_NONEXISTENT_REGION, 123456L),
                        new Notification(5120, Api5ErrorDetailsJava.INVALID_OR_NONEXISTENT_REGION, 123456L)),
                new TestCase("Exclude Crimea from Ukraine", asList(UKRAINE, -CRIMEA),
                        new Notification(5120,
                                Api5ErrorDetails.REGION_CANNOT_BE_EXLUDED_FOR_THIS_REQUEST,
                                localRegionNameGetter.apply(RegionNameValues.CRIMEA)),
                        new Notification(5120,
                                Api5ErrorDetailsJava.REGION_CANNOT_BE_EXLUDED_FOR_THIS_REQUEST,
                                localRegionNameGetter.apply(RegionNameValues.CRIMEA))),
                new TestCase("Exclude Crimea from Russia", asList(RUSSIA, -CRIMEA),
                        new Notification(5120,
                                Api5ErrorDetails.REGION_CANNOT_BE_EXLUDED_FOR_THIS_REQUEST,
                                localRegionNameGetter.apply(RegionNameValues.CRIMEA)),
                        new Notification(5120,
                                Api5ErrorDetailsJava.REGION_CANNOT_BE_EXLUDED_FOR_THIS_REQUEST,
                                localRegionNameGetter.apply(RegionNameValues.CRIMEA))),
                new TestCase("Exclude Crimea from Europe", asList(EUROPE, -CRIMEA),
                        new Notification(5120,
                                Api5ErrorDetails.REGION_CANNOT_BE_EXLUDED_FOR_THIS_REQUEST,
                                localRegionNameGetter.apply(RegionNameValues.CRIMEA)),
                        new Notification(5120,
                                Api5ErrorDetailsJava.REGION_CANNOT_BE_EXLUDED_FOR_THIS_REQUEST,
                                localRegionNameGetter.apply(RegionNameValues.CRIMEA))),
                new TestCase("Exclude Crimea from SNG and Ukraine", asList(SNG, UKRAINE, -CRIMEA),
                        new Notification(5120,
                                Api5ErrorDetails.REGION_CANNOT_BE_EXLUDED_FOR_THIS_REQUEST,
                                localRegionNameGetter.apply(RegionNameValues.CRIMEA)),
                        new Notification(5120,
                                Api5ErrorDetailsJava.REGION_CANNOT_BE_EXLUDED_FOR_THIS_REQUEST,
                                localRegionNameGetter.apply(RegionNameValues.CRIMEA))),
                new TestCase("Exclude parent from child", asList(MOSCOW, -MOSCOW_REGION),
                        new Notification(5120,
                                Api5ErrorDetails.REGION_CANNOT_BE_EXLUDED_FOR_THIS_REQUEST,
                                localRegionNameGetter.apply(RegionNameValues.MOSCOW_AND_MOSCOW_REGION)),
                        new Notification(5120,
                                Api5ErrorDetailsJava.REGION_CANNOT_BE_EXLUDED_FOR_THIS_REQUEST,
                                localRegionNameGetter.apply(RegionNameValues.MOSCOW_AND_MOSCOW_REGION))),
                new TestCase("Exclude other region", asList(MOSCOW, -SPB),
                        new Notification(5120,
                                Api5ErrorDetails.REGION_CANNOT_BE_EXLUDED_FOR_THIS_REQUEST,
                                localRegionNameGetter.apply(RegionNameValues.SAINT_PETERSBURG)),
                        new Notification(5120,
                                Api5ErrorDetailsJava.REGION_CANNOT_BE_EXLUDED_FOR_THIS_REQUEST,
                                localRegionNameGetter.apply(RegionNameValues.SAINT_PETERSBURG)))
        );

        return testCases.stream().map(testCase -> new Object[]{ testCase }).collect(toList());
    }

    private static AdGroupAddItemMap adGroupAddItemMap;

    @BeforeClass
    public static void createCampaign() {
        Long campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign(AdGroupsLogins.CLIENT);
        adGroupAddItemMap = new AdGroupAddItemMap()
                .withCampaignId(campaignId)
                .withName(AdGroupAddItemMap.DEFAULT_NAME);
    }

    @Before
    public void clearSpentUnits() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(AdGroupsLogins.CLIENT);
    }

    @Test
    public void addGroupWithRegionIds() {
        //DIRECT-34451, DIRECT-34452, DIRECT-34453, DIRECT-35721, DIRECT-35929

        AddRequestMap request = new AddRequestMap()
                .withAdGroups(adGroupAddItemMap.withRegionIds(testCase.regionIds));

        AddResponse response = api.userSteps.adGroupsSteps().defaultClientV5()
                .invokeMethod(ServiceNames.AD_GROUPS, null, Action.ADD, request.getBean());

        assumeThat("получен ответ сервиса", response, notNullValue());

        assertThat("содержание результатов соответствует ожидаемому", response.getAddResults(),
                contains(testCase.getMatcher()));

    }
}
