package ru.yandex.autotests.direct.api.adgroups.add;

import java.util.Arrays;

import com.yandex.direct.api.v5.adgroups.AddResponse;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.adgroups.AdGroupsFeatures;
import ru.yandex.autotests.direct.api.adgroups.AdGroupsLogins;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.darkside.datacontainers.jsonrpc.fake.GroupFakeInfo;
import ru.yandex.autotests.directapi.darkside.model.Status;
import ru.yandex.autotests.directapi.model.api5.adgroups.AdGroupAddItemMap;
import ru.yandex.autotests.directapi.model.api5.adgroups.AddRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.contains;
import static org.hamcrest.Matchers.either;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.notNullValue;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Author pavryabov
 * Date 14.07.14
 * https://st.yandex-team.ru/TESTIRT-2022
 */
@Aqua.Test
@Tag(TagDictionary.TRUNK)
@Features(AdGroupsFeatures.ADD)
public class AddTest {

    private static final String client = AdGroupsLogins.CLIENT;

    @ClassRule
    public static final ApiSteps api = new ApiSteps().as(client);

    @ClassRule
    public static final SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trashman = new Trashman(api);

    static Long campaignId;

    @BeforeClass
    public static void initTest() {
        campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign(client);
    }

    @Before
    public void clearSpentUnits() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
    }

    @Test
    public void addWithValidParams() {
        api.userSteps.adGroupsSteps().shouldGetResultOnAdd(
                new AddRequestMap().withAdGroups(
                        new AdGroupAddItemMap().defaultAdGroupAddItem(campaignId)
                ),
                ExpectedResult.success()
        );
    }

    @Test
    public void addTwoGroupsWithSameName() {
        api.userSteps.adGroupsSteps().shouldGetResultOnAdd(
                new AddRequestMap().withAdGroups(
                        new AdGroupAddItemMap().defaultAdGroupAddItem(campaignId),
                        new AdGroupAddItemMap().defaultAdGroupAddItem(campaignId)
                ),
                ExpectedResult.success(), ExpectedResult.success()
        );
    }

    @Test
    public void add100Groups() {
        AdGroupAddItemMap[] adGroupAddItemMaps = new AdGroupAddItemMap[100];
        Arrays.fill(adGroupAddItemMaps, new AdGroupAddItemMap().defaultAdGroupAddItem(campaignId));
        ExpectedResult[] expectedResults = new ExpectedResult[100];
        Arrays.fill(expectedResults, ExpectedResult.success());
        api.userSteps.adGroupsSteps().shouldGetResultOnAdd(
                new AddRequestMap().withAdGroups(adGroupAddItemMaps),
                expectedResults
        );
    }

    @Test
    public void addTwoInvalidGroups() {
        api.userSteps.adGroupsSteps().shouldGetResultOnAdd(
                new AddRequestMap().withAdGroups(
                        new AdGroupAddItemMap().defaultAdGroupAddItem(campaignId + 10000),
                        new AdGroupAddItemMap().defaultAdGroupAddItem(campaignId + 10000)
                ),
                ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.CAMPAIGN_NOT_FOUND)),
                ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.CAMPAIGN_NOT_FOUND))
        );
    }

    @Test
    public void addWithInvalidAndValidGroups() {
        api.userSteps.adGroupsSteps().shouldGetResultOnAdd(
                new AddRequestMap().withAdGroups(
                        new AdGroupAddItemMap().defaultAdGroupAddItem(campaignId + 10000),
                        new AdGroupAddItemMap().defaultAdGroupAddItem(campaignId)
                ),
                ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.CAMPAIGN_NOT_FOUND)),
                ExpectedResult.success()
        );
    }

    @Test
    public void addWithTwoInvalidFieldsInOneGroup() {
        api.userSteps.adGroupsSteps().shouldGetResultOnAdd(
                new AddRequestMap().withAdGroups(
                        new AdGroupAddItemMap()
                                .withCampaignId(campaignId)
                                .withRegionIds(AdGroupAddItemMap.DEFAULT_REGION_ID)
                                .withName(" ")
                                .withNegativeKeywords("test]")
                ),
                ExpectedResult.errors(
                        new Notification(5002, Api5ErrorDetails.INCORRECT_USE_OF_BRACKETS_IN_MINUS_PHRASE, "test]"),
                        new Notification(5004, Api5ErrorDetails.GROUP_NAME_CANNOT_BE_EMPTY_STRING))
        );
    }

    @Test
    public void add1001GroupsInOneCampaign() {
        //DIRECT-34455
        Long campaignIdFor1000Groups = api.userSteps.campaignSteps().addDefaultTextCampaign(client);
        AdGroupAddItemMap[] adGroupAddItemMaps = new AdGroupAddItemMap[100];
        Arrays.fill(adGroupAddItemMaps, new AdGroupAddItemMap().defaultAdGroupAddItem(campaignIdFor1000Groups));
        for (int i = 0; i < 10; i++) {
            api.userSteps.adGroupsSteps().adGroupsAdd(new AddRequestMap().withAdGroups(adGroupAddItemMaps));
        }
        api.userSteps.adGroupsSteps().shouldGetResultOnAdd(
                new AddRequestMap().withAdGroups(
                        new AdGroupAddItemMap().defaultAdGroupAddItem(campaignIdFor1000Groups)),
                ExpectedResult.errors(
                        new Notification(7001, Api5ErrorDetailsJava.MAXIMUM_GROUPS_IN_CAMPAIGN_REACHED))
        );
    }

    @Test
    public void addWithDifferentSetsOfFields() {
        api.userSteps.adGroupsSteps().shouldGetResultOnAdd(
                new AddRequestMap().withAdGroups(
                        new AdGroupAddItemMap().defaultAdGroupAddItem(campaignId)
                                .withNegativeKeywords(AdGroupAddItemMap.DEFAULT_NEGATIVE_KEYWORD),
                        new AdGroupAddItemMap().defaultAdGroupAddItem(campaignId)
                ),
                ExpectedResult.success(),
                ExpectedResult.success()
        );
    }

    @Test
    public void checkStatusAfterAdd() {
        Long id = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
        GroupFakeInfo groupFakeInfo = api.userSteps.groupFakeSteps().getGroupParams(id);
        assertThat("StatusModerate = New", groupFakeInfo.getStatusModerate(), equalTo(Status.NEW));
    }
}
