package ru.yandex.autotests.direct.api.adgroups.add;

import java.util.Arrays;
import java.util.Collection;

import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.adgroups.AdGroupsFeatures;
import ru.yandex.autotests.direct.api.adgroups.AdGroupsLogins;
import ru.yandex.autotests.direct.api.adgroups.AdGroupsStories;
import ru.yandex.autotests.direct.utils.model.ShardNumbers;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.model.api5.adgroups.AdGroupAddItemMap;
import ru.yandex.autotests.directapi.model.api5.adgroups.AddRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Stories;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

/**
 * Author pavryabov
 * Date 12.09.14
 * https://st.yandex-team.ru/TESTIRT-2022
 */
@Aqua.Test
@Features(AdGroupsFeatures.ADD)
@Stories(AdGroupsStories.AUTHORIZATION)
@RunWith(Parameterized.class)
public class AgencyCheckRightsToAddTest {

    @ClassRule
    public static ApiSteps api = new ApiSteps();

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static final String client = AdGroupsLogins.CLIENT;
    private static final String agency = AdGroupsLogins.AGENCY;
    private static final String agencyRep = AdGroupsLogins.AGENCY_REP;
    private static final String agencyRepCl = AdGroupsLogins.AGENCY_REP_CL;
    private static final String subclient = AdGroupsLogins.SUBCLIENT;
    private static final String subclientElse = AdGroupsLogins.SUBCLIENT_ELSE;

    static Long clientCampaignId;
    static Long subclientCampaignId;
    static Long subclientElseCampaignId;

    @Parameterized.Parameter(0)
    public String loginForAuthorization;

    @Parameterized.Parameter(1)
    public String loginForHeader;

    @Parameterized.Parameter(2)
    public ExpectedResult resultForSublientCampaign;

    @Parameterized.Parameter(3)
    public ExpectedResult resultForSublientElseCampaign;

    @Parameterized.Parameters(name = "loginForAuthorization = {0}, loginForHeader = {1}")
    public static Collection testData() {
        Object[][] data = new Object[][]{
                {agency, subclient, ExpectedResult.success(),
                        ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.CAMPAIGN_NOT_FOUND))},
                {agencyRep, subclient, ExpectedResult.success(),
                        ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.CAMPAIGN_NOT_FOUND))},
                {agencyRep, subclientElse,
                        ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.CAMPAIGN_NOT_FOUND)),
                        ExpectedResult.success()},
                {agencyRepCl, subclient, ExpectedResult.success(),
                        ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.CAMPAIGN_NOT_FOUND))},
                {subclient, null, ExpectedResult.errors(new Notification(54, Api5ErrorDetails.NO_RIGHTS_TO_WRITE)),
                        ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.CAMPAIGN_NOT_FOUND))}
        };
        return Arrays.asList(data);
    }

    @BeforeClass
    public static void createNotMineCampaign() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(client);
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(agency);
        api.userSteps.clientFakeSteps().defaultAPI(subclient);
        api.userSteps.clientFakeSteps().reshardUserWithLock(subclient, ShardNumbers.DEFAULT_SHARD);
        api.userSteps.clientFakeSteps().reshardUserWithLock(subclientElse, ShardNumbers.DEFAULT_SHARD);
        clientCampaignId = api.as(client).userSteps.campaignSteps().addDefaultTextCampaign(client);
        subclientCampaignId = api.as(agency).userSteps.campaignSteps().addDefaultTextCampaign(subclient);
        subclientElseCampaignId = api.as(agency).userSteps.campaignSteps().addDefaultTextCampaign(subclientElse);
    }

    @Test
    public void callAddWithSubclientCampaign() {
        api.as(loginForAuthorization).userSteps.adGroupsSteps().shouldGetResultOnAdd(
                new AddRequestMap().withAdGroups(
                        new AdGroupAddItemMap().defaultAdGroupAddItem(subclientCampaignId)
                ),
                loginForHeader,
                resultForSublientCampaign
        );
    }

    @Test
    public void callAddWithSubclientElseCampaign() {
        //DIRECT-36157
        api.as(loginForAuthorization).userSteps.adGroupsSteps().shouldGetResultOnAdd(
                new AddRequestMap().withAdGroups(
                        new AdGroupAddItemMap().defaultAdGroupAddItem(subclientElseCampaignId)
                ),
                loginForHeader,
                resultForSublientElseCampaign
        );
    }

    @Test
    public void callAddWithCampaignsOfBothSubclients() {
        api.as(loginForAuthorization).userSteps.adGroupsSteps().shouldGetResultOnAdd(
                new AddRequestMap().withAdGroups(
                        new AdGroupAddItemMap().defaultAdGroupAddItem(subclientCampaignId),
                        new AdGroupAddItemMap().defaultAdGroupAddItem(subclientElseCampaignId)
                ),
                loginForHeader,
                resultForSublientCampaign,
                resultForSublientElseCampaign
        );
    }

    @Test
    public void callAddWithClientCampaign() {
        api.as(loginForAuthorization).userSteps.adGroupsSteps().shouldGetResultOnAdd(
                new AddRequestMap().withAdGroups(
                        new AdGroupAddItemMap().defaultAdGroupAddItem(clientCampaignId)
                ),
                loginForHeader,
                ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.CAMPAIGN_NOT_FOUND))
        );
    }
}
