package ru.yandex.autotests.direct.api.adgroups.add.mobileappadgroup;

import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.collect.Iterables;
import org.junit.After;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.adgroups.AdGroupsFeatures;
import ru.yandex.autotests.direct.api.adgroups.AdGroupsLogins;
import ru.yandex.autotests.direct.api.adgroups.AdGroupsStories;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.MobileAppsRecord;
import ru.yandex.autotests.directapi.model.User;
import ru.yandex.autotests.directapi.model.api5.adgroups.AdGroupAddItemMap;
import ru.yandex.autotests.directapi.model.api5.adgroups.AddRequestMap;
import ru.yandex.autotests.directapi.model.api5.adgroups.MobileAppAdGroupAddMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Stories;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.empty;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.isIn;
import static org.hamcrest.Matchers.not;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

@Aqua.Test
@Features(AdGroupsFeatures.ADD)
@Stories(AdGroupsStories.MOBILE)
@Issue("https://st.yandex-team.ru/DIRECT-82971")
@Description("Проверка добавления групп в кампанию, где нет привязанного приложения")
public class MobileAppAdGroupAddNewRmpCampaignDoesNotHaveMobileAppTest {
    private static final String CLIENT = AdGroupsLogins.CLIENT_NEW_RMP;

    private static final String DIFFERENT_STORE_URL =
            "https://play.google.com/store/apps/details?id=com.agandeev.mathgames.free";

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(CLIENT);

    @Rule
    public Trashman trasher = new Trashman(api);

    private int shard;
    private Long clientId;
    private long campaignId;
    private Set<Long> existentMobileAppIds;
    private String someStoreUrlFromExistentMobileApp;

    @Before
    public void before() {
        shard = api.userSteps.clientFakeSteps().getUserShard(CLIENT);
        clientId = Long.valueOf(User.get(CLIENT).getClientID());

        List<MobileAppsRecord> existentMobileApps = getClientMobileApps();
        assumeThat("у тестового пользователя есть созданные приложения", existentMobileApps, not(empty()));

        existentMobileAppIds =
                existentMobileApps.stream().map(MobileAppsRecord::getMobileAppId).collect(Collectors.toSet());

        Set<String> storeUrls =
                existentMobileApps.stream().map(MobileAppsRecord::getStoreHref).collect(Collectors.toSet());
        someStoreUrlFromExistentMobileApp = Iterables.getFirst(storeUrls, null);

        campaignId = api.userSteps.campaignSteps().addDefaultMobileAppCampaign();

        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(CLIENT);
    }

    @After
    public void after() {
        Long maybeCreatedMobileApp = api.userSteps.getDirectJooqDbSteps().useShard(shard).campaignsSteps()
                .getCampaignMobileAppId(campaignId);
        if (!existentMobileAppIds.contains(maybeCreatedMobileApp)) {
            // создалось новое приложение, почистим после теста
            api.userSteps.getDirectJooqDbSteps().useShard(shard).mobileAppsSteps()
                    .deleteMobileApp(maybeCreatedMobileApp);
        }
        api.userSteps.getDirectJooqDbSteps().useShard(shard).mobileAppsSteps().getMobileAppsByClient(clientId).stream()
                .filter(r -> r.getStoreHref().equals(DIFFERENT_STORE_URL))
                .findFirst()
                .map(MobileAppsRecord::getMobileAppId)
                .ifPresent(id -> api.userSteps.getDirectJooqDbSteps().useShard(shard).mobileAppsSteps()
                        .deleteMobileApp(id));
        api.userSteps.getDirectJooqDbSteps().useShard(shard).campaignsSteps().unsetCampaignMobileAppId(campaignId);
    }

    @Test
    public void addGroupsToCampaignWithoutMobileAppAndClientDoesntHaveAnyAppsWithSameStoreUrl_SuccessAndAppIsCreated() {
        api.userSteps.adGroupsSteps().shouldGetResultOnAdd(
                new AddRequestMap().withAdGroups(createAdGroupAddItemMap(campaignId, DIFFERENT_STORE_URL)),
                ExpectedResult.success()
        );
        Long mobileAppId = api.userSteps.getDirectJooqDbSteps().useShard(shard).campaignsSteps()
                .getCampaignMobileAppId(campaignId);
        assertThat("Привязали мобильное приложении к кампании", mobileAppId, not(equalTo(0L)));
        assertThat("Мобильное приложение новое для клиента", mobileAppId, not(isIn(existentMobileAppIds)));
    }

    @Test
    public void addGroupsToCampaignWithoutMobileAppAndClientHasAppsWithSameStoreUrl_SuccessAndAppIsLinkedToCampaign() {
        api.userSteps.adGroupsSteps().shouldGetResultOnAdd(
                new AddRequestMap().withAdGroups(
                        createAdGroupAddItemMap(campaignId, someStoreUrlFromExistentMobileApp)
                ),
                ExpectedResult.success()
        );
        Long mobileAppId = api.userSteps.getDirectJooqDbSteps().useShard(shard).campaignsSteps()
                .getCampaignMobileAppId(campaignId);
        assertThat("Привязали мобильное приложении к кампании", mobileAppId, not(equalTo(0L)));
        assertThat("Используем существующее мобильное приложение клиента", mobileAppId, isIn(existentMobileAppIds));
    }


    private AdGroupAddItemMap createAdGroupAddItemMap(long campaignId, String storeUrl) {
        return new AdGroupAddItemMap()
                .defaultAdGroupAddItem(campaignId)
                .withMobileAppAdGroup(
                        new MobileAppAdGroupAddMap()
                                .defaultGroup()
                                .withStoreUrl(storeUrl)
                );
    }

    private List<MobileAppsRecord> getClientMobileApps() {
        return api.userSteps.getDirectJooqDbSteps().useShard(shard).mobileAppsSteps()
                .getMobileAppsByClient(clientId);
    }
}
