package ru.yandex.autotests.direct.api.adgroups.get;


import com.yandex.direct.api.v5.adgroups.AdGroupFieldEnum;
import com.yandex.direct.api.v5.adgroups.GetResponse;
import com.yandex.direct.api.v5.general.StatusEnum;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.adgroups.AdGroupsFeatures;
import ru.yandex.autotests.direct.api.adgroups.AdGroupsLogins;
import ru.yandex.autotests.direct.utils.matchers.BeanCompareStrategy;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.model.api5.adgroups.AdGroupAddItemMap;
import ru.yandex.autotests.directapi.model.api5.adgroups.AdGroupGetItemMap;
import ru.yandex.autotests.directapi.model.api5.adgroups.AdGroupsSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.adgroups.GetRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.ArrayOfStringMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.contains;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.notNullValue;
import static org.hamcrest.Matchers.nullValue;
import static ru.yandex.autotests.direct.utils.matchers.BeanEquals.beanEquals;
import static ru.yandex.autotests.directapi.matchers.beans.version5.BeanDifferMatcherV5.beanDifferV5;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Author pavryabov
 * Date 31.07.14
 * https://jira.yandex-team.ru/browse/TESTIRT-2022
 */
@Aqua.Test
@Features(AdGroupsFeatures.GET)
@Tag(TagDictionary.TRUNK)
public class FieldNamesTest {

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(AdGroupsLogins.CLIENT);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    static Long campaignId;
    static Long id;
    BeanCompareStrategy strategy;

    @BeforeClass
    public static void initTest() {
        campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign(AdGroupsLogins.CLIENT);
        id = api.userSteps.adGroupsSteps().addGroup(
                new AdGroupAddItemMap()
                        .defaultAdGroupAddItem(campaignId)
                        .withNegativeKeywords(AdGroupAddItemMap.DEFAULT_NEGATIVE_KEYWORD)
        );
    }

    @Before
    @Step("Подготовка данных для теста")
    public void createCompareStrategy() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(AdGroupsLogins.CLIENT);
        strategy = new BeanCompareStrategy()
                .putFieldMatcher(AdGroupGetItemMap.ID, nullValue())
                .putFieldMatcher(AdGroupGetItemMap.CAMPAIGN_ID, nullValue())
                .putFieldMatcher(AdGroupGetItemMap.STATUS, nullValue())
                .putFieldMatcher(AdGroupGetItemMap.NAME, nullValue())
                .putFieldMatcher(AdGroupGetItemMap.REGION_IDS, hasSize(0))
                .putFieldMatcher(AdGroupGetItemMap.NEGATIVE_KEYWORDS, nullValue());
    }

    @Test
    public void fieldNamesWithOnlyId() {
        GetResponse response = api.userSteps.adGroupsSteps().adGroupsGet(
                new GetRequestMap()
                        .withSelectionCriteria(
                                new AdGroupsSelectionCriteriaMap()
                                        .withIds(id)
                        )
                        .withFieldNames(AdGroupFieldEnum.ID)
        );
        strategy.putFieldMatcher(AdGroupGetItemMap.ID, equalTo(id));
        assertThat("вернулась правильная группа", response.getAdGroups(),
                contains(
                        beanEquals(new AdGroupGetItemMap().getBean()).accordingStrategy(strategy)));
    }

    @Test
    public void fieldNamesWithOnlyCampaignId() {
        GetResponse response = api.userSteps.adGroupsSteps().adGroupsGet(
                new GetRequestMap()
                        .withSelectionCriteria(
                                new AdGroupsSelectionCriteriaMap()
                                        .withIds(id)
                        )
                        .withFieldNames(AdGroupFieldEnum.CAMPAIGN_ID)
        );
        strategy.putFieldMatcher(AdGroupGetItemMap.CAMPAIGN_ID, equalTo(campaignId));
        assertThat("вернулась правильная группа", response.getAdGroups(),
                contains(
                        beanEquals(new AdGroupGetItemMap().getBean()).accordingStrategy(strategy)));
    }

    @Test
    public void fieldNamesWithOnlyName() {
        GetResponse response = api.userSteps.adGroupsSteps().adGroupsGet(
                new GetRequestMap()
                        .withSelectionCriteria(
                                new AdGroupsSelectionCriteriaMap()
                                        .withIds(id)
                        )
                        .withFieldNames(AdGroupFieldEnum.NAME)
        );
        String expectedName = AdGroupAddItemMap.DEFAULT_NAME;
        strategy.putFieldMatcher(AdGroupGetItemMap.NAME, beanDifferV5(expectedName));
        assertThat("вернулась правильная группа", response.getAdGroups(),
                contains(
                        beanEquals(new AdGroupGetItemMap().getBean()).accordingStrategy(strategy)));
    }

    @Test
    public void fieldNamesWithOnlyRegionIds() {
        //DIRECT-34191
        GetResponse response = api.userSteps.adGroupsSteps().adGroupsGet(
                new GetRequestMap()
                        .withSelectionCriteria(
                                new AdGroupsSelectionCriteriaMap()
                                        .withIds(id)
                        )
                        .withFieldNames(AdGroupFieldEnum.REGION_IDS)
        );
        strategy.putFieldMatcher(AdGroupGetItemMap.REGION_IDS, hasSize(1));
        assertThat("вернулась правильная группа", response.getAdGroups(),
                contains(
                        beanEquals(new AdGroupGetItemMap().getBean()).accordingStrategy(strategy)));
    }

    @Test
    public void fieldNamesWithOnlyNegativeKeywords() {
        GetResponse response = api.userSteps.adGroupsSteps().adGroupsGet(
                new GetRequestMap()
                        .withSelectionCriteria(
                                new AdGroupsSelectionCriteriaMap()
                                        .withIds(id)
                        )
                        .withFieldNames(AdGroupFieldEnum.NEGATIVE_KEYWORDS)
        );
        strategy.putFieldMatcher(AdGroupGetItemMap.NEGATIVE_KEYWORDS, notNullValue());
        assertThat("вернулась правильная группа", response.getAdGroups(),
                contains(
                        beanEquals(new AdGroupGetItemMap().getBean()).accordingStrategy(strategy)));
    }

    @Test
    public void fieldNamesWithOnlyStatus() {
        //DIRECT-33859
        GetResponse response = api.userSteps.adGroupsSteps().adGroupsGet(
                new GetRequestMap()
                        .withSelectionCriteria(
                                new AdGroupsSelectionCriteriaMap()
                                        .withIds(id)
                        )
                        .withFieldNames(AdGroupFieldEnum.STATUS)
        );
        strategy.putFieldMatcher(AdGroupGetItemMap.STATUS, equalTo(StatusEnum.DRAFT));
        assertThat("вернулась правильная группа", response.getAdGroups(),
                contains(
                        beanEquals(new AdGroupGetItemMap().getBean()).accordingStrategy(strategy)));
    }

    @Test
    public void fieldNamesWithTwoFields() {
        GetResponse response = api.userSteps.adGroupsSteps().adGroupsGet(
                new GetRequestMap()
                        .withSelectionCriteria(
                                new AdGroupsSelectionCriteriaMap()
                                        .withIds(id)
                        )
                        .withFieldNames(AdGroupFieldEnum.ID, AdGroupFieldEnum.CAMPAIGN_ID)
        );
        strategy.putFieldMatcher(AdGroupGetItemMap.ID, equalTo(id));
        strategy.putFieldMatcher(AdGroupGetItemMap.CAMPAIGN_ID, equalTo(campaignId));
        assertThat("вернулась правильная группа", response.getAdGroups(),
                contains(
                        beanEquals(new AdGroupGetItemMap().getBean()).accordingStrategy(strategy)));
    }

    @Test
    public void fieldNamesWithAllFields() {
        //DIRECT-33859
        GetResponse response = api.userSteps.adGroupsSteps().adGroupsGet(
                new GetRequestMap()
                        .withSelectionCriteria(
                                new AdGroupsSelectionCriteriaMap()
                                        .withIds(id)
                        )
                        .withFieldNames(
                                AdGroupFieldEnum.ID,
                                AdGroupFieldEnum.CAMPAIGN_ID,
                                AdGroupFieldEnum.NAME,
                                AdGroupFieldEnum.STATUS,
                                AdGroupFieldEnum.REGION_IDS,
                                AdGroupFieldEnum.NEGATIVE_KEYWORDS
                        )
        );
        strategy.putFieldMatcher(AdGroupGetItemMap.ID, equalTo(id));
        strategy.putFieldMatcher(AdGroupGetItemMap.CAMPAIGN_ID, equalTo(campaignId));
        strategy.putFieldMatcher(AdGroupGetItemMap.NAME, notNullValue());
        strategy.putFieldMatcher(AdGroupGetItemMap.STATUS, equalTo(StatusEnum.DRAFT));
        strategy.putFieldMatcher(AdGroupGetItemMap.REGION_IDS, hasSize(1));
        strategy.putFieldMatcher(AdGroupGetItemMap.NEGATIVE_KEYWORDS, notNullValue());
        assertThat("вернулась правильная группа", response.getAdGroups(),
                contains(beanEquals(new AdGroupGetItemMap().getBean()).accordingStrategy(strategy)));
    }

    @Test
    public void checkNotRequestedNegativeKeywords() {
        GetResponse response = api.userSteps.adGroupsSteps().adGroupsGet(
                new GetRequestMap()
                        .withSelectionCriteria(
                                new AdGroupsSelectionCriteriaMap()
                                        .withIds(id)
                        )
                        .withFieldNames(AdGroupFieldEnum.ID)
        );
        assumeThat("AdGroups.get вернул одну группу", response.getAdGroups(), hasSize(1));
        assertThat("в ответе нет поля NegativeKeywords",
                response.getAdGroups().get(0).getNegativeKeywords(), nullValue());
    }

    @Test
    public void checkRequestedNegativeKeywordsInGroupWithThem() {
        GetResponse response = api.userSteps.adGroupsSteps().adGroupsGet(
                new GetRequestMap()
                        .withSelectionCriteria(
                                new AdGroupsSelectionCriteriaMap()
                                        .withIds(id)
                        )
                        .withFieldNames(AdGroupFieldEnum.NEGATIVE_KEYWORDS)
        );
        assumeThat("AdGroups.get вернул одну группу", response.getAdGroups(), hasSize(1));
        assertThat("в ответе вернулись правильные NegativeKeywords",
                response.getAdGroups().get(0).getNegativeKeywords().getValue(),
                equalTo(new ArrayOfStringMap().withItems(AdGroupAddItemMap.DEFAULT_NEGATIVE_KEYWORD).getBean()));
    }

    @Test
    public void checkRequestedNegativeKeywordsInGroupWithoutThem() {
        //DIRECT-33815
        Long idWithoutKeywords = api.userSteps.adGroupsSteps().addDefaultGroup(campaignId);
        GetResponse response = api.userSteps.adGroupsSteps().adGroupsGet(
                new GetRequestMap()
                        .withSelectionCriteria(
                                new AdGroupsSelectionCriteriaMap()
                                        .withIds(idWithoutKeywords)
                        )
                        .withFieldNames(AdGroupFieldEnum.NEGATIVE_KEYWORDS)
        );
        assumeThat("AdGroups.get вернул одну группу", response.getAdGroups(), hasSize(1));
        assertThat("в ответе не вернулось ни одного минус-слова",
                response.getAdGroups().get(0).getNegativeKeywords().getValue(), nullValue());
        assertThat("в ответе есть тег NegativeKeywords",
                response.getAdGroups().get(0).getNegativeKeywords().isNil(), equalTo(true));
    }
}
