package ru.yandex.autotests.direct.api.adgroups.get;

import java.util.Arrays;
import java.util.Collection;

import com.yandex.direct.api.v5.adgroups.AdGroupFieldEnum;
import com.yandex.direct.api.v5.adgroups.GetResponse;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.adgroups.AdGroupsFeatures;
import ru.yandex.autotests.direct.api.adgroups.AdGroupsLogins;
import ru.yandex.autotests.direct.utils.model.RegionIDValues;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.model.api5.adgroups.AdGroupAddItemMap;
import ru.yandex.autotests.directapi.model.api5.adgroups.AdGroupsSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.adgroups.GetRequestMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.hasSize;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Author pavryabov
 * Date 24.08.14
 * https://st.yandex-team.ru/TESTIRT-2022
 */
@Aqua.Test
@Features(AdGroupsFeatures.GET)
@Tag(TagDictionary.TRUNK)
@RunWith(Parameterized.class)
public class GetRegionIdsTest {

    @ClassRule
    public static final ApiSteps api = new ApiSteps().as(AdGroupsLogins.CLIENT);

    @ClassRule
    public static final SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static final Long MOSCOW = RegionIDValues.MOSCOW.getId().longValue();
    private static final Long CHERNOGOLOVKA = RegionIDValues.CHERNOGOLOVKA.getId().longValue();
    private static final Long MOSCOW_REGION = RegionIDValues.MOSCOW_AND_MOSCOW_REGION.getId().longValue();
    private static final Long RUSSIA = RegionIDValues.RUSSIA_COUNTRY.getId().longValue();
    private static final Long UKRAINE = RegionIDValues.UKRAINE.getId().longValue();
    private static final Long KIEV = RegionIDValues.KIEV.getId().longValue();
    private static final Long CRIMEA = RegionIDValues.CRIMEA.getId().longValue();
    private static final Long SEVASTOPOL = RegionIDValues.SEVASTOPOL.getId().longValue();

    @SuppressWarnings("DefaultAnnotationParam")
    @Parameterized.Parameter(0)
    public String caseDescription;

    @Parameterized.Parameter(1)
    public Long[] regionIds;

    @Parameterized.Parameter(2)
    public Long[] expectedRegionIds;

    @Parameterized.Parameters(name = "{0}")
    public static Collection testData() {
        Object[][] data = new Object[][]{
                {"world", new Long[]{0L}, new Long[]{0L}},
                {"MOSCOW", new Long[]{MOSCOW}, new Long[]{MOSCOW}},
                {"MOSCOW, CHERNOGOLOVKA", new Long[]{MOSCOW, CHERNOGOLOVKA}, new Long[]{MOSCOW, CHERNOGOLOVKA}},
                {"MOSCOW, MOSCOW_REGION", new Long[]{MOSCOW, MOSCOW_REGION}, new Long[]{MOSCOW, MOSCOW_REGION}},
                {"MOSCOW, world", new Long[]{MOSCOW, 0L}, new Long[]{MOSCOW, 0L}},
                {"MOSCOW_REGION, -MOSCOW", new Long[]{MOSCOW_REGION, -MOSCOW}, new Long[]{MOSCOW_REGION, -MOSCOW}},
                {"-MOSCOW, -MOSCOW_REGION, RUSSIA", new Long[]{-MOSCOW, -MOSCOW_REGION, RUSSIA}, new Long[]{RUSSIA, -MOSCOW, -MOSCOW_REGION}},
                {"world, MOSCOW_REGION, -MOSCOW", new Long[]{0L, MOSCOW_REGION, -MOSCOW}, new Long[]{0L, MOSCOW_REGION, -MOSCOW}},
                {"RUSSIA, UKRAINE, -KIEV, -MOSCOW", new Long[]{RUSSIA, UKRAINE, -KIEV, -MOSCOW}, new Long[]{RUSSIA, -MOSCOW, UKRAINE, -KIEV}},
                {"CRIMEA", new Long[]{CRIMEA}, new Long[]{CRIMEA}},
                {"CRIMEA, -SEVASTOPOL", new Long[]{CRIMEA, -SEVASTOPOL}, new Long[]{CRIMEA, -SEVASTOPOL}},
                {"CRIMEA, UKRAINE", new Long[]{CRIMEA, UKRAINE}, new Long[]{CRIMEA, UKRAINE}},
                {"CRIMEA, RUSSIA", new Long[]{CRIMEA, RUSSIA}, new Long[]{CRIMEA, RUSSIA}}
        };
        return Arrays.asList(data);
    }

    private static AdGroupAddItemMap adGroupAddItemMap;
    private Long id;

    @BeforeClass
    public static void createCampaign() {
        Long campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign(AdGroupsLogins.CLIENT);
        adGroupAddItemMap = new AdGroupAddItemMap()
                .withCampaignId(campaignId)
                .withName(AdGroupAddItemMap.DEFAULT_NAME);
    }

    @Before
    @Step("Подготовка данных для теста")
    public void createGroup() {
        api.userSteps.clientFakeSteps().fakeClearClientSpentUnits(AdGroupsLogins.CLIENT);
        id = api.userSteps.adGroupsSteps().addGroup(adGroupAddItemMap.withRegionIds(regionIds));
    }

    @Test
    public void getGroupWithRegionIds() {
        //DIRECT-34454
        GetResponse response = api.userSteps.adGroupsSteps().adGroupsGet(
                new GetRequestMap()
                        .withSelectionCriteria(
                                new AdGroupsSelectionCriteriaMap()
                                        .withIds(id)
                        )
                        .withFieldNames(AdGroupFieldEnum.REGION_IDS)
        );
        assumeThat("AdGroups.get вернул одну группу", response.getAdGroups(), hasSize(1));
        assertThat("регионы вернулись правильно",
                response.getAdGroups().get(0).getRegionIds(), equalTo(Arrays.asList(expectedRegionIds)));
    }
}
